
#include <microfeed-provider/microfeeddatabase.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedthread.h>
#include <microfeed-common/microfeedstore.h>

#include <errno.h>
#include <string.h>
#include <db.h>

struct _MicrofeedDatabaseEnvironment {
	unsigned int reference_count;
	MicrofeedMutex* mutex;
	char* name;
	DB_ENV* db_env;
	MicrofeedStore* databases;
};

struct _MicrofeedDatabase {
	unsigned int reference_count;
	MicrofeedMutex* mutex;
	MicrofeedDatabaseEnvironment* database_environment;
	char* name;
	MicrofeedDatabaseCompareFunction compare_function;
	DB* db;
	MicrofeedDatabaseIndex* indices;
	MicrofeedDatabaseIterator* iterators;
};

struct _MicrofeedDatabaseIndex {
	MicrofeedDatabaseIndex* next;
	
	MicrofeedDatabase* database;
	char* name;
	DB* db;
	MicrofeedDatabaseIndexFunction index_function;
};

struct _MicrofeedDatabaseIterator {
	MicrofeedDatabaseIterator* previous;
	MicrofeedDatabaseIterator* next;

	MicrofeedDatabase* database;
	DBC* db_cursor;
	DBT key;
	DBT data;
	void (*next_function)(MicrofeedDatabaseIterator* iterator);
	DB* index_db;
	DBT index_key;
	int backwards : 1;
};

static void lock_environment_for_writing(MicrofeedDatabaseEnvironment* database_environment);
static void unlock_environment(MicrofeedDatabaseEnvironment* database_environment);
static int key_compare_function(DB *db, const DBT *dbt1, const DBT *dbt2);
static int index_callback(DB* secondary, const DBT* key, const DBT* data, DBT* result);
static void remove_data(MicrofeedDatabase* database, DB* db, const void* key, const size_t key_size);
static void remove_data_range(MicrofeedDatabase* database, DB* db, const void* start_key, const size_t start_key_size, const void* end_key, const size_t end_key_size);
static void iterator_free(MicrofeedDatabaseIterator* iterator);
static void database_iterator_next(MicrofeedDatabaseIterator* iterator);
static void database_index_iterator_next(MicrofeedDatabaseIterator* iterator);
	
MicrofeedDatabaseEnvironment* microfeed_database_environment_new(const char* name, const char* directory) {
	MicrofeedDatabaseEnvironment* database_environment = NULL;

	if (microfeed_util_create_directory_recursively(directory)) {
		database_environment = microfeed_memory_allocate(MicrofeedDatabaseEnvironment);
		microfeed_util_create_directory_recursively(directory);
		if (db_env_create(&database_environment->db_env, 0) || database_environment->db_env->open(database_environment->db_env, directory, DB_CREATE | DB_THREAD | DB_PRIVATE | DB_INIT_CDB | DB_INIT_MPOOL, 0)) {
			microfeed_memory_free(database_environment);
			database_environment = NULL;
		} else {
			database_environment->reference_count = 1;
			database_environment->mutex = microfeed_mutex_new();
			database_environment->name = strdup(name);
			database_environment->databases = microfeed_store_new_sorted((MicrofeedStoreCompareKeysFunction)strcmp,
			                                                	  (MicrofeedStoreGetKeyFunction)microfeed_database_get_name);	
		}
	}

	return database_environment;
}

void microfeed_database_environment_free(MicrofeedDatabaseEnvironment* database_environment) {
	free(database_environment->name);
	database_environment->db_env->close(database_environment->db_env, 0);
	database_environment->db_env = NULL;
	microfeed_memory_free(database_environment);
}

MicrofeedDatabaseEnvironment* microfeed_database_environment_ref(MicrofeedDatabaseEnvironment* database_environment) {
	microfeed_mutex_lock(database_environment->mutex);

	database_environment->reference_count++;

	microfeed_mutex_unlock(database_environment->mutex);
	
	return database_environment;
}

void microfeed_database_environment_unref(MicrofeedDatabaseEnvironment* database_environment) {
	microfeed_mutex_lock(database_environment->mutex);

	database_environment->reference_count--;
	if (database_environment->reference_count == 0) {
		microfeed_database_environment_free(database_environment);
	} else {

			microfeed_mutex_unlock(database_environment->mutex);
	}		
}
	
MicrofeedDatabase* microfeed_database_environment_get_database(MicrofeedDatabaseEnvironment* database_environment, const char* name, MicrofeedDatabaseCompareFunction compare_function) {
	MicrofeedDatabase* database;
	
	microfeed_mutex_lock(database_environment->mutex);

	if ((database = microfeed_store_get(database_environment->databases, name, MicrofeedDatabase))) {
		microfeed_database_ref(database);
	} else {
		database = microfeed_memory_allocate(MicrofeedDatabase);
		if (db_create(&database->db, database_environment->db_env, 0) ||
		    (compare_function && database->db->set_bt_compare(database->db, key_compare_function)) ||
		    database->db->open(database->db, NULL, database_environment->name, name, DB_BTREE, DB_CREATE | DB_THREAD, 0)) {
			microfeed_memory_free(database);
			database = NULL;
		} else {
			database->reference_count = 1;
			database->mutex = microfeed_mutex_new();
			database->database_environment = database_environment;
			database_environment->reference_count++;
			database->name = strdup(name);
			database->compare_function = compare_function;
			database->indices = NULL;
			database->iterators = NULL;
			database->db->app_private = database;
				
			microfeed_store_insert(database_environment->databases, database);
		}
	}

	microfeed_mutex_unlock(database_environment->mutex);
	
	return database;
}

MicrofeedDatabaseIndex* microfeed_database_get_index(MicrofeedDatabase* database, const char* name, MicrofeedDatabaseIndexFunction index_function) {
	MicrofeedDatabaseIndex* database_index;
	char* full_name;
	
	microfeed_mutex_lock(database->mutex);

	for (database_index = database->indices; database_index; database_index = database_index->next) {
		if (!strcmp(database_index->name, name)) {
			break;
		}
	}
	if (!database_index) {
		database_index = microfeed_memory_allocate(MicrofeedDatabaseIndex);
		full_name = microfeed_util_string_concatenate(database->name, ":microfeed.index:", name, NULL);
		if (db_create(&database_index->db, database->database_environment->db_env, 0) ||
		    database_index->db->open(database_index->db, NULL, database->database_environment->name, full_name, DB_BTREE, DB_CREATE | DB_THREAD, 0)) {
			microfeed_memory_free(database_index);
			database_index = NULL;
		} else if (database->db->associate(database->db, NULL, database_index->db, index_callback, DB_CREATE)) { // | DB_IMMUTABLE_KEY)) {
			database_index->db->close(database_index->db, 0);
			microfeed_memory_free(database_index);
			database_index = NULL;
		} else {
			database_index->database = database;
			database_index->name = strdup(name);
			database_index->index_function = index_function;			
			database_index->db->app_private = database_index;
				
			database_index->next = database->indices;
			database->indices = database_index;
		}
		free(full_name);
	}

	microfeed_mutex_unlock(database->mutex);
	
	return database_index;	
}

void microfeed_database_free(MicrofeedDatabase* database) {
	MicrofeedDatabaseIndex* database_index;
	MicrofeedDatabaseIndex* next_database_index;
	
	for (database_index = database->indices; database_index; database_index = next_database_index) {
		next_database_index = database_index->next;
		database_index->db->close(database_index->db, 0);
		free(database_index->name);
		microfeed_memory_free(database_index);
	}	
	database->db->close(database->db, 0);
	database->db = NULL;

	microfeed_mutex_lock(database->database_environment->mutex);

	microfeed_store_remove(database->database_environment->databases, database);

	microfeed_mutex_unlock(database->database_environment->mutex);

	microfeed_mutex_free(database->mutex);
	database->mutex = NULL;
	microfeed_database_environment_unref(database->database_environment);
	database->database_environment = NULL;
	free(database->name);
	database->name = NULL;
		
	microfeed_memory_free(database);
}

MicrofeedDatabase* microfeed_database_ref(MicrofeedDatabase* database) {
	microfeed_mutex_lock(database->mutex);
	
	database->reference_count++;
	
	microfeed_mutex_unlock(database->mutex);
	
	return database;
}

void microfeed_database_unref(MicrofeedDatabase* database) {
	microfeed_mutex_lock(database->mutex);
	
	database->reference_count--;
	if (database->reference_count == 0) {
		microfeed_database_free(database);
	} else {
	
		microfeed_mutex_unlock(database->mutex);
	}
}

const char* microfeed_database_get_name(MicrofeedDatabase* database) {
	
	return database->name;
}

int microfeed_database_get_data(MicrofeedDatabase* database, const void* key, size_t key_size, void** data, size_t* data_size) {
	int retvalue = 0;
	DBT dbt_key;
	DBT dbt_data;
	
	microfeed_mutex_lock(database->mutex);

	memset(&dbt_key, 0, sizeof(DBT));
	memset(&dbt_data, 0, sizeof(DBT));
	dbt_key.data = (void*)key;
	dbt_key.size = key_size;
	dbt_data.flags = DB_DBT_MALLOC;
	
	if (!database->db->get(database->db, NULL, &dbt_key, &dbt_data, 0)) {
		*data = dbt_data.data;
		*data_size = dbt_data.size;
		retvalue = 1;
	}
	
	microfeed_mutex_unlock(database->mutex);
	
	return retvalue;
}

int microfeed_database_get_data_partial(MicrofeedDatabase* database, const void* key, size_t key_size, void* data, size_t* data_size, size_t offset) {
	int retvalue = 0;
	DBT dbt_key;
	DBT dbt_data;
	
	microfeed_mutex_lock(database->mutex);

	memset(&dbt_key, 0, sizeof(DBT));
	memset(&dbt_data, 0, sizeof(DBT));
	dbt_key.data = (void*)key;
	dbt_key.size = key_size;
	dbt_data.data = (void*)data;
	dbt_data.ulen = *data_size;
	dbt_data.flags = DB_DBT_USERMEM | DB_DBT_PARTIAL;
	dbt_data.dlen = *data_size;
	dbt_data.doff = offset;
	
	if (!database->db->get(database->db, NULL, &dbt_key, &dbt_data, 0)) {
		*data_size = dbt_data.size;
		retvalue = 1;
	}
	
	microfeed_mutex_unlock(database->mutex);
	
	return retvalue;
}


MicrofeedDatabaseIterator* microfeed_database_iterate(MicrofeedDatabase* database, const void* start_key, const size_t start_key_size, int backwards) {
	MicrofeedDatabaseIterator* iterator;

	microfeed_mutex_lock(database->mutex);

	iterator = microfeed_memory_allocate(MicrofeedDatabaseIterator);
	iterator->database = database;
	iterator->next_function = database_iterator_next;
	iterator->backwards = backwards;
	iterator->index_db = NULL;
	memset(&iterator->key, 0, sizeof(DBT));
	memset(&iterator->data, 0, sizeof(DBT));
	iterator->key.flags = DB_DBT_REALLOC;
	iterator->data.flags = DB_DBT_REALLOC;
	iterator->data.data = malloc(1);
	iterator->data.size = 1;

	if (start_key) {
		iterator->key.data = malloc(start_key_size);
		memcpy(iterator->key.data, start_key, start_key_size);
		iterator->key.size = start_key_size;
	}
	iterator->next_function(iterator);

	if (database->iterators) {
		database->iterators->previous = iterator;
		iterator->next = database->iterators;
	} else {
		iterator->next = NULL;
	}
	iterator->previous = NULL;
	database->iterators = iterator;

	microfeed_mutex_unlock(database->mutex);

	return iterator;
}

/*
void print(const char* text, const char* data, const size_t size) {
	size_t i;
	
	printf("%s: ", text);
	for (i = 0; i < size; i++) {
		if ((data[i] >= 'a' && data[i] <= 'z') || (data[i] >= 'A' && data[i] <= 'Z') || (data[i] >= '0' && data[i] <= '9')) {
			printf("%c", data[i]);
		} else {
			printf("_");
		}
	}
	printf("\n");
}
*/

void microfeed_database_replace_data(MicrofeedDatabase* database, const void* key, const size_t key_size, const void* data, const size_t data_size) {
	DBT dbt_key;
	DBT dbt_data;
	int error;
	
	lock_environment_for_writing(database->database_environment);

	memset(&dbt_key, 0, sizeof(DBT));
	memset(&dbt_data, 0, sizeof(DBT));
	dbt_key.data = (void*)key;
	dbt_key.size = key_size;
	dbt_data.data = (void*)data;
	dbt_data.size = data_size;
	if ((error = database->db->put(database->db, NULL, &dbt_key, &dbt_data, 0))) {
		fprintf(stderr, "DB->put failed: %s\n", db_strerror(error));
	}

	database->db->sync(database->db, 0);

	unlock_environment(database->database_environment);
}

void microfeed_database_replace_data_partial(MicrofeedDatabase* database, const void* key, size_t key_size, const void* data, size_t data_size, size_t offset) {
	DBT dbt_key;
	DBT dbt_data;
	int error;
	
	lock_environment_for_writing(database->database_environment);

	memset(&dbt_key, 0, sizeof(DBT));
	memset(&dbt_data, 0, sizeof(DBT));
	dbt_key.data = (void*)key;
	dbt_key.size = key_size;
	dbt_data.data = (void*)data;
	dbt_data.size = data_size;
	dbt_data.flags = DB_DBT_PARTIAL;
	dbt_data.dlen = data_size;
	dbt_data.doff = offset;
	if ((error = database->db->put(database->db, NULL, &dbt_key, &dbt_data, 0))) {
		fprintf(stderr, "DB->put failed in replace_data_partial: %s\n", db_strerror(error));
	}

	database->db->sync(database->db, 0);

	unlock_environment(database->database_environment);
}

void microfeed_database_remove_data(MicrofeedDatabase* database, const void* key, const size_t key_size) {
	remove_data(database, database->db, key, key_size);
}

void microfeed_database_remove_data_range(MicrofeedDatabase* database, const void* start_key, const size_t start_key_size, const void* end_key, const size_t end_key_size) {
	remove_data_range(database, database->db, start_key, start_key_size, end_key, end_key_size);
}

MicrofeedDatabase* microfeed_database_index_get_database(MicrofeedDatabaseIndex* database_index) {
	
	return database_index->database;
}

const char* microfeed_database_index_get_name(MicrofeedDatabaseIndex* database_index) {
	
	return database_index->name;
}

int microfeed_database_index_get_data(MicrofeedDatabaseIndex* database_index, const void* index_key, const size_t index_key_size, void** key, size_t* key_size, void** data, size_t* data_size) {
	int retvalue = 0;
	DBT dbt_index_key;
	DBT dbt_key;
	DBT dbt_data;

	microfeed_mutex_lock(database_index->database->mutex);

	memset(&dbt_index_key, 0, sizeof(DBT));
	memset(&dbt_key, 0, sizeof(DBT));
	memset(&dbt_data, 0, sizeof(DBT));
	dbt_index_key.data = (void*)index_key;
	dbt_index_key.size = index_key_size;
	dbt_key.flags = DB_DBT_MALLOC;
	dbt_data.flags = DB_DBT_MALLOC;
	
	if (!database_index->db->pget(database_index->db, NULL, &dbt_index_key, &dbt_key, &dbt_data, 0)) {
		*key = (char*)dbt_key.data;
		*key_size = dbt_key.size;
		*data = dbt_data.data;
		*data_size = dbt_data.size;
		retvalue = 1;
	}
	
	microfeed_mutex_unlock(database_index->database->mutex);
	
	return retvalue;
	
}

MicrofeedDatabaseIterator* microfeed_database_index_iterate(MicrofeedDatabaseIndex* database_index, const void* start_key, const size_t start_key_size, int backwards) {
	MicrofeedDatabaseIterator* iterator;

	microfeed_mutex_lock(database_index->database->mutex);

	iterator = microfeed_memory_allocate(MicrofeedDatabaseIterator);
	iterator->database = database_index->database;
	iterator->next_function = database_index_iterator_next;
	iterator->backwards = backwards;
	iterator->index_db = database_index->db;
	memset(&iterator->key, 0, sizeof(DBT));
	memset(&iterator->data, 0, sizeof(DBT));
	memset(&iterator->index_key, 0, sizeof(DBT));
	iterator->key.flags = DB_DBT_REALLOC;
	iterator->data.flags = DB_DBT_REALLOC;
	iterator->index_key.flags = DB_DBT_REALLOC;
	iterator->data.data = malloc(1);
	iterator->data.size = 1;

	if (start_key) {
		iterator->index_key.data = malloc(start_key_size);
		memcpy(iterator->index_key.data, start_key, start_key_size);
		iterator->index_key.size = start_key_size;
	}
	iterator->next_function(iterator);

	if (database_index->database->iterators) {
		database_index->database->iterators->previous = iterator;
		iterator->next = database_index->database->iterators;
	} else {
		iterator->next = NULL;
	}
	iterator->previous = NULL;
	database_index->database->iterators = iterator;

	microfeed_mutex_unlock(database_index->database->mutex);

	return iterator;
}
	
void microfeed_database_index_remove_data(MicrofeedDatabaseIndex* database_index, const void* key, const size_t key_size) {
	remove_data(database_index->database, database_index->db, key, key_size);
}

void microfeed_database_index_remove_data_range(MicrofeedDatabaseIndex* database_index, const void* start_key, const size_t start_key_size, const void* end_key, const size_t end_key_size) {
	remove_data_range(database_index->database, database_index->db, start_key, start_key_size, end_key, end_key_size);	
}

void microfeed_database_iterator_free(MicrofeedDatabaseIterator* iterator) {
	microfeed_mutex_lock(iterator->database->mutex);

	if (iterator->next) {
		iterator->next->previous = iterator->previous;
	}
	if (iterator->previous) {
		iterator->previous->next = iterator->next;
	} else if (iterator->database) {
		iterator->database->iterators = iterator->next;
	}

	microfeed_mutex_unlock(iterator->database->mutex);

	iterator_free(iterator);
	iterator->database = NULL;
	microfeed_memory_free(iterator);
}
	
int microfeed_database_iterator_get(MicrofeedDatabaseIterator* iterator, const void** key, size_t* key_size, const void** data, size_t* data_size) {
	int success = 0;

	microfeed_mutex_lock(iterator->database->mutex);
	
	if (iterator->data.data) {
		*key = iterator->key.data;
		*key_size = iterator->key.size;
		*data = iterator->data.data;
		*data_size = iterator->data.size;
		success = 1;
	}
	
	microfeed_mutex_unlock(iterator->database->mutex);

	return success;
}

void microfeed_database_iterator_next(MicrofeedDatabaseIterator* iterator) {
	microfeed_mutex_lock(iterator->database->mutex);

	iterator->next_function(iterator);
	
	microfeed_mutex_unlock(iterator->database->mutex);
}

static void remove_data(MicrofeedDatabase* database, DB* db, const void* key, const size_t key_size) {
	DBT dbt_key;
	
	lock_environment_for_writing(database->database_environment);

	memset(&dbt_key, 0, sizeof(DBT));
	dbt_key.data = (void*)key;
	dbt_key.size = key_size;

	db->del(db, NULL, &dbt_key, 0);	

	unlock_environment(database->database_environment);
}

static void remove_data_range(MicrofeedDatabase* database, DB* db, const void* start_key, const size_t start_key_size, const void* end_key, const size_t end_key_size) {
	DBC* db_cursor;
	DBT dbt_key;
	DBT dbt_data;

	lock_environment_for_writing(database->database_environment);	

	if (!db->cursor(db, NULL, &db_cursor, DB_WRITECURSOR)) {
		memset(&dbt_key, 0, sizeof(DBT));
		memset(&dbt_data, 0, sizeof(DBT));
		dbt_key.flags = DB_DBT_REALLOC;
		dbt_data.flags = DB_DBT_REALLOC;
		if (start_key) {
			dbt_key.data = malloc(start_key_size);
			memcpy(dbt_key.data, start_key, start_key_size);
			dbt_key.size = start_key_size;
		}
		if ((start_key && !db_cursor->c_get(db_cursor, &dbt_key, &dbt_data, DB_SET_RANGE)) || (!start_key && !db_cursor->c_get(db_cursor, &dbt_key, &dbt_data, DB_FIRST))) {
			do {
				db_cursor->c_del(db_cursor, 0);
				if (end_key && end_key_size == dbt_key.size && memcmp(end_key, dbt_key.data, end_key_size) <= 0) {
					break;
				}
			} while (!db_cursor->c_get(db_cursor, &dbt_key, &dbt_data, DB_NEXT));
		}
		db_cursor->c_close(db_cursor);
	}

	unlock_environment(database->database_environment);
	
}

static void iterator_free(MicrofeedDatabaseIterator* iterator) {
	if (iterator->db_cursor) {
		iterator->db_cursor->c_close(iterator->db_cursor);
		iterator->db_cursor = NULL;
	}
	free(iterator->key.data);
	free(iterator->data.data);
	free(iterator->index_key.data);
	iterator->key.data = NULL;
	iterator->data.data = NULL;
	iterator->index_key.data = NULL;
}

static void database_iterator_next(MicrofeedDatabaseIterator* iterator) {
	char* previous_key = NULL;
	
	if (iterator->data.data) {
		if (!iterator->db_cursor) {
			if (iterator->key.data) {
				previous_key = malloc(iterator->key.size);
				memcpy(previous_key, iterator->key.data, iterator->key.size);		
				if (iterator->database->db->cursor(iterator->database->db, NULL, &iterator->db_cursor, 0) ||
				    iterator->db_cursor->c_get(iterator->db_cursor, &iterator->key, &iterator->data, DB_SET_RANGE) ||
				    (!strcmp(iterator->key.data, previous_key) && iterator->db_cursor->c_get(iterator->db_cursor, &iterator->key, &iterator->data, (iterator->backwards ? DB_PREV : DB_NEXT)))) {
					iterator_free(iterator);
				}
				free(previous_key);
			} else {
				if (iterator->database->db->cursor(iterator->database->db, NULL, &iterator->db_cursor, 0) ||
				    iterator->db_cursor->c_get(iterator->db_cursor, &iterator->key, &iterator->data, (iterator->key.data ? DB_SET_RANGE : (iterator->backwards ? DB_LAST : DB_FIRST)))) {
					iterator_free(iterator);
				}
			}
		} else {
			if (iterator->db_cursor->c_get(iterator->db_cursor, &iterator->key, &iterator->data, (iterator->backwards ? DB_PREV : DB_NEXT))) {
				iterator_free(iterator);
			}
		}
	}
}

static void database_index_iterator_next(MicrofeedDatabaseIterator* iterator) {
	char* previous_key = NULL;
	
	if (iterator->data.data) {
		if (!iterator->db_cursor) {
			if (iterator->index_key.data) {
				previous_key = malloc(iterator->index_key.size);
				memcpy(previous_key, iterator->index_key.data, iterator->index_key.size);
				if (iterator->index_db->cursor(iterator->index_db, NULL, &iterator->db_cursor, 0) ||
				    iterator->db_cursor->c_pget(iterator->db_cursor, &iterator->index_key, &iterator->key, &iterator->data, DB_SET_RANGE) ||
				    (!strcmp(iterator->index_key.data, previous_key) && iterator->db_cursor->c_pget(iterator->db_cursor, &iterator->index_key, &iterator->key, &iterator->data, (iterator->backwards ? DB_PREV : DB_NEXT)))) {
					iterator_free(iterator);
				}
				free(previous_key);
			} else {
				if (iterator->index_db->cursor(iterator->index_db, NULL, &iterator->db_cursor, 0) ||
				    iterator->db_cursor->c_pget(iterator->db_cursor, &iterator->index_key, &iterator->key, &iterator->data, (iterator->index_key.data ? DB_SET_RANGE : (iterator->backwards ? DB_LAST : DB_FIRST)))) {
					iterator_free(iterator);
				}
			}
		} else {
			if (iterator->db_cursor->c_pget(iterator->db_cursor, &iterator->index_key, &iterator->key, &iterator->data, (iterator->backwards ? DB_PREV : DB_NEXT))) {
				iterator_free(iterator);
			}
		}
	}
}

static void lock_environment_for_writing(MicrofeedDatabaseEnvironment* database_environment) {
	MicrofeedStoreIterator* store_iterator;
	MicrofeedDatabase* database;
	MicrofeedDatabaseIterator* database_iterator;

	microfeed_mutex_lock(database_environment->mutex);

	for (store_iterator = microfeed_store_iterate(database_environment->databases, NULL);
	     (database = microfeed_store_iterator_get(store_iterator, MicrofeedDatabase));
	     microfeed_store_iterator_next(store_iterator)) {
		microfeed_mutex_lock(database->mutex);
		for (database_iterator = database->iterators; database_iterator; database_iterator = database_iterator->next) {
			if (database_iterator->data.data && database_iterator->db_cursor) {
				database_iterator->db_cursor->c_close(database_iterator->db_cursor);
				database_iterator->db_cursor = NULL;
			}
		}
	}
}

static void unlock_environment(MicrofeedDatabaseEnvironment* database_environment) {
	MicrofeedStoreIterator* store_iterator;
	MicrofeedDatabase* database;

	for (store_iterator = microfeed_store_iterate(database_environment->databases, NULL);
	     (database = microfeed_store_iterator_get(store_iterator, MicrofeedDatabase));
	     microfeed_store_iterator_next(store_iterator)) {
	    microfeed_mutex_unlock(database->mutex);
	}

	microfeed_mutex_unlock(database_environment->mutex);
}

static int index_callback(DB* db, const DBT* key, const DBT* data, DBT* result) {
	int retvalue = EINVAL;
	MicrofeedDatabaseIndex* database_index;
	void* index_key;
	size_t index_key_size;
	
	if ((database_index = (MicrofeedDatabaseIndex*)db->app_private)) {
		database_index->index_function(key->data, key->size, data->data, data->size, &index_key, &index_key_size);
		result->data = index_key;
		result->size = index_key_size;
		retvalue = 0;
	}
	
	return retvalue;
}

static int key_compare_function(DB *db, const DBT *dbt1, const DBT *dbt2) {
	int retvalue = 0;
	MicrofeedDatabase* database;
	
	if ((database = (MicrofeedDatabase*)db->app_private)) {
		retvalue = database->compare_function(dbt1->data, dbt1->size, dbt2->data, dbt2->size);	
	}

	return retvalue;
}
