
/** @mainpage Microfeed Documentation
 *
 * Microfeed is a specification and a reference implementation of client-server architecture
 * providing access to various information sources that have a feed-type interface.
 * Examples of those feed sources include micro-blogging services, such as Jaiku, Twitter,
 * Laconi.ca, and FriendFeed. By utilizing Microfeed architecture, the client application can
 * focus on user interface, while the actual feed fetching in done in the background
 * independently. The communication between the local server publishing information about
 * the feeds and the client application displaying that information to an user is done with
 * the DBus messaging following the publisher-subscriber principle.
 *
 * The local server is called provider. There can be - and usually are - many providers,
 * one for each micro-blogging or similar service, and thus a provider can focus solely
 * on polling one service. A provider may be instantiated and configured several ways at the
 * same time, for example, with different service URLs or authentication information.
 * One configuration forms a publisher, and thus a publisher is a configured instance of
 * a provider that provides the access to the service.
 *
 * The client application is acting as a subscriber. It subscribes the feeds it is interested.
 * A publisher sends a notification each time an item is added into, changed in, or removed from
 * the feed. It also notifies the changed status of the feed itself. One publisher can serve
 * multiple subscribers at the same time.
 *
 * If you are implementing a publisher, you need the following modules:
 * @link MicrofeedPublisher MicrofeedPublisher@endlink,
 * @link MicrofeedFeed MicrofeedFeed@endlink,
 * @link MicrofeedItem MicrofeedItem@endlink, and
 * @link MicrofeedConfiguration MicrofeedConfiguration@endlink. It is strongly adviced that you use also
 * @link MicrofeedProvider MicrofeedProvider@endlink. You may also find useful the following
 * modules:
 * @link MicrofeedMain MicrofeedMain@endlink,
 * @link MicrofeedHttp MicrofeedHttp@endlink, and
 * @link MicrofeedJson MicrofeedJson@endlink.
 * 
 *
 * If you are implementing a subscriber, you need the following modules:
 * @link MicrofeedSubscriber MicrofeedSubsrciber@endlink and
 * @link MicrofeedItem MicrofeedItem@endlink. It is highly propable that you need also
 * @link MicrofeedConfiguration MicrofeedConfiguration@endlink. You may also find useful
 * the following modules:
 * @link MicrofeedMain MicrofeedMain@endlink.
 *
 * The following modules are used internally in the implementation, but can be utilized
 * also outside:
 * @link MicrofeedDatabase MicrofeedDatabase@endlink,
 * @link MicrofeedStore MicrofeedStore@endlink, and
 * @link MicrofeedThread MicrofeedThread@endlink.
 */

#ifndef MICROFEEDPROTOCOL_H
#define MICROFEEDPROTOCOL_H

/**
 * @addtogroup MicrofeedProtocol
 * @{
 */


#define MICROFEED_DBUS_INTERFACE_PREFIX "org.microfeed."
#define MICROFEED_DBUS_INTERFACE_PUBLISHER MICROFEED_DBUS_INTERFACE_PREFIX "Publisher"
#define MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION MICROFEED_DBUS_INTERFACE_PUBLISHER ".ToDestination"
#define MICROFEED_DBUS_INTERFACE_SUBSCRIBER MICROFEED_DBUS_INTERFACE_PREFIX "Subscriber"
#define MICROFEED_DBUS_INTERFACE_ERROR MICROFEED_DBUS_INTERFACE_PREFIX "Error"
#define MICROFEED_DBUS_INTERFACE_ERROR_TO_DESTINATION MICROFEED_DBUS_INTERFACE_ERROR ".ToDestination"
#define MICROFEED_DBUS_OBJECT_PATH_PREFIX "/com/henrikhedberg/microfeed/"
#define MICROFEED_DBUS_OBJECT_PATH_PUBLISHER MICROFEED_DBUS_OBJECT_PATH_PREFIX "publisher/"
#define MICROFEED_DBUS_OBJECT_PATH_SUBSCRIBER MICROFEED_DBUS_OBJECT_PATH_PREFIX "subscriber/"


#define MICROFEED_ERROR_UNKNOWN "UnknownError"
#define MICROFEED_ERROR_DBUS_MESSAGE_FAILED "DBusMessageFailed"
#define MICROFEED_ERROR_PROVIDER_CLOSED_CONNECTION "ProviderClosedConnection"
#define MICROFEED_ERROR_INVALID_PUBLISHER_IDENTIFIER "InvalidPublisherIdentifier"
#define MICROFEED_ERROR_INVALID_FEED_URI "InvalidFeedUri"
#define MICROFEED_ERROR_INVALID_ARGUMENTS "InvalidArguments"
#define MICROFEED_ERROR_MISSING_PROPERTY "MissingProperty"
#define MICROFEED_ERROR_FEED_NOT_SUBSCRIBED "FeedNotSubscribed"
#define MICROFEED_ERROR_FEED_ALREADY_SUBSCRIBED "FeedAlreadySubscribed"
#define MICROFEED_ERROR_PUBLISHER_ALREADY_EXISTS "PublisherAlreadyExists"
#define MICROFEED_ERROR_NO_SUCH_PROVIDER "NoSuchProvider"
#define MICROFEED_ERROR_NO_SUCH_PUBLISHER "NoSuchPublisher"
#define MICROFEED_ERROR_NO_SUCH_FEED "NoSuchFeed"
#define MICROFEED_ERROR_NO_SUCH_ITEM "NoSuchItem"
#define MICROFEED_ERROR_NO_SUCH_PROPERTY "NoSuchProperty"
#define MICROFEED_ERROR_NO_ITEM_DATA "NoItemData"
#define MICROFEED_ERROR_CANNOT_MODIFY_ITEM "CannotModifyItem"
#define MICROFEED_ERROR_CONNECTION_FAILED "ConnectionFailed"
#define MICROFEED_ERROR_LOGIN_FAILED "LoginFailed"
#define MICROFEED_ERROR_COMMUNICATION_FAILED "CommunicationFailed"

#define MICROFEED_SIGNAL_NAME_FEED_UPDATE_STARTED "FeedUpdateStarted"
#define MICROFEED_SIGNAL_NAME_ITEM_DATA "ItemData"
#define MICROFEED_SIGNAL_NAME_ITEM_ADDED "ItemAdded"
/* TODO: others */

#define MICROFEED_FEED_URI_IMAGES "http://microfeed.org/Feed/Images"
#define MICROFEED_FEED_URI_OVERVIEW "http://microfeed.org/Feed/Overview"
#define MICROFEED_FEED_URI_SETTINGS "http://microfeed.org/Feed/Settings"
#define MICROFEED_FEED_URI_CONTACTS "http://microfeed.org/Feed/Contacts"
#define MICROFEED_FEED_URI_USER_PREFIX "http://microfeed.org/Feed/User/"

#define MICROFEED_ITEM_UID_FEED_METADATA "http://microfeed.org/Metadata/Feed"

/** @} */

#endif
