#ifndef MICROFEEDFEED_H
#define MICROFEEDFEED_H

/**
 * @addtogroup MicrofeedFeed
 *
 * This module implementing a feed is used only in the publisher side.
 *
 * MicrofeedFeed is thread-aware, because it implements internal lock.
 * Usually an MicrofeedFeed object is accessed simultaneously from multiple threads, for example from
 * a feed update thread and from the main thread responding to DBus messages. When one thread is
 * accessing the object, others are waiting.
 *
 * Note that MicrofeedFeed launches a new thread for every update and similar operation that may take
 * some time. Thus, you have to call #microfeed_thread_init at the very first and
 * #microfeed_thread_cleanup at the very last in your application.
 * 
 * @{
 */

/**
 * Opaque data type representing Microfeed feed.
 */
typedef struct _MicrofeedFeed MicrofeedFeed;

/**
 * Opaque data type representing Microfeed feed iterator that iterates over the items of the feed.
 */
typedef struct _MicrofeedFeedIterator MicrofeedFeedIterator;

/** @} */

#include <microfeed/microfeedpublisher.h>
#include <microfeed/microfeedstore.h>
#include <microfeed/microfeeditem.h>
#include <microfeed/microfeedmisc.h>
#include <microfeed/microfeederror.h>

/**
 * @addtogroup MicrofeedFeed
 * @{
 */

typedef void (*MicrofeedFeedDestroyCallback)(MicrofeedFeed* feed, void* user_data);

/**
 * This function is called when a subscriber wants to update the feed.
 * 
 * The actual network operations are happening here. The function is called in a separate thread, so it
 * may also block. For pull-type services, typically one update cycle is implemented. For
 * push-type services, this function initializes the connection and waits for the events indefinitely.
 * However, the function must check peridiocally if the subscriber count (#microfeed_feed_get_subscriber_count)
 * has dropped to zero and in that case close the connection and return.
 *
 * The update thread has one reference to the feed meaning that the feed is not freed before the thread
 * finishes. The reference count is incremented and decremented automatically.
 *
 * If this function returns non-NULL error object, a signal to subscribers is sent.
 * 
 * @param feed Instantiated MicrofeedFeed.
 * @param user_intiatied Boolean value telling if the update is requested by a subscriber (true), or periodical event (false).
 * @param user_data The pointer given when the feed has been instantiated.
 * @return Error or NULL, if the operation succeeded.
 */
typedef MicrofeedError* (*MicrofeedFeedUpdateCallback)(MicrofeedFeed* feed, int user_initiated, void* user_data);

/**
 * This function is called when a subscriber wants to add a new item, modify an existing item, or delete an existing item.
 *
 * If the existing item parameter is null, the new item parameter contains a new item to be added.
 * If the new item parameter is null, the existing item parameter contains an esiting item to be deleted.
 * In other case, the existing item parameter contains the current properties of a modified item and
 * the new item parameter contains the new properties for the existing item.
 * 
 * @param feed Instantiated MicrofeedFeed.
 * @param existing_item An existing item or NULL.
 * @param new_item A new item or NULL.
 * @param user_data The pointer given when the feed has been instantiated.
 * @return Error or NULL, if the operation succeeded.
 */
typedef MicrofeedError* (*MicrofeedFeedModifyItemCallback)(MicrofeedFeed* feed, MicrofeedItem* existing_item, MicrofeedItem* new_item, void* user_data);
typedef MicrofeedError* (*MicrofeedFeedDownloadItemDataCallback)(MicrofeedFeed* feed, const char* uid, void** data, size_t* length, void* user_data);
typedef MicrofeedError* (*MicrofeedFeedMarkCallback)(MicrofeedFeed* feed, const char* uid, int mark_status, void* user_data);

typedef struct {
	MicrofeedFeedDestroyCallback destroy;
	MicrofeedFeedUpdateCallback update;
	MicrofeedFeedModifyItemCallback modify_item;
	MicrofeedFeedDownloadItemDataCallback download_item_data;
	MicrofeedFeedMarkCallback mark_item;
} MicrofeedFeedCallbacks;

MicrofeedFeed* microfeed_feed_new(MicrofeedPublisher* publisher, const char* uri, const char* name, MicrofeedItem* metadata_item, MicrofeedFeedCallbacks* callbacks, void* user_data);
void microfeed_feed_free(MicrofeedFeed* feed);
MicrofeedFeed* microfeed_feed_ref(MicrofeedFeed* feed);
void microfeed_feed_unref(MicrofeedFeed* feed);
MicrofeedWeakReference* microfeed_feed_get_weak_reference(MicrofeedFeed* feed);

MicrofeedPublisher* microfeed_feed_get_publisher(MicrofeedFeed* feed);
int microfeed_feed_is_using_threads(MicrofeedFeed* feed);
void microfeed_feed_stop_update(MicrofeedFeed* feed);
void microfeed_feed_replace_item(MicrofeedFeed* feed, MicrofeedItem* item);
void microfeed_feed_remove_item(MicrofeedFeed* feed, const char* uid);
void microfeed_feed_remove_items(MicrofeedFeed* feed, const char* start_uid, const char* end_uid);
void microfeed_feed_add_subscriber(MicrofeedFeed* feed, const char* bus_name);
MicrofeedItem* microfeed_feed_get_item(MicrofeedFeed* feed, const char* uid);
int microfeed_feed_get_subscriber_count(MicrofeedFeed* feed);
const char* microfeed_feed_get_uri(MicrofeedFeed* feed);
const char* microfeed_feed_get_name(MicrofeedFeed* feed);
void microfeed_feed_set_name(MicrofeedFeed* feed, const char* name);
MicrofeedItem* microfeed_feed_get_metadata_item(MicrofeedFeed* feed);
MicrofeedFeedIterator* microfeed_feed_iterate(MicrofeedFeed* feed, const char* start_uid, int backwards);
MicrofeedFeedIterator* microfeed_feed_iterate_timeline(MicrofeedFeed* feed, time_t start_timestamp, int backwards);
void microfeed_feed_remove_subscriber(MicrofeedFeed* feed, const char* bus_name);
void microfeed_feed_update(MicrofeedFeed* feed, const char* bus_name);
void microfeed_feed_republish(MicrofeedFeed* feed, const char* start_uid, const char* end_uid, unsigned int max_count, const char* bus_name);
int microfeed_feed_set_item_status(MicrofeedFeed* feed, const char* uid, MicrofeedItemStatus status_to_set);
int microfeed_feed_unset_item_status(MicrofeedFeed* feed, const char* uid, MicrofeedItemStatus status_to_unset);
void microfeed_feed_unset_item_statuses(MicrofeedFeed* feed, const char* start_uid, const char* end_uid, MicrofeedItemStatus status_to_unset);
void microfeed_feed_send_item_data(MicrofeedFeed* feed, const char* uid, const char* bus_name);
void microfeed_feed_ref_item_data(MicrofeedFeed* feed, const char* uid);
void microfeed_feed_unref_item_data(MicrofeedFeed* feed, const char* uid);

size_t microfeed_feed_get_uri_offset();

void microfeed_feed_iterator_free(MicrofeedFeedIterator* iterator);
MicrofeedFeed* microfeed_feed_iterator_get_feed(MicrofeedFeedIterator* iterator);
MicrofeedItem* microfeed_feed_iterator_get_item(MicrofeedFeedIterator* iterator);
void microfeed_feed_iterator_next(MicrofeedFeedIterator* iterator);
int microfeed_feed_iterator_jump_and_remove_previous_items(MicrofeedFeedIterator* iterator, const char* uid);

/** @} */

/* For internal use only. */
void microfeed_feed_call_modify_item_callback(MicrofeedFeed* feed, MicrofeedItem* existing_item, MicrofeedItem* new_item);

#endif
