/*
 * This file is a part of hildon-extras
 *
 * Copyright (C) 2010 Faheem Pervez <trippin1@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; version 2 of the license.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser Public License for more details.
 */

/* he-tz-chooser.h */

/**
 * SECTION:he-tz-chooser
 * @short_description: Function definitions for the closed #HildonTimeZoneChooser
 *
 * #HildonTimeZoneChooser is a closed widget in Maemo 5, with no headers currently 
 * available.
 * #HildonTimeZoneChooser, used in the Clock application, features
 * a pannable map of the world, along with a button in its EditToolbar to allow you
 * to input the name of a city directly.
 *
 * Note: Function defs. for the HildonPannableMap widget are still not available.
 *
 * <example>
 * <title>Basic program</title>
 * <programlisting>
 * void
 * create_he_tz_instance (void)
 * {
 *     Citytime *home_loc = clock_get_home_location ();
 *     HildonTimeZoneChooser* time_chooser = hildon_time_zone_chooser_new ();;
 * <!-- -->
 *     hildon_time_zone_chooser_set_city (time_chooser, home_loc->city);
 * <!-- -->
 *	   switch (hildon_time_zone_chooser_run (time_chooser))
 *	   {
 *        case FEEDBACK_DIALOG_RESPONSE_CITY_CHOSEN:
 *            break;
 *
 *        case FEEDBACK_DIALOG_RESPONSE_CANCELLED:
 *            g_assert_not_reached ();
 *     }
 * <!-- -->
 *     Cityinfo *time_chooser_time = hildon_time_zone_chooser_get_city (time_chooser);
 *     g_printf ("%s\n", cityinfo_get_zone (time_chooser_time));
 * <!-- --> 
 *     cityinfo_free (time_chooser_time);
 *     clock_citytime_free (home_loc);
 *     hildon_time_zone_chooser_free (time_chooser);
 * }
 * </programlisting>
 * </example>
 */ 

#include <hildon/hildon.h>
#include <clock/clockcore.h>

#ifndef __HE_TZ_CHOOSER_H__
#define __HE_TZ_CHOOSER_H__

G_BEGIN_DECLS

typedef struct HildonTimeZoneChooser_ HildonTimeZoneChooser;

/**
 *FeedbackDialogResponse:
 *@FEEDBACK_DIALOG_RESPONSE_CITY_CHOSEN: A city was chosen
 *@FEEDBACK_DIALOG_RESPONSE_CANCELLED: The HildonTimeZoneChooser window was closed, without choosing a city
 **/
typedef enum
{
	FEEDBACK_DIALOG_RESPONSE_CITY_CHOSEN = 1,
	FEEDBACK_DIALOG_RESPONSE_CANCELLED
} FeedbackDialogResponse;

/**
 * hildon_time_zone_chooser_new:
 *
 * Return value: A new #HildonTimeZoneChooser instance, or %NULL. Free with hildon_time_zone_chooser_free().
 **/ 
HildonTimeZoneChooser* hildon_time_zone_chooser_new (void);

/**
 * hildon_time_zone_chooser_run:
 * @chooser: A #HildonTimeZoneChooser instance.
 *
 * Display the #HildonTimeZoneChooser.
 *
 * Return value: A value in the #FeedbackDialogResponse enum.
 **/ 
FeedbackDialogResponse hildon_time_zone_chooser_run (HildonTimeZoneChooser *chooser);

/**
 * hildon_time_zone_chooser_set_city:
 * @chooser: A #HildonTimeZoneChooser instance.
 * @cityinfo: A #Cityinfo instance.
 *
 * Sets the city to be displayed in the #HildonTimeZoneChooser.
 **/ 
void hildon_time_zone_chooser_set_city (HildonTimeZoneChooser *chooser, Cityinfo *cityinfo);

/**
 * hildon_time_zone_chooser_get_city:
 * @chooser: A #HildonTimeZoneChooser instance.
 *
 * Gets the currently displayed city in the #HildonTimeZoneChooser.
 *
 * Return value: A newly-allocated #Cityinfo instance. Free with cityinfo_free().
 **/ 
Cityinfo* hildon_time_zone_chooser_get_city (HildonTimeZoneChooser *chooser);

/**
 * hildon_time_zone_chooser_free:
 * @chooser: A #HildonTimeZoneChooser instance.
 *
 * Frees the allocated #HildonTimeZoneChooser instance returned from hildon_time_zone_chooser_new().
 **/ 
void hildon_time_zone_chooser_free (HildonTimeZoneChooser *chooser);

G_END_DECLS

#endif /* __HE_TZ_CHOOSER_H__ */
