/*
 * This file is a part of hildon-extras
 *
 * Copyright (C) 2010 Gabriel Schulhof <nix@go-nix.ca>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version. or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**
 * SECTION:he-menu-view-column
 * @short_description: A pannable tree view of a single #GtkMenu widget
 *
 * #HeMenuViewColumn is a #HildonPannableArea containing a #GtkTreeView whose model is a #HeMenuStore. It takes care of asking
 * for the size it needs to display the widget, and pans if it doesn't get it. It never pans horizontally.
 */

#include <hildon/hildon.h>
#include "he-menu-view-column.h"
#include "he-menu-store.h"
#include "he-menu-view_priv.h"
#include "he-cell-renderer-toggle.h"

struct _HeMenuViewColumnPrivate
{
	GtkWidget *tv;
	GtkWidget *menu_widget;
	gboolean do_activate;
	GObject *cr_text;
	GtkTreeViewColumn *col_text;
	GtkTreeModel *old_tm;
	GtkTreeModel *old_hms;
	int max_tv_width;
	GtkWidget *menu;
	guint menu_destroy_signal_id;
};

G_DEFINE_TYPE(HeMenuViewColumn, he_menu_view_column, HILDON_TYPE_PANNABLE_AREA);

enum {
	HE_MENU_VIEW_COLUMN_MENU_PROPERTY = 1,
	HE_MENU_VIEW_COLUMN_MENU_WIDGET_PROPERTY,
	HE_MENU_VIEW_COLUMN_DO_ACTIVATE_PROPERTY,
};

/*
 * Calculate the desired size for this widget
 */
static void
size_request(GtkWidget *widget, GtkRequisition *rq)
{
	HeMenuViewColumn *hmvc = HE_MENU_VIEW_COLUMN(widget);
	GtkRequisition rq_tv = {.width = 0, .height = 0};
	guint indicator_width;
	gboolean ellipsize_set;

	g_object_get(G_OBJECT(hmvc->priv->cr_text), "ellipsize-set", &ellipsize_set, NULL);

	gtk_widget_style_get(widget, "indicator-width", &indicator_width, NULL);
	gtk_widget_size_request(hmvc->priv->tv, &rq_tv);

	if (rq_tv.width > hmvc->priv->max_tv_width)
		hmvc->priv->max_tv_width = rq_tv.width;

	if (ellipsize_set && rq_tv.width < hmvc->priv->max_tv_width)
		rq_tv.width = hmvc->priv->max_tv_width;

	rq->width  = 2 * GTK_CONTAINER(widget)->border_width + rq_tv.width + indicator_width;
	rq->height = 2 * GTK_CONTAINER(widget)->border_width + rq_tv.height;
}

static void
size_allocate(GtkWidget *widget, GtkAllocation *alloc)
{
	HeMenuViewColumn *hmvc = HE_MENU_VIEW_COLUMN(widget);
	gboolean ellipsize_set;
	GtkWidgetClass *gtkwidget_class_parent = GTK_WIDGET_CLASS(he_menu_view_column_parent_class);

	if (gtkwidget_class_parent->size_allocate)
		gtkwidget_class_parent->size_allocate(widget, alloc);

	g_object_get(hmvc->priv->cr_text, "ellipsize-set", &ellipsize_set, NULL);

	if (widget->allocation.width < widget->requisition.width && !ellipsize_set) {
		g_object_set(hmvc->priv->cr_text, "ellipsize", PANGO_ELLIPSIZE_END, "ellipsize-set", TRUE, NULL);
		gtk_tree_view_columns_autosize(GTK_TREE_VIEW(hmvc->priv->tv));
	}
}

/*
 * Entry point for property changes
 */
static void
set_property(GObject *obj, guint property_id, const GValue *val, GParamSpec *pspec)
{
	switch (property_id) {
		case HE_MENU_VIEW_COLUMN_MENU_PROPERTY:
			he_menu_view_column_set_menu(HE_MENU_VIEW_COLUMN(obj), GTK_MENU(g_value_get_object(val)));
			break;

		case HE_MENU_VIEW_COLUMN_DO_ACTIVATE_PROPERTY:
			he_menu_view_column_set_do_activate(HE_MENU_VIEW_COLUMN(obj), g_value_get_boolean(val));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id, pspec);
			break;
	}
}

/*
 * Entry point for property value retrievals
 */
static void
get_property(GObject *obj, guint property_id, GValue *val, GParamSpec *pspec)
{
	HeMenuViewColumn *hmvc = HE_MENU_VIEW_COLUMN(obj);

	switch (property_id) {
		case HE_MENU_VIEW_COLUMN_MENU_PROPERTY:
			g_value_set_object(val, he_menu_view_column_get_menu(hmvc));
			break;

		case HE_MENU_VIEW_COLUMN_MENU_WIDGET_PROPERTY:
			g_value_set_object(val, hmvc->priv->menu_widget);
			break;

		case HE_MENU_VIEW_COLUMN_DO_ACTIVATE_PROPERTY:
			g_value_set_boolean(val, hmvc->priv->do_activate);
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id, pspec);
			break;
	}
}

static void
finalize(GObject *obj)
{
	he_menu_view_column_set_menu(HE_MENU_VIEW_COLUMN(obj), NULL);
}

/*
 * Initialize the #HeMenuViewColumn class
 */
static void
he_menu_view_column_class_init(HeMenuViewColumnClass *hmvc_class)
{
	GObjectClass   *gobject_class = G_OBJECT_CLASS(hmvc_class);
	GtkWidgetClass *gtkwidget_class = GTK_WIDGET_CLASS(hmvc_class);

	gobject_class->set_property = set_property;
	gobject_class->get_property = get_property;
	gobject_class->finalize     = finalize;

	gtkwidget_class->size_request  = size_request;
	gtkwidget_class->size_allocate = size_allocate;

	g_object_class_install_property(gobject_class, HE_MENU_VIEW_COLUMN_MENU_PROPERTY,
		g_param_spec_object("menu", "Menu", "The menu to display",
			GTK_TYPE_MENU, G_PARAM_READWRITE));

	g_object_class_install_property(gobject_class, HE_MENU_VIEW_COLUMN_MENU_WIDGET_PROPERTY,
		g_param_spec_object("menu-widget", "Menu Widget", "The last menu widget that was activated",
			GTK_TYPE_WIDGET, G_PARAM_READABLE));

	g_object_class_install_property(gobject_class, HE_MENU_VIEW_COLUMN_DO_ACTIVATE_PROPERTY,
		g_param_spec_boolean("do-activate", "Activate Menu Widget", "Whether to activate the GtkMenuItem widget when the corresponding item is chosen",
			TRUE, G_PARAM_READWRITE));

	g_type_class_add_private(hmvc_class, sizeof(HeMenuViewColumnPrivate));
}

/*
 * Activate the #GtkMenuItem corresponding to the row that was activated
 */
static void
tv_row_activated(GtkTreeView *tv, GtkTreePath *tp, GtkTreeViewColumn *col, HeMenuViewColumn *hmvc)
{
	GtkWidget *menu_widget = NULL;
	GtkTreeIter itr;
	GtkTreeModel *tm = gtk_tree_view_get_model(tv);

	gtk_tree_model_get_iter(tm, &itr, tp);

	gtk_tree_model_get(tm, &itr, HE_MENU_STORE_MENU_WIDGET, &menu_widget, -1);

	if (menu_widget)
		if (GTK_WIDGET_SENSITIVE(menu_widget)) {
			hmvc->priv->menu_widget = menu_widget;
			if (hmvc->priv->do_activate)
				gtk_menu_item_activate(GTK_MENU_ITEM(menu_widget));
			g_object_notify(G_OBJECT(hmvc), "menu-widget");
		}
}

/*
 * For some reason we need to emit row-activate for check menu items
 */
static gboolean
tv_button_release(GtkTreeView *tv, GdkEventButton *event, HeMenuViewColumn *col)
{
	GtkTreePath *tp;

	if (gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(tv), event->x, event->y, &tp, NULL, NULL, NULL)) {
		GtkTreeModel *tm;

		g_object_get(G_OBJECT(tv), "model", &tm, NULL);

		if (tm) {
			GtkTreeIter itr;

			if (gtk_tree_model_get_iter(tm, &itr, tp)) {
				GtkWidget *widget = NULL;

				gtk_tree_model_get(tm, &itr, HE_MENU_STORE_MENU_WIDGET, &widget, -1);
				if (widget) {
					if (GTK_IS_CHECK_MENU_ITEM(widget))
						gtk_tree_view_row_activated(tv, tp, gtk_tree_view_get_column(tv, 0));
				}
			}
		}

		gtk_tree_path_free(tp);
	}

	return FALSE;
}

/*
 * Initialize the #HeMenuViewColumn instance
 */
static void
he_menu_view_column_init(HeMenuViewColumn *hmvc)
{
	GtkCellRenderer *cr;
	GtkTreeViewColumn *col;

	hmvc->priv = G_TYPE_INSTANCE_GET_PRIVATE(hmvc, HE_TYPE_MENU_VIEW_COLUMN, HeMenuViewColumnPrivate);

	hmvc->priv->old_tm = NULL;
	hmvc->priv->old_hms = NULL;
	hmvc->priv->menu_widget = NULL;
	hmvc->priv->do_activate = TRUE;
	hmvc->priv->max_tv_width = -1;
	hmvc->priv->menu = NULL;
	hmvc->priv->menu_destroy_signal_id = 0;

	hmvc->priv->tv = g_object_new(GTK_TYPE_TREE_VIEW, "visible", TRUE, NULL);
	g_object_add_weak_pointer(G_OBJECT(hmvc->priv->tv), (gpointer *)&(hmvc->priv->tv));

	/* Pixbuf/toggle/radio column */
	col = g_object_new(GTK_TYPE_TREE_VIEW_COLUMN, NULL);
	cr = g_object_new(GTK_TYPE_CELL_RENDERER_PIXBUF, "xalign", 0.5, "yalign", 0.5, NULL);
	gtk_tree_view_column_pack_start(col, cr, TRUE);
	gtk_tree_view_column_set_attributes(col, cr,
		"pixbuf",     HE_MENU_STORE_IMAGE_PIXBUF,
		"visible",    HE_MENU_STORE_IS_IMAGE_ITEM,
		"sensitive",  HE_MENU_STORE_SENSITIVE,
		NULL);

	cr = g_object_new(HE_TYPE_CELL_RENDERER_TOGGLE, "xalign", 0.5, "yalign", 0.5, NULL);
	gtk_tree_view_column_pack_start(col, cr, TRUE);
	gtk_tree_view_column_set_attributes(col, cr,
		"sensitive",    HE_MENU_STORE_SENSITIVE,
		"active",       HE_MENU_STORE_ACTIVE,
		"inconsistent", HE_MENU_STORE_INCONSISTENT,
		"radio",        HE_MENU_STORE_IS_RADIO_ITEM,
		"visible",      HE_MENU_STORE_IS_CHECK_ITEM,
		NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(hmvc->priv->tv), col);

	/* Label column */
	col = g_object_new(GTK_TYPE_TREE_VIEW_COLUMN, "expand", TRUE, NULL);
	cr = g_object_new(GTK_TYPE_CELL_RENDERER_TEXT, NULL);
	gtk_tree_view_column_pack_start(col, cr, TRUE);
	gtk_tree_view_column_set_attributes(col, cr,
		"sensitive", HE_MENU_STORE_SENSITIVE,
		"markup",    HE_MENU_STORE_MARKUP,
		NULL);
	hmvc->priv->cr_text = G_OBJECT(cr);
	hmvc->priv->col_text = col;
	gtk_tree_view_append_column(GTK_TREE_VIEW(hmvc->priv->tv), col);

	/* Submenu column */
	col = g_object_new(GTK_TYPE_TREE_VIEW_COLUMN, NULL);
	cr = g_object_new(GTK_TYPE_CELL_RENDERER_PIXBUF, "stock-id", GTK_STOCK_GO_FORWARD, NULL);
	gtk_tree_view_column_pack_start(col, cr, FALSE);
	gtk_tree_view_column_set_attributes(col, cr,
		"visible",   HE_MENU_STORE_HAS_CHILDREN,
		"sensitive", HE_MENU_STORE_SENSITIVE,
		NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(hmvc->priv->tv), col);

	gtk_container_add(GTK_CONTAINER(hmvc), hmvc->priv->tv);
	g_signal_connect(G_OBJECT(hmvc->priv->tv), "row-activated", (GCallback)tv_row_activated, hmvc);
	g_signal_connect(G_OBJECT(hmvc->priv->tv), "button-release-event", (GCallback)tv_button_release, hmvc);
}

/**
 * he_menu_view_column_get_menu:
 * @hmvc: The #HeMenuViewColumn whose associated #GtkMenu to retrieve
 *
 * Get the #GtkMenu associated with this #HeMenuViewColumn.
 *
 * Returns: The #GtkMenu widget, or %NULL
 *
 * Since: 0.9.1
 */
GtkWidget *
he_menu_view_column_get_menu(HeMenuViewColumn *hmvc)
{
	GObject *model = NULL;
	GtkWidget *menu = NULL;

	g_return_val_if_fail(HE_IS_MENU_VIEW_COLUMN(hmvc), NULL);

	g_object_get(G_OBJECT(hmvc->priv->tv), "model", &model, NULL);
	if (model)
		g_object_get(G_OBJECT(model), "child-model", &model, NULL);
	if (model)
		g_object_get(G_OBJECT(model), "menu", &menu, NULL);

	return menu;
}

/*
 * Turn off the ellipsize if it's on, because it may no longer be necessary
 */
static void
reset_resize(HeMenuViewColumn *hmvc)
{
	gboolean ellipsize_set;

	g_object_get(G_OBJECT(hmvc->priv->cr_text), "ellipsize-set", &ellipsize_set, NULL);

	if (ellipsize_set) {
		g_object_set(G_OBJECT(hmvc->priv->cr_text), "ellipsize", PANGO_ELLIPSIZE_NONE, "ellipsize-set", FALSE, NULL);
		hmvc->priv->max_tv_width = -1;
		gtk_tree_view_columns_autosize(GTK_TREE_VIEW(hmvc->priv->tv));
	}
}

/**
 * he_menu_view_column_set_menu:
 * @hmvc: The #HeMenuViewColumn to display the #GtkMenu
 * @menu: The #GtkMenu to display, or %NULL
 * 
 * Associate a #GtkMenu with this #HeMenuViewColumn. This replaces the currently wrapped #GtkMenu.
 *
 * Since: 0.9.1
 */
void
he_menu_view_column_set_menu(HeMenuViewColumn *hmvc, GtkMenu *menu)
{
	GtkTreeModel *tm = NULL;
	GtkTreeModel *hms = NULL;

	g_return_if_fail(HE_IS_MENU_VIEW_COLUMN(hmvc));
	if (menu)
		g_return_if_fail(GTK_IS_MENU(menu));

	if (hmvc->priv->menu_widget) {
		hmvc->priv->menu_widget = NULL;
		g_object_notify(G_OBJECT(hmvc), "menu-widget");
	}

	if (hmvc->priv->menu) {
		g_signal_handler_disconnect(hmvc->priv->menu, hmvc->priv->menu_destroy_signal_id);
		g_object_unref(hmvc->priv->menu);
		hmvc->priv->menu = NULL;
		hmvc->priv->menu_destroy_signal_id = 0;
	}

	if (menu) {
		hms = g_object_new(HE_TYPE_MENU_STORE, "menu", menu, NULL);
		tm = g_object_new(GTK_TYPE_TREE_MODEL_FILTER, "child-model", hms, NULL);

		gtk_tree_model_filter_set_visible_column(GTK_TREE_MODEL_FILTER(tm), HE_MENU_STORE_VISIBLE);
		gtk_widget_set_name(GTK_WIDGET(hmvc), gtk_widget_get_name(GTK_WIDGET(menu)));
		g_signal_connect_swapped(G_OBJECT(tm), "row-changed", (GCallback)reset_resize, hmvc);
		g_signal_connect_swapped(G_OBJECT(tm), "row-deleted", (GCallback)reset_resize, hmvc);
		g_signal_connect_swapped(G_OBJECT(tm), "row-inserted", (GCallback)reset_resize, hmvc);

		hmvc->priv->menu_destroy_signal_id = g_signal_connect_swapped(G_OBJECT(menu), "destroy", (GCallback)gtk_widget_destroy, hmvc);

		hmvc->priv->menu = g_object_ref(menu);
	}

	if (hmvc->priv->tv)
		g_object_set(G_OBJECT(hmvc->priv->tv), "model", tm, NULL);

	/* This is terrible! Why doesn't the model die with the tree view?! */
	while (hmvc->priv->old_tm)
		g_object_unref(hmvc->priv->old_tm);

	/* This is terrible! Why doesn't the model die with the tree view?! */
	while (hmvc->priv->old_hms)
		g_object_unref(hmvc->priv->old_hms);

	hmvc->priv->old_hms = hms;
	hmvc->priv->old_tm = tm;

	if (hms) {
		g_object_add_weak_pointer(G_OBJECT(hms), (gpointer *)&(hmvc->priv->old_hms));
		g_object_unref(hms);
	}

	if (tm) {
		g_object_add_weak_pointer(G_OBJECT(tm), (gpointer *)&(hmvc->priv->old_tm));
		g_object_unref(tm);
	}
}

/**
 * he_menu_view_column_get_menu_widget:
 * @hmvc: The #HeMenuViewColumn
 *
 * Retrieve the #GtkMenuItem last selected by the user.
 *
 * Returns: The last selected #GtkMenuItem, or %NULL
 *
 * Since: 0.9.1
 */
GtkWidget *
he_menu_view_column_get_menu_widget(HeMenuViewColumn *hmvc)
{
	return hmvc->priv->menu_widget;
}

/**
 * he_menu_view_column_set_do_activate:
 * @hmvc: The #HeMenuViewColumn to affect
 * @do_activate: Whether to turn on #GtkMenuItem activation
 *
 * Set whether clicking a row in the #HeMenuViewColumn activates the corresponding #GtkMenuItem.
 *
 * Since: 0.9.1
 */
void
he_menu_view_column_set_do_activate(HeMenuViewColumn *hmvc, gboolean do_activate)
{
	g_return_if_fail(hmvc != NULL);
	g_return_if_fail(HE_IS_MENU_VIEW_COLUMN(hmvc));
	hmvc->priv->do_activate = do_activate;
	g_object_notify(G_OBJECT(hmvc), "do-activate");
}

/**
 * he_menu_view_column_get_do_activate:
 * @hmvc: The #HeMenuViewColumn
 *
 * Get whether clicking a row in the #HeMenuViewColumn activates the corresponding #GtkMenuItem.
 *
 * Returns: Whether #GtkMenuItem activation is on.
 */
gboolean
he_menu_view_column_get_do_activate(HeMenuViewColumn *hmvc)
{
	g_return_val_if_fail(hmvc != NULL, TRUE);
	g_return_val_if_fail(HE_IS_MENU_VIEW_COLUMN(hmvc), TRUE);
	return hmvc->priv->do_activate;
}
