/*
 * This file is a part of hildon-extras
 *
 * Copyright (C) 2010 Thomas Perl
 * Copyright (C) 2010 Timur Kristf
 * Copyright (C) 2005, 2008 Nokia Corporation.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version. or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**
 * SECTION:he-simple-color-dialog
 * @short_description: An simple_color dialog for Hildon-based applications
 *
 * #HeSimpleColorDialog works as a nice default simple_color dialog for Maemo apps
 */

#define _GNU_SOURCE     /* needed for GNU nl_langinfo_l */
#define __USE_GNU       /* needed for locale */

#include <locale.h>

#include <string.h>
#include <stdlib.h>

#include <libintl.h>
#include <langinfo.h>

#include <hildon/hildon.h>

#include <gdk/gdk.h>

#include "he-simple-color-dialog.h"
#include "he-color-dialog.h"

#define PALETTE_SIZE 21
#define PALETTE_CUSTOM_COLOR (PALETTE_SIZE-1)

#define ICON_SIZE 50

static const gchar* DEFAULT_PALETTE[PALETTE_SIZE] = {
    "#fcaf3e", "#f57900", "#ce5c00", /* orange */
    "#8ae234", "#73d215", "#4e9a06", /* green */
    "#729fcf", "#3465a4", "#204a87", /* blue */
    "#ad7fa8", "#75507b", "#5c3566", /* purple */
    "#ef2929", "#cc0000", "#a40000", /* red */
    "#888a85", "#555753", "#2e3436", /* grey */
    "#ffffff", "#000000", "#ffffff", /* white, black, custom */
};

#define HE_SIMPLE_COLOR_DIALOG_GET_PRIVATE(obj)                           \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HE_TYPE_SIMPLE_COLOR_DIALOG, HeSimpleColorDialogPrivate))

#define _(String) dgettext("hildon-extras", String)

struct _HeSimpleColorDialogPrivate
{
    /* Widgets */
    GtkWidget* table_layout;
    GtkWidget* buttons[PALETTE_SIZE];

    /* Data */
    GdkColor palette[PALETTE_SIZE];
    guint color_index;
};

/* static GObject * he_simple_color_dialog_constructor (GType                  type,
                                                   guint                  n_construct_properties,
                                                   GObjectConstructParam *construct_properties); */
//static void he_simple_color_dialog_finalize (GObject * object);
//static void he_simple_color_dialog_response (GtkDialog *dialog, gint response_id, HeSimpleColorDialog *fd);
static void he_simple_color_dialog_class_init (HeSimpleColorDialogClass * class);
static void he_simple_color_dialog_init (HeSimpleColorDialog * fd);
//static void he_simple_color_dialog_show (GtkWidget *widget);
//static void he_simple_color_dialog_destroy (GtkObject *object);

static gpointer                                 parent_class = NULL;

GType G_GNUC_CONST
he_simple_color_dialog_get_type            (void)
{
    static GType dialog_type = 0;
	dialog_type = g_type_from_name ("HeSimpleColorDialog");

    if (!dialog_type) {
        static const GTypeInfo dialog_info =
        {
            sizeof (HeSimpleColorDialogClass),
            NULL,
            NULL,
            (GClassInitFunc) he_simple_color_dialog_class_init,
            NULL,
            NULL,
            sizeof (HeSimpleColorDialog),
            0,
            (GInstanceInitFunc) he_simple_color_dialog_init,
            NULL
        };

        dialog_type = g_type_register_static (GTK_TYPE_DIALOG,
                "HeSimpleColorDialog", &dialog_info, 0);
    }

    return dialog_type;
}

static void
he_simple_color_dialog_class_init (HeSimpleColorDialogClass * class)
{
  GObjectClass *gobject_class;
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  gobject_class = (GObjectClass *) class;
  object_class = (GtkObjectClass *) class;
  widget_class = (GtkWidgetClass *) class;

  parent_class = g_type_class_peek_parent (class);

  g_type_class_add_private (object_class, sizeof (HeSimpleColorDialogPrivate));
}

/* static void
he_simple_color_dialog_show (GtkWidget *widget)
{
    //HeSimpleColorDialogPrivate *priv = HE_SIMPLE_COLOR_DIALOG_GET_PRIVATE (widget);
    GTK_WIDGET_CLASS (parent_class)->show (widget);
} */

/* static void
he_simple_color_dialog_destroy (GtkObject *object)
{
    //HeSimpleColorDialogPrivate *priv = HE_SIMPLE_COLOR_DIALOG_GET_PRIVATE (object);
    GTK_OBJECT_CLASS (parent_class)->destroy (object);
} */

static void
_he_simple_color_dialog_toggled(GtkToggleButton* toggle_button, gpointer user_data)
{
    HeSimpleColorDialog* scd = HE_SIMPLE_COLOR_DIALOG(user_data);
    g_return_if_fail (HE_IS_SIMPLE_COLOR_DIALOG(scd));

    guint button_index = (guint)g_object_get_data(G_OBJECT(toggle_button), "color-index");

    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle_button))) {
        if (scd->priv->color_index != button_index) {
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(scd->priv->buttons[scd->priv->color_index]), FALSE);
            scd->priv->color_index = button_index;
        }
    }
}

static GtkWidget*
_he_simple_color_dialog_create_color_widget(GdkColor* color)
{
    GdkPixmap* pixmap = gdk_pixmap_new(NULL, ICON_SIZE, ICON_SIZE, 16);

    cairo_t* cr = gdk_cairo_create(pixmap);
    gdk_cairo_set_source_color(cr, color);

    cairo_rectangle(cr, 0, 0, ICON_SIZE, ICON_SIZE);
    cairo_fill(cr);
    cairo_destroy(cr);

    GtkWidget* result = gtk_image_new_from_pixmap(pixmap, NULL);
    gdk_pixmap_unref(pixmap);

    return result;
}

static void
he_simple_color_dialog_init (HeSimpleColorDialog *ad)
{
    ad->priv = HE_SIMPLE_COLOR_DIALOG_GET_PRIVATE (ad);
    guint i;

    ad->priv->color_index = 0;

    /* Parse default colors */
    for (i=0; i<PALETTE_SIZE; i++) {
        if (!gdk_color_parse(DEFAULT_PALETTE[i], &(ad->priv->palette[i]))) {
            g_warning("Cannot parse color from default palette: %s (falling back to black)", DEFAULT_PALETTE[i]);
            ad->priv->palette[i].red = 0;
            ad->priv->palette[i].green = 0;
            ad->priv->palette[i].blue = 0;
        }
    }

    gtk_window_set_title(GTK_WINDOW(ad), _(HE_SIMPLE_COLOR_DIALOG_DEFAULT_TITLE));

    /* Create table and insert color chooser buttons */
    GtkTable* t = GTK_TABLE(gtk_table_new(PALETTE_SIZE/3, 3, FALSE));

    for (i=0; i<PALETTE_SIZE; i++) {
        ad->priv->buttons[i] = hildon_gtk_toggle_button_new(HILDON_SIZE_THUMB_HEIGHT);
        gtk_widget_set_size_request(GTK_WIDGET(ad->priv->buttons[i]), 80, 80);
        gtk_container_add(GTK_CONTAINER(ad->priv->buttons[i]), _he_simple_color_dialog_create_color_widget(&(ad->priv->palette[i])));
        g_object_set_data(G_OBJECT(ad->priv->buttons[i]), "color-index", (gpointer)i);
        g_signal_connect(G_OBJECT(ad->priv->buttons[i]), "toggled", G_CALLBACK(_he_simple_color_dialog_toggled), ad);
        /* Set color image on button */
        gtk_table_attach(GTK_TABLE(t), ad->priv->buttons[i],
                i/3, i/3+1, /* xattach */
                i%3, i%3+1, /* yattach */
                GTK_EXPAND | GTK_FILL, /* xoptions */
                GTK_EXPAND | GTK_FILL, /* yoptions */
                0, /* xpadding */
                0 /* ypadding */);
    }

    ad->priv->table_layout = GTK_WIDGET (t);
    GtkWidget* content_area = gtk_dialog_get_content_area (GTK_DIALOG (ad));
    gtk_container_add (GTK_CONTAINER (content_area), GTK_WIDGET (t));

    gtk_dialog_add_button (GTK_DIALOG (ad), _("Select"), GTK_RESPONSE_OK);

    /* By default, toggle the first button */
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ad->priv->buttons[0]), TRUE);

    gtk_widget_show_all (content_area);

    /* Hide the "custom color" button */
    gtk_widget_hide (GTK_WIDGET (ad->priv->buttons [PALETTE_CUSTOM_COLOR]));

    GTK_WIDGET_SET_FLAGS (GTK_WIDGET (ad), GTK_NO_WINDOW);
    gtk_widget_set_redraw_on_allocate (GTK_WIDGET (ad), FALSE);
}

/* static void
he_simple_color_dialog_finalize (GObject * object)
{
  //HeSimpleColorDialogPrivate *priv = HE_SIMPLE_COLOR_DIALOG_GET_PRIVATE (object);//

  if (G_OBJECT_CLASS (parent_class)->finalize)
	G_OBJECT_CLASS (parent_class)->finalize (object);
} */

/* ------------------------------ PRIVATE METHODS ---------------------------- */

/* ------------------------------ PUBLIC METHODS ---------------------------- */

/**
 * he_simple_color_dialog_new:
 *
 * Creates a new #HeSimpleColorDialog
 *
 * Returns: The #HeSimpleColorDialog
 */
GtkWidget*
he_simple_color_dialog_new()
{
    return g_object_new(HE_TYPE_SIMPLE_COLOR_DIALOG, NULL);
}

/**
 * he_simple_color_dialog_set_color:
 * @scd: The #HeSimpleColorDialog
 * @color: The #GdkColor to display
 *
 * Sets the currently selected color of this color dialog.
 * The color is copied to an internal structure, so the
 * caller can free the #GdkColor passed to this object after
 * the function call.
 */
void
he_simple_color_dialog_set_color(HeSimpleColorDialog* scd, GdkColor* color)
{
    g_return_if_fail (HE_IS_SIMPLE_COLOR_DIALOG(scd));
    guint i;

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(scd->priv->buttons[scd->priv->color_index]), FALSE);

    /* Try to find the color in the existing palette */
    for (i=0; i<PALETTE_SIZE; i++) {
        if (gdk_color_equal(color, &(scd->priv->palette[i]))) {
            scd->priv->color_index = i;
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(scd->priv->buttons[scd->priv->color_index]), TRUE);
            return;
        }
    }

    /* Replace the custom color field with the new color */
    scd->priv->color_index = PALETTE_CUSTOM_COLOR;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(scd->priv->buttons[scd->priv->color_index]), TRUE);
    scd->priv->palette[scd->priv->color_index] = *color;

    /* Update button image */
    GtkWidget* old_child = gtk_bin_get_child(GTK_BIN(scd->priv->buttons[scd->priv->color_index]));
    gtk_widget_destroy(old_child);
    gtk_container_add(GTK_CONTAINER(scd->priv->buttons[scd->priv->color_index]), _he_simple_color_dialog_create_color_widget(color));

    /* Show the custom button now that we have a color set */
    gtk_widget_show_all(GTK_WIDGET(scd->priv->buttons[scd->priv->color_index]));
}

/**
 * he_simple_color_dialog_get_color:
 * @scd: The #HeSimpleColorDialog whose #GdkColor to retrieve
 *
 * Retrieve the #GdkColor from the #HeSimpleColorDialog.
 *
 * Returns: the currently selected color. The caller has to
 * free the returned value using gdk_color_free().
 */
GdkColor*
he_simple_color_dialog_get_color(HeSimpleColorDialog* scd)
{
    g_assert(HE_IS_SIMPLE_COLOR_DIALOG(scd));
    return gdk_color_copy(&(scd->priv->palette[scd->priv->color_index]));
}

/* Callback function for the "Advanced" button. */
static void
he_simple_color_dialog_advanced_button_callback(GtkWidget *widget, HeSimpleColorDialog *scd)
{
    /* Supress the warning about the unused parameter */
    (void)widget;

    /* Get the color from the HeSimpleColorDialog */
    GdkColor *current_color = he_simple_color_dialog_get_color(scd);
    /* Present a HeColorDialog */
    GdkColor *selected_color = he_color_dialog_present(GTK_WINDOW(scd), current_color, NULL);
    /* Set the new color to the HeSimpleColorDialog */
    he_simple_color_dialog_set_color(scd, selected_color);

    /* Free up the colors' memory */
    gdk_color_free(current_color);
    gdk_color_free(selected_color);
}

/**
 * he_simple_color_dialog_add_advanced_button:
 * @scd: The #HeSimpleColorDialog
 *
 * Adds an "Advanced" button to a #HeSimpleColorDialog.
 * This method should only be called once on a given dialog.
 * Its behaviour is undefined if it is called more than once.
 */
void
he_simple_color_dialog_add_advanced_button(HeSimpleColorDialog *scd)
{
    g_return_if_fail (HE_IS_SIMPLE_COLOR_DIALOG(scd));

    /* Create and set up the new button */
    GtkWidget* advanced_button = gtk_button_new_from_stock(_("Advanced"));
    GTK_WIDGET_SET_FLAGS (advanced_button, GTK_CAN_DEFAULT);
    g_signal_connect (advanced_button, "clicked", G_CALLBACK (he_simple_color_dialog_advanced_button_callback), (gpointer) scd);
    gtk_widget_show (GTK_WIDGET (advanced_button));

    /* This call sets the desired appearance to the button - without it, the button looks out of place */
    hildon_gtk_widget_set_theme_size (advanced_button, HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);

    /* Place the button to the dialog's action area */
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(scd)->action_area), GTK_WIDGET (advanced_button), FALSE, TRUE, 0);
    /* BUG? Since gtk_box_pack_start places it at the end, we have to manually set the order */
    gtk_box_reorder_child (GTK_BOX (GTK_DIALOG(scd)->action_area), advanced_button, 0);
}

/**
 * he_simple_color_dialog_present:
 * @parent: The parent #GtkWindow, or %NULL
 * @add_advanced_button: Whether to display the "Advanced" button
 * @default_color: The #GdkColor to display initially, or %NULL
 * @title: The title of the dialog, or %NULL
 *
 * Creates and presents a #HeSimpleColorDialog with a default color.
 * If the title is %NULL, the default is displayed instead.
 *
 * Returns: If the dialog is cancelled, a copy of the @default_color parameter is returned. (Or %NULL if @default_color is %NULL.)
 * Otherwise, the color chosen by the user is returned. The caller is responsible for freeing the returned value using gdk_color_free().
 **/
GdkColor*
he_simple_color_dialog_present(GtkWindow* parent, gboolean add_advanced_button, GdkColor* default_color, gchar *title)
{
    /* Create the dialog */
    HeSimpleColorDialog* dialog = HE_SIMPLE_COLOR_DIALOG (he_simple_color_dialog_new());

    /* Set the parent (if not NULL) */
    if (parent != NULL) {
        if (GTK_IS_WINDOW (parent)) {
            gtk_window_set_transient_for (GTK_WINDOW (dialog), parent);
            gtk_window_set_destroy_with_parent(GTK_WINDOW(dialog), TRUE);
        }
        else if (GDK_IS_WINDOW (parent)) {
            gtk_widget_realize (GTK_WIDGET (dialog));
            gdk_window_set_transient_for(gtk_widget_get_window(GTK_WIDGET(dialog)), GDK_WINDOW(parent));
            gtk_window_set_destroy_with_parent(GTK_WINDOW(dialog), TRUE);
        }
        else
        {
            g_printerr ("The parent parameter is invalid.");
        }
    }

    /* Set the title
       (if it is NULL, we don't do anything as HeSimpleColorDialog already has a default title. */
    if (title != NULL) {
        gtk_window_set_title (GTK_WINDOW (dialog), title);
    }

    /* Set the selected color of the dialog to the default color (if not NULL) */
    if (default_color != NULL) {
        he_simple_color_dialog_set_color (dialog, default_color);
    }

    if (add_advanced_button == TRUE) {
        he_simple_color_dialog_add_advanced_button (dialog);
    }

    /* Run the dialog */
    if (gtk_dialog_run (GTK_DIALOG (dialog)) != GTK_RESPONSE_OK) {
        /* If it is cancelled, release its memory... */
        gtk_widget_destroy (GTK_WIDGET (dialog));
        /* ... and return a copy of the default color */
        return (default_color == NULL ? NULL : gdk_color_copy (default_color));
    }

    /* Get the selected color from the dialog */
    GdkColor* result = he_simple_color_dialog_get_color (dialog);
    /* Free up the memory associated with the dialog */
    gtk_widget_destroy (GTK_WIDGET (dialog));
    /* Return the resulting color */
    return result;
}

/**
 * he_simple_color_dialog_present_interop:
 * @parent_winid: The parent #GdkNativeWindow, or %0
 * @add_advanced_button: Whether to display the "Advanced" button
 * @default_color: The #GdkColor to display initially, or %NULL
 * @title: The title of the dialog, or %NULL
 *
 * Creates and presents a #HeSimpleColorDialog with a default color.
 * If the title is %NULL, the default is displayed instead.
 *
 * Returns: If the dialog is cancelled, a copy of the @default_color parameter is returned. (Or %NULL if @default_color is %NULL.)
 * Otherwise, the color chosen by the user is returned. The caller is responsible for freeing the returned value using gdk_color_free().
 **/
GdkColor*
he_simple_color_dialog_present_interop(GdkNativeWindow parent_winid, gboolean add_advanced_button, GdkColor* default_color, gchar *title)
{
    GdkWindow *foreign_window = NULL;

    if (parent_winid != 0)
        foreign_window = gdk_window_foreign_new (parent_winid);

    GdkColor *result = he_simple_color_dialog_present ((GtkWindow*) foreign_window, add_advanced_button, default_color, title);

    if (foreign_window != NULL)
        g_object_unref(foreign_window);

    return result;
}
