#include <ColumbusPluginManager.h>

#include <ColumbusController.h>

#include <QDebug>
#include <QSettings>

#include <QDir>
#include <QFileInfo>
#include <QPluginLoader>

#define CONFIG_KEY_PLUGIN_PATH "/plugins/path"
#define CONFIG_KEY_PLUGIN_PATH_DEFAULT "/opt/usr/lib/columbus/plugins"

class PluginManagerPrivate
{
public:
    ColumbusController *controller;

    QList<QPluginLoader*> loaders;
    QHash<QString,ColumbusPluginInterface*> directory;
};

PluginManager::PluginManager(QObject *parent) : QObject(parent)
{
    qDebug() << "PluginManager: ctor()";
    this->d = new PluginManagerPrivate;
}

PluginManager::~PluginManager()
{
    qDebug() << "PluginManager: dtor()";
    qDebug() << "PluginManager: Unregistering plugins.";
    foreach(QString pluginId, this->plugins().keys())
    {
        this->unregisterPlugin(pluginId);
    }
}

ColumbusController* PluginManager::controller() const
{
    return d->controller;
}

QHash<QString,ColumbusPluginInterface*> PluginManager::plugins() const
{
    return d->directory;
}

void PluginManager::initialize(ColumbusController *controller)
{
    d->controller = controller;
    this->loadPlugins();
}

void PluginManager::loadPlugins()
{
    QDir dir(QSettings().value(CONFIG_KEY_PLUGIN_PATH, CONFIG_KEY_PLUGIN_PATH_DEFAULT).toString());

    qDebug() << "PluginManager: Loading plugins from:" << dir.absolutePath();
    foreach(QFileInfo pInfo, dir.entryInfoList(QStringList() << "lib*.so", QDir::Files))
    {
        QPluginLoader *loader = new QPluginLoader(this);
        loader->setLoadHints(QLibrary::ResolveAllSymbolsHint);

        qDebug() << "PluginManager: Found plugin candidate:" << pInfo.fileName();
        loader->setFileName(pInfo.absoluteFilePath());

        qDebug() << "PluginManager: Loading plugin candidate:" << pInfo.fileName();
        if(QLibrary::isLibrary(pInfo.fileName()) && !loader->load())
        {
            qWarning() << "PluginManager: Failed to load plugin candidate:"
                       << pInfo.fileName()
                       << loader->errorString();
            delete loader;
            continue;
        }

        QObject *instance = loader->instance();
        if(instance == NULL)
        {
            qWarning() << "PluginManager: Failed to instantiate plugin candidate:" << pInfo.fileName()
                       << loader->errorString();
            delete loader;
            continue;
        }

        if(!this->registerPlugin(qobject_cast<ColumbusPluginInterface*>(instance)))
        {
            delete loader;
            continue;
        }

        d->loaders.append(loader);
    }
}

bool PluginManager::registerPlugin(ColumbusPluginInterface *plugin)
{
    if(plugin->initialize(this))
    {
        qDebug() << "PluginManager: Registered plugin:" << plugin->id();
        d->directory.insert(plugin->id(), plugin);
        return true;
    }
    else
    {
        qDebug() << "PluginManager: registerPlugin(): Failed to register plugin:" << plugin->id();
        return false;
    }
}

void PluginManager::unregisterPlugin(const QString &pluginId)
{
    ColumbusPluginInterface *plugin = d->directory.value(pluginId);

    if(plugin == NULL)
    {
        qWarning() << "PluginManager: Plugin" << pluginId << "not found!";
        return;
    }

    qDebug() << "PluginManager: Removing plugin" << pluginId;
    d->directory.remove(pluginId);
    delete plugin;
}
