#include <QDebug>

#include <QDir>
#include <QDesktopServices>

#include "config.h"
#include "ColumbusController.h"

#include <QGeoPositionInfoSource>
#include <QGeoSatelliteInfoSource>

class ColumbusControllerPrivate
{
public:
    ConfigManager *config;
    PluginManager *plugins;
    LoggingManager *logger;

    ColumbusModel  model;

    QGeoPositionInfoSource *positionSource;
    QGeoSatelliteInfoSource *satelliteSource;
};

ColumbusController::ColumbusController(QObject *parent)
    : QObject(parent)
{
    this->d = new ColumbusControllerPrivate;

    qDebug() << "ApplicationController: Creating telemetry source objects.";
    d->satelliteSource = QGeoSatelliteInfoSource::createDefaultSource(this);
    d->positionSource  = QGeoPositionInfoSource::createDefaultSource(this);

    qDebug() << "ApplicationController: Initializing configuration manager.";
    d->config = new ConfigManager(this);
    this->setupConfig();

    qDebug() << "ApplicationController: Initializing logging manager.";
    d->logger = new LoggingManager(this);
    d->logger->initialize(this);

    qDebug() << "ApplicationController: Initializing plugin management system.";
    d->plugins = new PluginManager(this);
    d->plugins->initialize(this);

    // Force loading of configuration data.
    qDebug() << "ApplicationController: Loading application configuration.";
    this->onConfigUpdated(CONFIG_KEY_DISPLAY_POSITION_FORMAT, QVariant());
    this->onConfigUpdated(CONFIG_KEY_DISPLAY_UNITS, QVariant());

    d->positionSource->setPreferredPositioningMethods(QGeoPositionInfoSource::SatellitePositioningMethods);
    d->positionSource->setUpdateInterval(1000);

    // NOTE: I put this here to try and work out why sometimes under certain
    // versions geolocation services do not start!
    if(d->positionSource == NULL || d->satelliteSource == NULL)
    {
        qWarning() << "ApplicationController: Failed to create telemetry sources.";
        return;
    }

    qDebug() << "ApplicationController: Attempting to start up location services.";
    d->positionSource->startUpdates();
    d->satelliteSource->startUpdates();
    qDebug() << "ApplicationController: Location services started.";

    QObject::connect(d->config,
                     SIGNAL(updated(QString,QVariant)),
                     this,
                     SLOT(onConfigUpdated(QString,QVariant)));

    qDebug() << "ApplicationController: Connecting telemetry source signals.";
    QObject::connect(d->positionSource,
                     SIGNAL(positionUpdated(QGeoPositionInfo)),
                     this,
                     SLOT(onPositionUpdated(QGeoPositionInfo)));

    QObject::connect(d->positionSource,
                     SIGNAL(updateTimeout()), this, SLOT(onPositionTimeout()));

    QObject::connect(d->satelliteSource,
                     SIGNAL(satellitesInUseUpdated(QList<QGeoSatelliteInfo>)),
                     this,
                     SIGNAL(satellitesInUseUpdated(QList<QGeoSatelliteInfo>)));

    QObject::connect(d->satelliteSource,
                     SIGNAL(satellitesInViewUpdated(QList<QGeoSatelliteInfo>)),
                     this,
                     SIGNAL(satellitesInViewUpdated(QList<QGeoSatelliteInfo>)));
}

ColumbusController::~ColumbusController()
{
    //   Need to destroy the plugin manager and its' plugins first, to
    // ensure resources it may rely on are still available when they're
    // removed.
    delete d->plugins;
    delete this->d;
}

ColumbusModel* ColumbusController::model() const {return &d->model;}
ConfigManager* ColumbusController::config() const {return d->config;}
PluginManager* ColumbusController::plugins() const {return d->plugins;}
LoggingManager* ColumbusController::logger() const {return d->logger;}

bool ColumbusController::initialize()
{
    return true;
}

void ColumbusController::setupConfig()
{
    //TODO: Move out graphical options into application actual.
    this->config()->addToggle(CONFIG_KEY_DISPLAY_KEEP_ALIVE, tr("Keep-Alive Backlight"), true);

    this->config()->addOption(CONFIG_KEY_DISPLAY_ORIENTATION,
                              tr("Orientation"),
                              QStringList() << tr("Automatic") << tr("Landscape") << tr("Portrait"));

    // Aligned with ColumbusModel::DisplayUnits enumeration.
    this->config()->addOption(CONFIG_KEY_DISPLAY_UNITS,
                              tr("Numerical Units"),
                              QStringList() << tr("Metric") << tr("Imperial") << tr("Nautical"));

    // Organised to be inline with QGeoCoordinate enumeration.
    this->config()->addOption(CONFIG_KEY_DISPLAY_POSITION_FORMAT,
                              tr("Position Format"),
                              QStringList()
                              << QString::fromUtf8("[-] dd.ddddd°")             // QGeoCoordinate::Degrees
                              << QString::fromUtf8("dd.ddddd° [N/S]")           // QGeoCoordinate::DegreesWithHemisphere
                              << QString::fromUtf8("[-] dd° mm.mmm'")           // QGeoCoordinate::DegreesMinutes
                              << QString::fromUtf8("dd° mm.mmm' [N/S]")         // QGeoCoordinate::DegreesMinutesWithHemisphere
                              << QString::fromUtf8("[-] dd° mm' ss.s\"")        // QGeoCoordinate::DegreesMinutesSeconds
                              << QString::fromUtf8("dd° mm' ss.s\" [N/S]"));    // QGeoCoordinate::DegreesMinutesSecondsWithHemisphere
}

void ColumbusController::onConfigUpdated(const QString &key, QVariant)
{
    if(key == CONFIG_KEY_DISPLAY_POSITION_FORMAT)
    {
        d->model.setPositionFormat(((QGeoCoordinate::CoordinateFormat)d->config->valueIndex(CONFIG_KEY_DISPLAY_POSITION_FORMAT)));
    }
    else if(key == CONFIG_KEY_DISPLAY_UNITS)
    {
        d->model.setDisplayUnits((ColumbusModel::DisplayUnits)d->config->valueIndex(CONFIG_KEY_DISPLAY_UNITS));
    }
}

void ColumbusController::onPositionUpdated(const QGeoPositionInfo &position)
{
    QGeoPositionInfo nPosition(position);

    if(!position.isValid() || position.attribute(QGeoPositionInfo::HorizontalAccuracy) > 400.0f)
    {
        // HACK: This means we've not got a GPS fix or is so bad we might as well ignore it...
        return;
    }

// HACK: for buggy Qt Mobility liblocation wrapper
#if defined(Q_WS_MAEMO_5) && QTM_VERSION <= 0x010100
    if(nPosition.hasAttribute(QGeoPositionInfo::GroundSpeed))
    {
        nPosition.setAttribute(QGeoPositionInfo::GroundSpeed, nPosition.attribute(QGeoPositionInfo::GroundSpeed) / 3.6);
    }

    if(nPosition.hasAttribute(QGeoPositionInfo::VerticalAccuracy))
    {
        nPosition.setAttribute(QGeoPositionInfo::VerticalAccuracy, nPosition.attribute(QGeoPositionInfo::VerticalAccuracy) / 3.6);
    }
#endif

    d->model.onPositionUpdated(nPosition);
    emit this->positionUpdated(nPosition);
}

void ColumbusController::onPositionTimeout()
{
    qDebug() << "ApplicationController: Position update timeout occured!";
}
