/* vim: set sw=4 ts=4 et: */
/*
 * This file is part of Live Wallpaper (livewp)
 * 
 * Copyright (C) 2010 Vlad Vasiliev
 * Copyright (C) 2010 Tanya Makova
 *       for the code
 * 
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 * 
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
*/
/*******************************************************************************/
#include "livewp-common.h" 
#include "livewp-home-widget.h"
#include <gconf/gconf-client.h>
#include "livewp-rules.h"
#include <sys/time.h>
#include <stdlib.h>

#define PLUGIN_NAME "livewp-home-widget.desktop-0"
#define GCONF_KEY_POSITION "/apps/osso/hildon-desktop/applets/%s/position"
#define GCONF_KEY_MODIFIED "/apps/osso/hildon-desktop/applets/%s/modified"
#define GCONF_KEY_VIEW     "/apps/osso/hildon-desktop/applets/%s/view"

HD_DEFINE_PLUGIN_MODULE (AWallpaperPlugin, animation_wallpaper_plugin, HD_TYPE_HOME_PLUGIN_ITEM)
#define Animation_Wallpaper_HOME_PLUGIN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE (obj,\
                                                          Animation_Wallpaper_TYPE_HOME_PLUGIN,\
                                                          Animation_WallpaperPrivate))

/* Position of plugin on desktop */
#define Xstartposition 700 
#define Ystartposition 448 

gint xapplet = 0, yapplet = 0;
Scene scene;

static void
lw_applet_realize (GtkWidget *widget)
{
      GdkScreen *screen;

      screen = gtk_widget_get_screen (widget);
      gtk_widget_set_colormap (widget,
                                gdk_screen_get_rgba_colormap (screen));
      gtk_widget_set_app_paintable (widget,
                                TRUE);
      GTK_WIDGET_CLASS (animation_wallpaper_plugin_parent_class)->realize (widget);
}


static gboolean
lw_applet_expose_event(GtkWidget      *widget,
                                        GdkEventExpose *event)
{
  cairo_t *cr;

  /* Create cairo context */
  cr = gdk_cairo_create (GDK_DRAWABLE (widget->window));
  gdk_cairo_region (cr, event->region);
  cairo_clip (cr);

  /* Draw alpha background */
  cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
  cairo_set_source_rgba (cr, 0.0, 0.0, 0.0, 0.0);
  cairo_paint (cr);

  /* Free context */
  cairo_destroy (cr);

  return GTK_WIDGET_CLASS (animation_wallpaper_plugin_parent_class)->expose_event (widget,
                                                                                  event);
}

static gboolean
expose_event (GtkWidget *widget,GdkEventExpose *event,
     gpointer data)
{
    cairo_t *cr;
    GdkPixbuf *pixbuf = (GdkPixbuf *) data;
	
    cr = gdk_cairo_create(widget->window);
    if (cr){
        gdk_cairo_region(cr, event->region);
        cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
        gdk_cairo_set_source_pixbuf(cr, pixbuf, 0.0, 0.0);
        cairo_paint(cr);
        cairo_destroy(cr);
    }
    return TRUE;
}

static void
realize (GtkWidget *widget)
{
    GdkScreen *screen;
    screen = gtk_widget_get_screen (widget);
    gtk_widget_set_colormap (widget, gdk_screen_get_rgba_colormap (screen));
}

/* Set position of widget on desktop */
static void
init_applet_position(void)
{
  GSList *position = NULL;
  gchar *position_key;
  gchar *modified_key;
  gchar *modified;
  GError *error = NULL;
  GConfClient   *gconf_client = gconf_client_get_default ();
  position_key = g_strdup_printf (GCONF_KEY_POSITION, PLUGIN_NAME);
  position = gconf_client_get_list (gconf_client,
                                    position_key,
                                    GCONF_VALUE_INT,
                                    NULL);
  if (position && position->data && position->next->data){
  	xapplet = GPOINTER_TO_INT (position->data);
  	yapplet = GPOINTER_TO_INT (position->next->data);
  }else{
	position = g_slist_prepend (g_slist_prepend (NULL,
                                      GINT_TO_POINTER (Ystartposition)),
			              GINT_TO_POINTER (Xstartposition));
	gconf_client_set_list (gconf_client,
                               position_key,
                               GCONF_VALUE_INT,
	                       position,
			       &error);
	xapplet = Xstartposition;
	yapplet = Ystartposition;
  }
  g_free (position_key);
  modified = g_strdup_printf ("%i", 0);
  modified_key = g_strdup_printf (GCONF_KEY_MODIFIED, PLUGIN_NAME);
  gconf_client_set_string (gconf_client,
                           modified_key,
                           modified,
                           &error);
  g_free(modified);
  g_free(modified_key);
  gconf_client_clear_cache(gconf_client);
  g_object_unref(gconf_client);
}

void
actor_set_position_full(GtkWidget *actor, gint x, gint y, gint z)
{
/*  fprintf(stderr, "actor_set_position_full z=%d\n", z); */
    hildon_animation_actor_set_position_full (HILDON_ANIMATION_ACTOR (actor),x-xapplet, y-yapplet, z);
}

static gint 
path_line(gint x0, gint x1, double t)
{
    // уравниение прямой
    return ((x1 - x0) * t + x0);
}


void
destroy_hildon_actor(Actor *actor)
{
    //fprintf(stderr, "destroy_hildon_actor %s\n",actor->name);
    gtk_widget_destroy(actor->widget);
    actor->widget = NULL;
}

static Actor* 
init_object(AWallpaperPlugin *desktop_plugin, 
            gchar * name, 
            gchar * filename, 
            gint x, 
            gint y, 
            gint z, 
            gint width, 
            gint height, 
            gboolean visible, 
            gint scale, 
            gint opacity, 
            void (*pfunc_change)(Actor*),
            void (*pfunc_probability)(Actor*)
           )
{
  Actor *actor = NULL;
  actor = g_new0(Actor, 1);
  actor->x = x;
  actor->y = y;
  actor->z = z;
  actor->width = width;
  actor->height = height;
  actor->visible = visible;
  actor->scale = scale;
  actor->opacity = opacity;
  actor->filename = g_strdup(filename);
  actor->name = g_strdup(name);
  actor->func_change = (gpointer)pfunc_change; 
  actor->func_probability = (gpointer)pfunc_probability;
  if (visible)
    create_hildon_actor(actor, desktop_plugin);
  else 
    actor->widget = NULL;
  actor->time_start_animation = 0;
  actor->duration_animation = 0;
  /*
  a.widget = actor;
  a.name = name;
  a.x = x;
  a.y = y;
  a.z = z;
  */
  //objects_list = g_slist_append(objects_list, G_OBJECT(actor));
  //objects_list = g_slist_append(objects_list, G_OBJECT(a));
  return actor;
}

gint 
rnd(gint max)
{
    srand(time(NULL));
    return rand() % max;
}
gint fast_rnd(gint max)
{
    guint offset = 12923;
    guint multiplier = 4079;
    
    scene.seed = scene.seed * multiplier + offset;
    return (gint)(scene.seed % max);
}

gint 
probability_sun()
{
    /* update sun position after ...  second */
    return 60;
}


gint 
probability_plane()
{
    return (fast_rnd(10) + 1) * 60;
}

void 
change_moon(Actor * actor, AWallpaperPlugin *desktop_plugin)
{
    gint phase;
    char *newfile;

    if (actor){
        if (scene.daytime == TIME_NIGHT){
            if (!actor->visible){
                actor->visible = TRUE;
                phase = get_moon_phase();
                newfile = g_strdup_printf( "%s%d.png", actor->name, phase);
                if (actor->filename)
                    g_free(actor->filename);
                actor->filename = newfile;
                create_hildon_actor(actor, desktop_plugin);

            }
            //actor->x = 400;
            //actor->y = 10;
           // actor_set_position_full(actor->widget, x, y, actor->z);
            //probability_sun(actor);
            //fprintf(stderr, "after change sun %d\n", actor->time_start_animation);
         }else if (actor->visible){
            actor->visible = FALSE;
            fprintf(stderr, "destroy moon \n");
            destroy_hildon_actor(actor);
            /* TO DO make moonrise*/
            actor->time_start_animation = 0;
        } 
    }
    
}

void 
change_sun(Actor * actor, AWallpaperPlugin *desktop_plugin)
{
    double alt, azm;
    gint x, y;

    //fprintf(stderr, "change sun\n");
    if (actor){
        if (scene.daytime != TIME_NIGHT){
            if (!actor->visible){
                actor->visible = TRUE;
                create_hildon_actor(actor, desktop_plugin);
            }
            get_sun_pos(&alt, &azm);
            get_sun_screen_pos(alt, azm, &x, &y);
            actor->x = x;
            actor->y = y;
            actor_set_position_full(actor->widget, x, y, actor->z);
            //probability_sun(actor);
            actor->time_start_animation = time(NULL) + probability_sun();
         }else if (actor->visible){
            actor->visible = FALSE;
            destroy_hildon_actor(actor);
            actor->time_start_animation = 0;
        } 
    }
    
}

static void 
change_tram(Actor * actor, AWallpaperPlugin *desktop_plugin)
{
    gint x0 = -300, y0 = 225, scale0 = 100,
         x1 = 800, y1 = 162, scale1 = 130, 
         x, y, scale;
    struct timeval tvb;     
    suseconds_t ms;
    long sec;
    double t;

    //fprintf(stderr, "change tram\n");
    gettimeofday(&tvb, NULL);
    
    ms = tvb.tv_usec;
    sec = tvb.tv_sec;
    
    if (!actor->visible){
        actor->visible = TRUE;
        if (scene.daytime == TIME_NIGHT){
            if (actor->filename)
                g_free(actor->filename);
            actor->filename = g_strdup("tram_dark.png");
        } else{
            if (actor->filename)
                g_free(actor->filename);
            actor->filename = g_strdup("tram.png");
        }
        create_hildon_actor(actor, desktop_plugin);
    }
    t = (double)((double)sec+(double)ms/1000000 - actor->time_start_animation) / actor->duration_animation;
    x = path_line(x0, x1, t);
    y = path_line(y0, y1, t);
    scale = path_line(scale0, scale1, t);
    //fprintf(stderr, "change tram t=%f x=%d y=%d scale=%d\n", t, x, y, scale);
    actor_set_position_full(actor->widget, x, y, actor->z);
    hildon_animation_actor_set_scale(HILDON_ANIMATION_ACTOR(actor->widget), (double)scale/100, (double)scale/100);
    if (t >= 1){
        /* stop animation */
        actor->visible = FALSE;
        destroy_hildon_actor(actor);
        actor->time_start_animation = sec + fast_rnd(300);
    }
}

void
change_plane1(Actor *actor, AWallpaperPlugin *desktop_plugin)
{
    gint x0 = 620, y0 = 233, scale0 = 100,
         x1 = 79, y1 = -146, scale1 = 100, 
         x, y, scale;
    struct timeval tvb;     
    suseconds_t ms;
    long sec;
    double t;

    gettimeofday(&tvb, NULL);
    
    ms = tvb.tv_usec;
    sec = tvb.tv_sec;
//    fprintf(stderr, "1 %f - %d\n", sec+(double)ms/100000, now);
   
    if (scene.daytime != TIME_NIGHT){
        if (actor->time_start_animation == 0){
            actor->time_start_animation = sec + probability_plane();
            return;
        }
    }
    if (!actor->visible){
        actor->visible = TRUE;
        create_hildon_actor(actor, desktop_plugin);
    }
    t = (double)((double)sec+(double)ms/1000000 - actor->time_start_animation) / actor->duration_animation;
    x = path_line(x0, x1, t);
    y = path_line(y0, y1, t);
    //scale = path_line(scale0, scale1, t);
    //fprintf(stderr, "change tram t=%f x=%d y=%d scale=%d\n", t, x, y, scale);
    actor_set_position_full(actor->widget, x, y, actor->z);
    //hildon_animation_actor_set_scale(HILDON_ANIMATION_ACTOR(actor->widget), (double)scale/100, (double)scale/100);
    if (t >= 1){
        /* stop animation */
        actor->visible = FALSE;
        destroy_hildon_actor(actor);
        if (scene.daytime == TIME_NIGHT) 
            actor->time_start_animation = 0;
        else 
            actor->time_start_animation = sec + probability_plane();
    }

}

void
change_plane2(Actor *actor, AWallpaperPlugin *desktop_plugin)
{
    gint x0 = -actor->width, y0 = 45, scale0 = 100,
         x1 = 800, y1 = 20, scale1 = 100, 
         x, y, scale;
    struct timeval tvb;     
    suseconds_t ms;
    long sec;
    double t;

    gettimeofday(&tvb, NULL);
    
    ms = tvb.tv_usec;
    sec = tvb.tv_sec;
//    fprintf(stderr, "1 %f - %d\n", sec+(double)ms/100000, now);
    if (scene.daytime != TIME_NIGHT){
        if (actor->time_start_animation == 0){
            actor->time_start_animation = sec + probability_plane();
            return;
        }
    }
    if (!actor->visible){
        actor->visible = TRUE;
        create_hildon_actor(actor, desktop_plugin);
    }

    t = (double)((double)sec+(double)ms/1000000 - actor->time_start_animation) / actor->duration_animation;
    x = path_line(x0, x1, t);
    y = path_line(y0, y1, t);
    //scale = path_line(scale0, scale1, t);
    //fprintf(stderr, "change tram t=%f x=%d y=%d scale=%d\n", t, x, y, scale);
    actor_set_position_full(actor->widget, x, y, actor->z);
    //hildon_animation_actor_set_scale(HILDON_ANIMATION_ACTOR(actor->widget), (double)scale/100, (double)scale/100);
    if (t >= 1){
        /* stop animation */
        actor->visible = FALSE;
        destroy_hildon_actor(actor);
        if (scene.daytime == TIME_NIGHT) 
            actor->time_start_animation = 0;
        else 
            actor->time_start_animation = sec + probability_plane();
    }

}

void
change_cloud(Actor *actor, AWallpaperPlugin *desktop_plugin)
{
    gint x0, y0 = 300, scale0 = 100,
         x1, y1 = -actor->height, scale1 = 150, 
         x, y, scale;
    struct timeval tvb;     
    suseconds_t ms;
    long sec;
    double t;

    //fprintf(stderr, "change cloud\n");
    gettimeofday(&tvb, NULL);
    
    ms = tvb.tv_usec;
    sec = tvb.tv_sec;
    //fprintf(stderr, "c1oud %s - y0=%d\n", actor->name, actor->y);
   
    if (!actor->visible){
        actor->visible = TRUE;
        if (scene.daytime == TIME_NIGHT)
            actor->filename = actor->filename;
        else
            actor->filename = actor->filename;
        create_hildon_actor(actor, desktop_plugin);
    }
    t = (double)((double)sec+(double)ms/1000000 - actor->time_start_animation) / actor->duration_animation;
    
    if (scene.wind_orientation == 1){
        x0 = -actor->width;
        x1 = 800;
    }
    else {
        x0 = 800;
        x1 = -actor->width;
    }

    x = path_line(x0, x1, t);    
    y = -scene.wind_angle * (x - x0) + actor->y;
    scale = path_line(scale0, scale1, (double)(y - y0)/(y1 - y0));

    //fprintf(stderr, "change cloud t=%f x=%d y=%d scale=%d\n", t, x, y, scale);
    actor_set_position_full(actor->widget, x, y, actor->z);
    hildon_animation_actor_set_scale(HILDON_ANIMATION_ACTOR(actor->widget), (double)scale/100, (double)scale/100);
    if ((y < y1 || y > y0) || t >= 1){
        /* stop animation */
        actor->visible = FALSE;
        destroy_hildon_actor(actor);
        actor->time_start_animation = sec + fast_rnd(300);
        actor->y = fast_rnd(300);
    }

}

void
change_wind(Actor *actor, AWallpaperPlugin *desktop_plugin)
{
    scene.wind_orientation = fast_rnd(2);
    if (scene.wind_orientation == 0) scene.wind_orientation = -1;
    scene.wind_angle = (double)(fast_rnd(200) - 100) / 100;
    actor->time_start_animation = time(NULL) + (fast_rnd(10) + 10) * 60;
    //fprintf(stderr, "change wind orient = %d angle = %f after = %d\n", scene.wind_orientation, scene.wind_angle, actor->time_start_animation-time(NULL));
}

void 
change_window1(Actor * actor, AWallpaperPlugin *desktop_plugin)
{
    
    gint now = time(NULL);

    if (scene.daytime == TIME_DAY){
        if (actor->visible){
            actor->visible = FALSE;
            destroy_hildon_actor(actor);
        }
        actor->time_start_animation = 0;
        return;
    }else if (actor->time_start_animation == 0){
        actor->time_start_animation = now + fast_rnd(30);
        return;
    }

    if (!actor->visible){
        actor->visible = TRUE;
        create_hildon_actor(actor, desktop_plugin);
        actor->time_start_animation = now + fast_rnd(60) + 10;
    }else {
        actor->visible = FALSE;
        destroy_hildon_actor(actor);
        actor->time_start_animation = now + fast_rnd(60) + 10;
    }
   
}

void
create_hildon_actor(Actor *actor, AWallpaperPlugin *desktop_plugin) 
{
  GtkWidget *ha = NULL;
  GdkPixbuf *pixbuf = NULL;
  GtkWidget *image = NULL;
  gchar     *str = NULL;

  /* fprintf(stderr, "create_hildon_actor %s\n", actor->name);*/
  ha = hildon_animation_actor_new();
  str = g_strdup_printf( "%s/%s/%s", THEME_PATH, 
                        desktop_plugin->priv->theme, actor->filename);
  pixbuf = gdk_pixbuf_new_from_file_at_size (str, 
                                             actor->width, 
                                             actor->height, 
                                             NULL);
  if (str)
      g_free(str);
  if (pixbuf){
      image = gtk_image_new_from_pixbuf (pixbuf);
      g_object_unref(G_OBJECT(pixbuf));
  }
  if (image){
    g_signal_connect(G_OBJECT(image), "expose_event",
                           G_CALLBACK(expose_event), pixbuf);
    gtk_container_add (GTK_CONTAINER (ha), image);
  }  
  actor_set_position_full(ha, actor->x, actor->y, actor->z);
  hildon_animation_actor_set_show (HILDON_ANIMATION_ACTOR(ha), actor->visible);
  hildon_animation_actor_set_scale(HILDON_ANIMATION_ACTOR(ha), (double)actor->scale/100, (double)actor->scale/100);
  realize(ha);
  gtk_widget_show_all(ha);

  /* TO DO check it */
  /*  gdk_flush (); */

  //g_object_set_data(G_OBJECT(ha), "name", name);
  //g_object_set_data(G_OBJECT(ha), "filename", filename);
  g_object_set_data(G_OBJECT(ha), "image", image);
  /*
  g_object_set_data(G_OBJECT(ha), "x", x);
  g_object_set_data(G_OBJECT(ha), "y", y);
  g_object_set_data(G_OBJECT(ha), "z", z);
  g_object_set_data(G_OBJECT(ha), "width", width);
  g_object_set_data(G_OBJECT(ha), "height", height);
  g_object_set_data(G_OBJECT(ha), "scale", scale);
  g_object_set_data(G_OBJECT(ha), "visible", visible);
  g_object_set_data(G_OBJECT(ha), "opacity", opacity);
  g_object_set_data(G_OBJECT(ha), "func", pfunc);
  */
  hildon_animation_actor_set_parent (HILDON_ANIMATION_ACTOR (ha), GTK_WINDOW(desktop_plugin));
  actor->widget = ha;
}


void
change_hildon_actor(Actor *actor, AWallpaperPlugin *desktop_plugin)
{
    GtkWidget *image = NULL;
    GdkPixbuf *pixbuf = NULL;
    gchar     *str = NULL;

    str = g_strdup_printf( "%s/%s/%s", THEME_PATH, 
                            desktop_plugin->priv->theme, actor->filename);
 
    pixbuf = gdk_pixbuf_new_from_file_at_size (str, 
                                               actor->width, 
                                               actor->height, 
                                               NULL);
    if(str)
        g_free(str);
    if (pixbuf){
        image = gtk_image_new_from_pixbuf (pixbuf);
        g_object_unref(G_OBJECT(pixbuf));
    }
    if (image){ 
        g_signal_connect(G_OBJECT(image), "expose_event",
                                       G_CALLBACK(expose_event), pixbuf);
        if (g_object_get_data(G_OBJECT(actor->widget), "image")){
            gtk_container_remove(GTK_CONTAINER(actor->widget), g_object_get_data(G_OBJECT(actor->widget), "image"));  
        }
        g_object_set_data(G_OBJECT(actor->widget), "image", image);
        gtk_container_add (GTK_CONTAINER (actor->widget), image);
        realize(actor->widget);
        gtk_widget_show_all(actor->widget);
        /* TO DO check it */
       /*  gdk_flush (); */


    }

}


void 
change_static_actor(Actor * actor, AWallpaperPlugin *desktop_plugin)
{
    gchar *newfile;
    newfile = g_strdup_printf("%s%d.png", actor->name, scene.daytime); 
    if (actor->filename)
            g_free(actor->filename);
    actor->filename = newfile;
    change_hildon_actor(actor, desktop_plugin);
}

void 
change_static_actor_with_corner(Actor * actor, AWallpaperPlugin *desktop_plugin)
{
    gchar *newfile;
    gchar           buffer[2048];

    if (desktop_plugin->priv->right_corner)
        gtk_widget_destroy(desktop_plugin->priv->right_corner);
    snprintf(buffer, sizeof(buffer) - 1, "%s/%s/town%i_right_corner.png", \
                                  THEME_PATH, desktop_plugin->priv->theme, scene.daytime);
    desktop_plugin->priv->right_corner = gtk_image_new_from_file (buffer);
    if (desktop_plugin->priv->right_corner){
        gtk_fixed_put(GTK_FIXED(desktop_plugin->priv->main_widget), desktop_plugin->priv->right_corner, 0, 0);
        gtk_widget_show (desktop_plugin->priv->right_corner);
    }
    change_static_actor(actor, desktop_plugin);

}


/*
static gint 
get_time(gint t){
    // уравнение изменения времени
    return t*1.1;
}
*/
static void 
destroy_scene(void){
    GSList * tmp = scene.actors;
    Actor *actor;
    while (tmp != NULL){
        actor = tmp->data;
        if (actor){
            if (actor->filename)
                g_free(actor->filename);
            if (actor->name)
                g_free(actor->name);
            gtk_widget_destroy(actor->widget);
            //actor->widget = NULL;
            g_free(actor);
        }
        tmp = g_slist_next(tmp);
    }
    g_slist_free(tmp);

}

void
reload_scene(AWallpaperPlugin *desktop_plugin)
{
    fprintf(stderr,"Reload scene %s\n", desktop_plugin->priv->theme);  
    destroy_scene();
    if (!strcmp(desktop_plugin->priv->theme,"Modern"))
        init_scene(desktop_plugin);
    else if (!strcmp(desktop_plugin->priv->theme,"Berlin")) 
        init_scene1(desktop_plugin);
}


/* Init Modern Scene */
static void
init_scene(AWallpaperPlugin *desktop_plugin)
{
  Actor *actor;
  gint now = time(NULL);
  gint i;

  /* fprintf(stderr, "init scene \n");*/
  scene.daytime = get_daytime();
  scene.actors = NULL;
  scene.wind_orientation = -1;
  scene.wind_angle = 0.3;
  /* init value for random */
  scene.seed = time(NULL);

  

  actor = init_object(desktop_plugin, "sky", "sky0.png", 0, 0, 5, 800, 480, 
                      TRUE, 100, 255, (gpointer)&change_static_actor, NULL);
  change_static_actor(actor, desktop_plugin);
  scene.actors = g_slist_append(scene.actors, actor);

  actor = init_object(desktop_plugin, "sun", "sun.png", 0, 0, 6, 88, 88, 
                      TRUE, 100, 255, (gpointer)&change_sun, &probability_sun);
  actor->time_start_animation = now;
  actor->duration_animation = G_MAXINT;
  change_sun(actor, desktop_plugin);
  scene.actors = g_slist_append(scene.actors, actor);

  //actor = init_object(desktop_plugin, "dot", "dot1.png", 0, 0, 11, 50, 50, 
    //                  TRUE, 100, 255, NULL, NULL);
  //scene.actors = g_slist_append(scene.actors, actor);
  
  actor = init_object(desktop_plugin, "cloud1", "cloud1.png", 0, fast_rnd(300)-97, 7, 150, 97, 
                      FALSE, 100, 255, (gpointer)&change_cloud, NULL);
  actor->time_start_animation = now + fast_rnd(180);
  actor->duration_animation = 3*60;
  scene.actors = g_slist_append(scene.actors, actor);
  
  actor = init_object(desktop_plugin, "cloud2", "cloud2.png", 0, fast_rnd(300)-75, 7, 188, 75, 
                      FALSE, 100, 255, (gpointer)&change_cloud, NULL);
  actor->time_start_animation = now + fast_rnd(180);
  actor->duration_animation = 3*60;
  scene.actors = g_slist_append(scene.actors, actor);

  actor = init_object(desktop_plugin, "cloud4", "cloud4.png", 0, fast_rnd(300)-75, 7, 150, 75, 
                      FALSE, 100, 255, (gpointer)&change_cloud, NULL);
  actor->time_start_animation = now + fast_rnd(180);
  actor->duration_animation = 5*60;
  scene.actors = g_slist_append(scene.actors, actor);


  actor = init_object(desktop_plugin, "town", "town0.png", 0, 0, 8, 800, 480, 
                      TRUE, 100, 255, (gpointer)&change_static_actor, NULL);
  change_static_actor(actor, desktop_plugin);
  scene.actors = g_slist_append(scene.actors, actor);

  actor = init_object(desktop_plugin, "tram", "tram.png", -300, 225, 9, 350, 210, 
                      FALSE, 100, 255, (gpointer)&change_tram, NULL);
  actor->time_start_animation = time(NULL) + fast_rnd(60);
  actor->duration_animation = 60;
  scene.actors = g_slist_append(scene.actors, actor);

  actor = init_object(desktop_plugin, "border", "border0.png", 0, 480-79, 10, 800, 79,
                      TRUE, 100, 255, (gpointer)&change_static_actor_with_corner, NULL);
  change_static_actor_with_corner(actor, desktop_plugin);
  scene.actors = g_slist_append(scene.actors, actor);
  
  actor = init_object(desktop_plugin, "moon", "moon1.png", 400, 20, 6, 60, 60, 
                      FALSE, 100, 255, (gpointer)&change_moon, NULL);
  change_moon(actor, desktop_plugin);
  scene.actors = g_slist_append(scene.actors, actor);

  actor = init_object(desktop_plugin, "wind", "", 0, 0, 5, 0, 0, 
                      FALSE, 100, 255, (gpointer)&change_wind, NULL);
  change_wind(actor, desktop_plugin);
  scene.actors = g_slist_append(scene.actors, actor);

    /* windows in 4-th house  */
    gint winds[13][2];

    winds[0][0] = 482;
    winds[0][1] = 180;

    winds[1][0] = 495;
    winds[1][1] = 179;

    winds[2][0] = 482;
    winds[2][1] = 191;

    winds[3][0] = 495;
    winds[3][1] = 190;
    
    winds[4][0] = 482;
    winds[4][1] = 201;
    
    winds[5][0] = 495;
    winds[5][1] = 210;
    
    winds[6][0] = 482;
    winds[6][1] = 222;
    
    winds[7][0] = 495;
    winds[7][1] = 221;
    
    winds[8][0] = 459;
    winds[8][1] = 203;
    
    winds[9][0] = 495;
    winds[9][1] = 241;
    
    winds[10][0] = 495;
    winds[10][1] = 252;
    
    winds[11][0] = 482;
    winds[11][1] = 273;
    
    winds[12][0] = 495;
    winds[12][1] = 303;
    for (i=0; i<13; i++){
        actor = init_object(desktop_plugin, "window1", "window1.png", 
                            winds[i][0], winds[i][1], 8, 8, 10, 
                            FALSE, 100, 255, (gpointer)&change_window1, NULL);
        //change_window1(actor, desktop_plugin);
        actor->time_start_animation = now + fast_rnd(30);
        scene.actors = g_slist_append(scene.actors, actor);

    }
    
    /* windows in 1-th house  */
    
    winds[0][0] = 86;
    winds[0][1] = 321;

    winds[1][0] = 86;
    winds[1][1] = 363;

    winds[2][0] = 86;
    winds[2][1] = 385;

    winds[3][0] = 86;
    winds[3][1] = 286;
    
    winds[4][0] = 94;
    winds[4][1] = 232;
    
    winds[5][0] = 94;
    winds[5][1] = 243;
    
    winds[6][0] = 94;
    winds[6][1] = 265;
    
    winds[7][0] = 94;
    winds[7][1] = 331;
    for (i=0; i<8; i++){
        actor = init_object(desktop_plugin, "window2", "window2.png", 
                            winds[i][0], winds[i][1], 8, 8, 10, 
                            TRUE, 100, 255, (gpointer)&change_window1, NULL);
        //change_window1(actor, desktop_plugin);
        actor->time_start_animation = now + fast_rnd(30);
        scene.actors = g_slist_append(scene.actors, actor);

    }
    
    /* windows in 3-th house  */
    
    winds[0][0] = 251;
    winds[0][1] = 162;

    winds[1][0] = 251;
    winds[1][1] = 196;

    winds[2][0] = 251;
    winds[2][1] = 278;

    winds[3][0] = 251;
    winds[3][1] = 289;
    
    winds[4][0] = 313;
    winds[4][1] = 173;
    
    winds[5][0] = 322;
    winds[5][1] = 160;
    
    winds[6][0] = 303;
    winds[6][1] = 217;
    
    winds[7][0] = 322;
    winds[7][1] = 224;
    
    winds[8][0] = 323;
    winds[8][1] = 217;
    
    winds[9][0] = 322;
    winds[9][1] = 288;
    
    for (i=0; i<10; i++){
        actor = init_object(desktop_plugin, "window3", "window3.png", 
                            winds[i][0], winds[i][1], 8, 8, 10, 
                            FALSE, 100, 255, (gpointer)&change_window1, NULL);
        //change_window1(actor, desktop_plugin);
        actor->time_start_animation = now + fast_rnd(30);
        scene.actors = g_slist_append(scene.actors, actor);

    }

    /* windows in 5-th house  */
    
    winds[0][0] = 610;
    winds[0][1] = 224;

    winds[1][0] = 602;
    winds[1][1] = 245;

    winds[2][0] = 602;
    winds[2][1] = 264;

    winds[3][0] = 610;
    winds[3][1] = 301;
    
    winds[4][0] = 610;
    winds[4][1] = 320;
    
    winds[5][0] = 593;
    winds[5][1] = 352;
    
    winds[6][0] = 610;
    winds[6][1] = 368;
    
    for (i=0; i<7; i++){
        actor = init_object(desktop_plugin, "window4", "window4.png", 
                            winds[i][0], winds[i][1], 8, 8, 10, 
                            FALSE, 100, 255, (gpointer)&change_window1, NULL);
        //change_window1(actor, desktop_plugin);
        actor->time_start_animation = now + fast_rnd(30);
        scene.actors = g_slist_append(scene.actors, actor);

    }

    /* windows in 6-th house  */
    
    winds[0][0] = 717;
    winds[0][1] = 283;

    winds[1][0] = 698;
    winds[1][1] = 293;

    winds[2][0] = 717;
    winds[2][1] = 315;

    winds[3][0] = 717;
    winds[3][1] = 323;
    
    winds[4][0] = 698;
    winds[4][1] = 362;
    
    winds[5][0] = 698;
    winds[5][1] = 400;
    
    for (i=0; i<6; i++){
        actor = init_object(desktop_plugin, "window5", "window5.png", 
                            winds[i][0], winds[i][1], 8, 8, 10, 
                            FALSE, 100, 255, (gpointer)&change_window1, NULL);
        //change_window1(actor, desktop_plugin);
        actor->time_start_animation = now + fast_rnd(30);
        scene.actors = g_slist_append(scene.actors, actor);

    }

#if 0    
  anim = g_new0(Animation, 1);
  anim->count = 1;
  anim->actor = actor;
  anim->func_change = &change_tram;
  anim->func_time = NULL;
  anim->timestart = time(NULL); 
  anim->timeall = 10;
  
  scene.dynamic_actors = g_slist_append(scene.dynamic_actors, anim);
#endif  
 }

/* Init Berlin Scene */
static void
init_scene1(AWallpaperPlugin *desktop_plugin)
{
  Actor *actor;
  gint now = time(NULL);
  

  scene.daytime = get_daytime();
  scene.actors = NULL;
  scene.wind_orientation = -1;
  scene.wind_angle = 0.3;
  /* init value for random */
  scene.seed = time(NULL);

  
  actor = init_object(desktop_plugin, "sky", "sky.png", 0, 0, 5, 800, 480, 
                      TRUE, 100, 255, (gpointer)&change_static_actor, NULL);
  change_static_actor(actor, desktop_plugin);
  scene.actors = g_slist_append(scene.actors, actor);

  
  actor = init_object(desktop_plugin, "sun", "sun.png", 0, 0, 6, 88, 88, 
                      TRUE, 100, 255, (gpointer)&change_sun, &probability_sun);
  actor->time_start_animation = time(NULL);
  actor->duration_animation = G_MAXINT;
  change_sun(actor, desktop_plugin);
  scene.actors = g_slist_append(scene.actors, actor);

#if 0
  actor = init_object(desktop_plugin, "dot", "dot1.png", 0, 0, 11, 50, 50, 
                      TRUE, 100, 255, NULL, NULL);
  scene.actors = g_slist_append(scene.actors, actor);
#endif

  actor = init_object(desktop_plugin, "moon", "moon1.png", 400, 15, 6, 60, 60, 
                      FALSE, 100, 255, (gpointer)&change_moon, NULL);
  change_moon(actor, desktop_plugin);
  scene.actors = g_slist_append(scene.actors, actor);
  
  actor = init_object(desktop_plugin, "cloud1", "cloud1.png", 0, fast_rnd(300)-97, 7, 150, 97, 
                      FALSE, 100, 255, (gpointer)&change_cloud, NULL);
  actor->time_start_animation = now + fast_rnd(180);
  actor->duration_animation = 3*60;
  scene.actors = g_slist_append(scene.actors, actor);
  
  actor = init_object(desktop_plugin, "cloud2", "cloud2.png", 0, fast_rnd(300)-75, 7, 188, 75, 
                      FALSE, 100, 255, (gpointer)&change_cloud, NULL);
  actor->time_start_animation = now + fast_rnd(180);
  actor->duration_animation = 3*60;
  scene.actors = g_slist_append(scene.actors, actor);

  actor = init_object(desktop_plugin, "cloud4", "cloud4.png", 0, fast_rnd(300)-75, 7, 150, 75, 
                      FALSE, 100, 255, (gpointer)&change_cloud, NULL);
  actor->time_start_animation = now + fast_rnd(180);
  actor->duration_animation = 5*60;
  scene.actors = g_slist_append(scene.actors, actor);

 
  actor = init_object(desktop_plugin, "plane2", "plane3.png", 0, 45, 8, 160, 50, 
                      FALSE, 100, 255, (gpointer)&change_plane2, NULL);
  actor->time_start_animation = now + probability_plane();
  actor->duration_animation = 60;
  scene.actors = g_slist_append(scene.actors, actor);
  
  actor = init_object(desktop_plugin, "plane1", "tu154_small.png", 620, 233, 9, 300, 116, 
                      FALSE, 100, 255, (gpointer)&change_plane1, NULL);
  actor->time_start_animation = now + probability_plane();
  actor->duration_animation = 30;
  scene.actors = g_slist_append(scene.actors, actor);

  actor = init_object(desktop_plugin, "town", "town.png", 0, 0, 10, 800, 480, 
                      TRUE, 100, 255, (gpointer)&change_static_actor_with_corner, NULL);
  change_static_actor_with_corner(actor, desktop_plugin);
  scene.actors = g_slist_append(scene.actors, actor);

  actor = init_object(desktop_plugin, "wind", "", 0, 0, 5, 0, 0, 
                      FALSE, 100, 255, (gpointer)&change_wind, NULL);
  change_wind(actor, desktop_plugin);
  scene.actors = g_slist_append(scene.actors, actor);

}


void 
get_sun_screen_pos(double alt, double azm, gint * x, gint * y)
{
    gint y0 = 365;// - уровень горизонта
    *x = (int)(azm * 800) - 64;
    *y = (int)((1 - alt) * y0) - 64;
    //fprintf(stderr, "sun pos alt=%f azm=%f x=%d y=%d\n", alt, azm, *x, *y);
}
#if 0
static void 
change_actor(GtkWidget * actor)
{
    char * name;
    gint x, y, daytime, scale;
    gdouble sc;
    double alt, azm;

    GtkWidget *image;
    GdkPixbuf *pixbuf;

    void (*pfunc)(gpointer, gpointer);

    name = g_object_get_data(G_OBJECT(actor), "name");
    fprintf(stderr, "change actor %s\n", name);
    if (name == "sun"){
        pfunc = g_object_get_data(G_OBJECT(actor), "func");
        if (pfunc)
            (*pfunc)(actor, g_strdup(name));
        daytime = get_daytime();
        if (daytime != TIME_NIGHT){
            hildon_animation_actor_set_show(actor, 1);
            get_sun_pos(&alt, &azm);
            get_sun_screen_pos(alt, azm, &x, &y);
            actor_set_position_full(actor, x, y, g_object_get_data(G_OBJECT(actor), "z"));
        }
    }
    
    if (name == "cloud1"){
        x = g_object_get_data(G_OBJECT(actor), "x");
        y = g_object_get_data(G_OBJECT(actor), "y");
        scale = g_object_get_data(G_OBJECT(actor), "scale");

        /* Start */
        image = g_object_get_data(G_OBJECT(actor), "image");
        
        gtk_container_remove(actor, image);  
        pixbuf = gdk_pixbuf_new_from_file_at_size ("/usr/share/livewp/theme/Modern/sun.png", 
                                             200, 
                                             200, 
                                             NULL);
        if (pixbuf){
              image = gtk_image_new_from_pixbuf (pixbuf);
              g_object_unref(G_OBJECT(pixbuf));
        }
        g_signal_connect(G_OBJECT(image), "expose_event",
                                   G_CALLBACK(expose_event), pixbuf);
        gtk_container_add (GTK_CONTAINER (actor), image);
        realize(actor);
        gtk_widget_show_all(actor);
        /* End*/

            
        x += 40;
        y -= 20;
        scale -= 10;
        if (x > 500){
            x = 400;
            y = 150;
            sc = 1;
        }
        sc = (double)scale / 100;
        hildon_animation_actor_set_scale(actor, sc, sc);
        fprintf(stderr, "cloud x=%d y=%d scale=%f", x, y, sc);
        actor_set_position_full(actor, x, y, g_object_get_data(G_OBJECT(actor), "z"));
        g_object_set_data(G_OBJECT(actor), "x", x);
        g_object_set_data(G_OBJECT(actor), "y", y);
        g_object_set_data(G_OBJECT(actor), "scale", scale);
    }

}
#endif
static gboolean
short_timeout (AWallpaperPlugin *desktop_plugin)
{
    gint daytime = get_daytime();
    GSList * tmp;
    void (*pfunc)(gpointer, gpointer);
    time_t now;
    Actor *actor;
    gboolean stop_flag = TRUE;

    if (!desktop_plugin->priv->visible || !desktop_plugin->priv->rich_animation){
        desktop_plugin->priv->short_timer = 0;
        return FALSE;
    }

    now = time(NULL);
    //scene.daytime = daytime;
    /* fprintf(stderr, "Short timer %d\n", now); */
    tmp = scene.actors;
    while (tmp != NULL){
           actor = tmp->data;
           if (now >= actor->time_start_animation  
               && actor->time_start_animation > 0
               /* && now - actor->time_start_animation <= actor->duration_animation*/){
                pfunc = actor->func_change;
                if (pfunc){ 
                    (*pfunc)(actor, desktop_plugin);
                    stop_flag = FALSE;
                }
            }
            tmp = g_slist_next(tmp);
    }
    if (stop_flag){
         desktop_plugin->priv->short_timer = 0;
         return FALSE;
    }else
         return TRUE; /* keep running this event */
}

void
run_long_timeout(AWallpaperPlugin *desktop_plugin){

    gint daytime = get_daytime();
    GSList * tmp;
    void (*pfunc)(gpointer, gpointer);
    time_t now;
    Actor *actor;


    //fprintf(stderr, "timer daytime=%d\n", daytime);
    if (scene.daytime != daytime){
        scene.daytime = daytime;
        tmp = scene.actors;
        while (tmp != NULL){
            //change_actor(tmp->data);
            pfunc =((Actor*)tmp->data)->func_change;
            if (pfunc){
                (*pfunc)(tmp->data, desktop_plugin);
            }
            tmp = g_slist_next(tmp);
        }
    }
   
    now = time(NULL);
    //fprintf(stderr, "Now  %d\n", now);
    tmp = scene.actors;
    while (tmp != NULL){
        actor = tmp->data;
        if (now >= actor->time_start_animation  
            && actor->time_start_animation > 0
            && desktop_plugin->priv->short_timer == 0){
            /* fprintf(stderr, "start short timer act = %s %d\n", actor->name, now); */
            if (desktop_plugin->priv->rich_animation){
                actor->time_start_animation = now;
                desktop_plugin->priv->short_timer = g_timeout_add(SHORT_TIMER, (GtkFunction)short_timeout, desktop_plugin);
            }
            else {
                (*actor->func_change)(actor, desktop_plugin);
            }
        }
        tmp = g_slist_next(tmp);
    }
 
}

static gboolean
long_timeout (AWallpaperPlugin *desktop_plugin)
{
    /* fprintf(stderr, "long_timeout %i\n", desktop_plugin->priv->long_timer); */
    if (desktop_plugin->priv->long_timer == 0 )
        return FALSE;
    if (!desktop_plugin->priv->visible){
        if(desktop_plugin->priv->short_timer != 0){
            g_source_remove(desktop_plugin->priv->short_timer);
            desktop_plugin->priv->short_timer = 0;
        }
        desktop_plugin->priv->long_timer = 0;
        return FALSE;
    }
  

    run_long_timeout(desktop_plugin);
    return TRUE; /* keep running this event */
}

static void
desktop_plugin_visible_notify (GObject    *object,
                                          GParamSpec *spec,
                                          AWallpaperPlugin *desktop_plugin)
{
    gboolean visible;
    g_object_get (object, "is-on-current-desktop", &visible, NULL);
    /* fprintf (stderr, "is-on-current-desktop changed. visible: %u", visible); */
    if (visible){
        desktop_plugin->priv->visible = TRUE;
        if (desktop_plugin->priv->long_timer == 0 ){
            desktop_plugin->priv->long_timer = g_timeout_add(LONG_TIMER, (GtkFunction)long_timeout, desktop_plugin);
            run_long_timeout(desktop_plugin);
        }
    }else{
        desktop_plugin->priv->visible = FALSE;
        if (desktop_plugin->priv->long_timer != 0 ){
            g_source_remove(desktop_plugin->priv->long_timer);
            desktop_plugin->priv->long_timer = 0;
        }
    }
}

gboolean
rich_animation_press(GtkWidget *widget, GdkEvent *event,
                                            gpointer user_data){
    fprintf(stderr,"gggggggggggggggggggg2222\n");
}    
static void
animation_wallpaper_plugin_init (AWallpaperPlugin *desktop_plugin)
{
    GtkWidget *rich_animation;
    gchar           buffer[2048];


    fprintf(stderr, "!!!!!!!plugin init \n");
    Animation_WallpaperPrivate *priv =  Animation_Wallpaper_HOME_PLUGIN_GET_PRIVATE (desktop_plugin);
    desktop_plugin->priv =  Animation_Wallpaper_HOME_PLUGIN_GET_PRIVATE (desktop_plugin);

    /* Load config */
    read_config(priv);
    priv->desktop_plugin = desktop_plugin;
    desktop_plugin->priv->main_widget = gtk_fixed_new();

    gtk_widget_set_size_request(desktop_plugin->priv->main_widget, 100, 32);
    desktop_plugin->priv->right_corner = NULL;
    snprintf(buffer, sizeof(buffer) - 1, "%s/%s/%s", THEME_PATH, desktop_plugin->priv->theme, "town0_right_corner.png");
    desktop_plugin->priv->right_corner = gtk_image_new_from_file (buffer);
    if (desktop_plugin->priv->right_corner){
        gtk_fixed_put(GTK_FIXED(desktop_plugin->priv->main_widget), desktop_plugin->priv->right_corner, 0, 0);
        gtk_widget_show (desktop_plugin->priv->right_corner);
    }
    /* Create rich animation event */
    rich_animation = gtk_event_box_new();
    if(rich_animation){
        gtk_widget_set_events(rich_animation, GDK_BUTTON_PRESS_MASK);
        gtk_event_box_set_visible_window(GTK_EVENT_BOX(rich_animation), FALSE);
        gtk_widget_set_size_request(rich_animation, 100, 32);
        gtk_widget_show (rich_animation);
        g_signal_connect(rich_animation, "button-press-event", G_CALLBACK(rich_animation_press), desktop_plugin);
        gtk_fixed_put(GTK_FIXED(desktop_plugin->priv->main_widget), rich_animation, 0, 0);
    }
/*
    GtkWidget *label = gtk_label_new ("ddddddddddddd"); 
    gtk_widget_set_size_request(label, 95, 30);
    gtk_widget_show (label);
//    gtk_container_add (GTK_CONTAINER (desktop_plugin), label);
    gtk_fixed_put(GTK_FIXED(widget), label, 0, 0);
*/
    hd_home_plugin_item_set_settings (HD_HOME_PLUGIN_ITEM (desktop_plugin), TRUE);
    g_signal_connect (desktop_plugin, "show-settings",
		             G_CALLBACK (lw_settings), priv);
    gtk_widget_show (desktop_plugin->priv->main_widget);
    gtk_container_add (GTK_CONTAINER (desktop_plugin), desktop_plugin->priv->main_widget);
    init_applet_position();

    
    fprintf(stderr, "!!!theme = %s\n", priv->theme);
    if (!strcmp(priv->theme,"Modern"))
        init_scene(desktop_plugin);
    else if (!strcmp(priv->theme,"Berlin")) 
        init_scene1(desktop_plugin);
    priv->long_timer = g_timeout_add(LONG_TIMER, (GtkFunction)long_timeout, desktop_plugin);
    priv->short_timer = 0;
    /* TODO Move scene to priv */
    scene.timer_type = LONG_TIMER_TYPE;
    g_signal_connect (desktop_plugin, "notify::is-on-current-desktop",
                     G_CALLBACK (desktop_plugin_visible_notify), desktop_plugin);

   
    //sleep(2);
}

static void
lw_applet_finalize (GObject *object)
{
     AWallpaperPlugin *desktop_plugin = Animation_Wallpaper_HOME_PLUGIN (object);
     Animation_WallpaperPrivate *priv = desktop_plugin->priv;
     
     fprintf(stderr,"finalaze %i\n", priv->long_timer);
     if (priv->long_timer){
        g_source_remove(priv->long_timer);
        priv->long_timer = 0;
     }
     if (priv->short_timer){
        g_source_remove(priv->short_timer);
        priv->short_timer = 0;
     }

     destroy_scene();
}

static void
animation_wallpaper_plugin_class_init (AWallpaperPluginClass *klass) {

    GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);
    GtkObjectClass *gobject_class = GTK_OBJECT_CLASS (klass);

    /* gobject */
    gobject_class->destroy = (gpointer)lw_applet_finalize;
    widget_class->realize = lw_applet_realize;
    widget_class->expose_event = lw_applet_expose_event;

    g_type_class_add_private (klass, sizeof (Animation_WallpaperPrivate));

}

static void
animation_wallpaper_plugin_class_finalize (AWallpaperPluginClass *class) {
}
