/*
 * Licensed under BSD license.  See LICENCE.TXT  
 *
 * Produced by:	Jeff Lait
 *
 *      	7DRL Development
 *
 * NAME:        mob.cpp ( Live Once Library, C++ )
 *
 * COMMENTS:
 */

#include "mob.h"
#include "stdio.h"
#include "map.h"
#include "msg.h"
#include "text.h"
#include "speed.h"
#include "item.h"
#include "event.h"
#include "display.h"

#include <iostream>
using namespace std;

//
// MOB Implementation
//

MOB::MOB()
{
    myFleeCount = 0;
    myBoredom = 0;
    myYellHystersis = 0;
    myAIState = 0;
    myHP = 0;
    myIsSwallowed = false;
    myNumDeaths = 0;
    myUID = INVALID_UID;
    myStrategy = STRATEGY_SAMEROOM;
    mySkipNextTurn = false;
    myDelayMob = false;
    myDelayMobIdx = -1;
    myCollisionTarget = 0;
    mySearchPower = 0;

    YELL_NAMES	yell;
    FOREACH_YELL(yell)
    {
	myHeardYell[yell] = false;
	myHeardYellSameRoom[yell] = false;
    }
    mySawMurder = false;
    mySawMeanMurder = false;
    mySawVictory = false;
    myAvatarHasRanged = false;
}

MOB::~MOB()
{
    int		i;

    if (myPos.map() && myPos.map()->avatar() == this)
	myPos.map()->setAvatar(0);

    myPos.removeMob(this);

    for (i = 0; i < myInventory.entries(); i++)
	delete myInventory(i);
}

MOB *
MOB::create(MOB_NAMES def)
{
    MOB		*mob;

    mob = new MOB();

    mob->myDefinition = def;

    mob->myHP = glb_mobdefs[def].max_hp;
    mob->myMP = glb_mobdefs[def].max_mp;

    if (def == MOB_AVATAR)
    {
	ITEM		*i;
	i = ITEM::create(ITEM_BIGSWORD, 5);
	mob->addItem(i);
	i = ITEM::create(ITEM_SEARCHRING, 5);
	mob->addItem(i);
	i = ITEM::create(ITEM_AMULETTRUESIGHT, 5);
	mob->addItem(i);
	i = ITEM::create(ITEM_GOLD, 5);
	i->setStackCount(5893);
	mob->addItem(i);
    }

    mob->myUID = glb_allocUID();

    return mob;
}

MOB *
MOB::copy() const
{
    MOB		*mob;

    mob = new MOB();
    
    *mob = *this;

    // Copy inventory one item at a time.
    // We are careful to maintain the same list order here so restoring
    // won't shuffle things unexpectadly
    int		 i;

    mob->myInventory.clear();
    for (i = 0; i < myInventory.entries(); i++)
    {
	mob->myInventory.append(myInventory(i)->copy());
    }
    
    return mob;
}

void
MOB::setPos(POS pos)
{
    myPos.removeMob(this);
    myPos = pos;
    myPos.addMob(this);
}


void
MOB::setMap(MAP *map)
{
    myPos.setMap(map);
    myTarget.setMap(map);
    myHome.setMap(map);
    myMeditatePos.setMap(map);
}

void
MOB::clearAllPos()
{
    myPos = POS();
    myTarget = POS();
    myHome = POS();
    myMeditatePos = POS();
}

MOB *
MOB::createNPC(int depth)
{
    int		i;
    MOB_NAMES	mob = MOB_NONE;
    int		choice = 0;
    MOB		*m;

    // Given the letter choice, choose a mob that matches it appropriately.
    choice = 0;
    for (i = 0; i < NUM_MOBS; i++)
    {
	// Stuff with 0 depth is never created manually.
	if (!glb_mobdefs[i].depth)
	    continue;

	// Use the baseletter to bias the creation.
	if (glb_mobdefs[i].depth <= depth)
	{
	    if (rand_choice(choice + glb_mobdefs[i].rarity) < glb_mobdefs[i].rarity)
		mob = (MOB_NAMES) i;
	    choice += glb_mobdefs[i].rarity;
	}
    }

    // Testing..
    // mob = MOB_PYTHON;

    m = MOB::create(mob);

    if (0)
    {
	ITEM *item = ITEM::createRandom(depth);
	if (!rand_choice(5))
	    item = ITEM::create(ITEM_BREAD);
	m->addItem(item);
    }

    return m;
}

void
MOB::getLook(u8 &symbol, ATTR_NAMES &attr) const
{
    symbol = glb_mobdefs[myDefinition].symbol;
    attr = (ATTR_NAMES) glb_mobdefs[myDefinition].attr;

    if (!alive())
    {
	// Dead creatures override their attribute to blood
	attr = ATTR_RED;
    }

    if (getDefinition() == MOB_KOBOLD)
    {
	// Change appearance according to equipment.
	ITEM		*item;

	item = lookupWand();
	if (item && item->getDefinition() == ITEM_BOW)
	{
	    // We are an archer!
	    attr = ATTR_KOBOLD_ARCHER;
	}
	else if (item)
	{
	    // Real wand
	    attr = ATTR_KOBOLD_MAGE;
	}
	else if (lookupWeapon())
	{
	    attr = ATTR_KOBOLD_GUARD;
	}
	else if (hasItem(ITEM_BREAD))
	{
	    attr = ATTR_KOBOLD_BAKER;
	}
    }
}

const char *
MOB::getName() const
{
    if (isAvatar())
	return "you";

    if (getDefinition() == MOB_KOBOLDBABY)
    {
	if (myStrategy == STRATEGY_SAMEROOM ||
	    myStrategy == STRATEGY_PARTY)
	{
	    return "baby kobold";
	}

	return "scared baby kobold";
    }
    
    if (getDefinition() == MOB_KOBOLD)
    {
	// Change appearance according to equipment.
	ITEM		*item;

	item = lookupWand();
	if (item && item->getDefinition() == ITEM_BOW)
	{
	    // We are an archer!
	    return "kobold archer";
	}
	else if (item)
	{
	    // Real wand
	    return "kobold mage";
	}
	else if (lookupWeapon())
	{
	    return "kobold guard";
	}
	else if (hasItem(ITEM_BREAD))
	{
	    return "kobold baker";
	}
    }
    return glb_mobdefs[myDefinition].name;
}

const char *
MOB::getRawName() const
{
    if (isAvatar())
	return glb_mobdefs[myDefinition].name;

    return getName();
}

ITEM *
MOB::getRandomItem() const
{
    ITEM	*result;
    int		choice = 0;

    result = 0;

    if (myInventory.entries())
    {
	choice = rand_choice(myInventory.entries());
	return myInventory(choice);
    }
    return result;
}

bool
MOB::hasItem(ITEM_NAMES itemname) const
{
    if (lookupItem(itemname))
	return true;

    return false;
}

ITEM *
MOB::lookupWeapon() const
{
    ITEM 	*bestweapon = 0;
    for (int i = 0; i < myInventory.entries(); i++)
    {
	if (myInventory(i)->isMelee())
	    bestweapon = aiLikeMoreWeapon(myInventory(i), bestweapon);
    }
    return bestweapon;
}

ITEM *
MOB::lookupWand() const
{
    ITEM 	*bestwand = 0;
    for (int i = 0; i < myInventory.entries(); i++)
    {
	if (myInventory(i)->isRanged())
	    bestwand = aiLikeMoreWand(myInventory(i), bestwand);
    }
    return bestwand;
}

ITEM *
MOB::lookupItem(ITEM_NAMES itemname) const
{
    int		i;

    for (i = 0; i < myInventory.entries(); i++)
    {
	if (myInventory(i)->getDefinition() == itemname)
	    return myInventory(i);
    }
    return 0;
}

bool
MOB::canMoveDir(int dx, int dy, bool checkmob) const
{
    POS		goal;

    goal = myPos.delta(dx, dy);

    return canMove(goal, checkmob);
}

bool
MOB::canMove(POS pos, bool checkmob) const
{
    if (!pos.valid())
	return false;

    if (!pos.isPassable())
    {
	if (pos.defn().isphaseable && defn().passwall)
	{
	    // Can move through it...
	}
	else if (pos.defn().isdiggable && defn().candig)
	{
	    // Could dig through it.
	}
	else
	{
	    // Failed...
	    return false;
	}
    }

    if (checkmob && pos.mob())
	return false;

    return true;
}

void
MOB::move(POS newpos, bool ignoreangle)
{
    // If we have swallowed something, move it too.
    if (!isSwallowed())
    {
	PTRLIST<MOB *>	moblist;
	int		i;

	pos().getAllMobs(moblist);
	for (i = 0; i < moblist.entries(); i++)
	{
	    if (moblist(i)->isSwallowed())
	    {
		moblist(i)->move(newpos, true);
	    }
	}
    }

    if (ignoreangle)
	newpos.setAngle(pos().angle());

    setPos(newpos);

    reportSquare(pos());
}

void
MOB::gainHP(int hp)
{
    int		maxhp;

    maxhp = defn().max_hp;

    myHP += hp;
    if (myHP > maxhp)
	myHP = maxhp;
}

void
MOB::gainMP(int mp)
{
    int		maxmp;

    maxmp = defn().max_mp;

    myMP += mp;
    if (myMP > maxmp)
	myMP = maxmp;
}

void
MOB::postEvent(EVENTTYPE_NAMES type, u8 sym, ATTR_NAMES attr, const char *text) const
{
    if (pos().map())
	pos().map()->getDisplay()->queue().append(EVENT((MOB *)this, sym, attr, type, text));
}


bool
MOB::applyDamage(MOB *src, int hits)
{
    // Being hit isn't boring.
    clearBoredom();

    // Tell the world we were hit!
    if (src && src->isAvatar() && pos().isFOV())
    {
	MOBLIST		allmobs;

	pos().map()->getAllMobs(allmobs);

	for (int i = 0; i < allmobs.entries(); i++)
	{
	    if (allmobs(i)->pos().isFOV())
	    {
		allmobs(i)->mySawMurder = true;
		if (getDefinition() == MOB_KOBOLDBABY)
		    allmobs(i)->mySawMeanMurder = true;
	    }
	}
    }

    if (hasItem(ITEM_INVULNERABLE))
	return false;

    if (hits >= getHP())
    {
	// Ensure we are dead to alive()
	myHP = 0;
	myNumDeaths++;
	// Rather mean.
	myMP = 0;

	// Death!
	if (src)
	    src->formatAndReport("%S <kill> %O!", this);
	else
	    formatAndReport("%S <be> killed!");

	// If there is a source, and they are swallowed,
	// they are released.
	if (src && src->isSwallowed())
	{
	    src->setSwallowed(false);
	}
	
	// If we are the avatar, special stuff happens.
	if (isAvatar())
	{
	    // never update on this thread!
	    // msg_update();
	    // TODO: Pause something here?
	}
	else
	{
	    // Drop any stuff we have.
	    int i;
	    for (i = 0; i < myInventory.entries(); i++)
	    {
		myInventory(i)->move(pos());
	    }
	    myInventory.clear();
	}

	if (src && src->isAvatar())
	{
	    // Award our score.

	}

	// Flash the screen
	pos().postEvent(EVENTTYPE_FOREBACK, ' ', ATTR_INVULNERABLE);

	// Note that avatar doesn't actually die...
	if (!isAvatar())
	{
	    // Don't actually delete, but die.
	    MAP		*map = pos().map();
	    ITEM	*corpse = ITEM::create(ITEM_CORPSE);

	    corpse->move(pos());

	    myPos.removeMob(this);
	    map->addDeadMob(this);
	    clearAllPos();
	    loseTempItems();
	}
	else
	{
	    // Make sure we drop our blind attribute..
	    loseTempItems();

	    // End any meditation
	    myMeditatePos = POS();

	    // No matter what, the source sees it (we may be meditating)
	    if (src)
		src->mySawVictory = true;

	    if (pos().isFOV())
	    {
		MOBLIST		allmobs;

		pos().map()->getAllMobs(allmobs);

		for (int i = 0; i < allmobs.entries(); i++)
		{
		    if (allmobs(i)->pos().isFOV())
			allmobs(i)->mySawVictory = true;
		}
	    }
	}
	return true;
    }

    // Flash that they are hit.
    if (hits)
	pos().postEvent(EVENTTYPE_FOREBACK, ' ', ATTR_HEALTH);

    myHP -= hits;
    return false;
}

VERB_PERSON
MOB::getPerson() const
{
    if (isAvatar())
	return VERB_YOU;

    return VERB_IT;
}

bool
MOB::isFriends(const MOB *other) const
{
    if (other == this)
	return true;
    
    if (isAvatar())
    {
	if (other->defn().isfriendly)
	    return true;
	else
	    return false;
    }

    // Only the avatar is evil!
    if (defn().isfriendly)
    {
	return true;
    }
    else
    {
	// Monsters hate the avtar!
	if (other->isAvatar())
	    return false;
    }
    return true;
}

void
MOB::viewDescription() const
{
    char	buf[100];
    int		min, q1, q2, q3, max;

    sprintf(buf, "Name: %s\n", getName());
    msg_report(buf);

    sprintf(buf, "Health: %d\n", getHP());
    msg_report(buf);

    getMeleeDPDF().getQuartile(min, q1, q2, q3, max);
    sprintf(buf, "Melee Weapon: %s (%d..%d..%d..%d..%d)\n",
		defn().melee_name,
		min, q1, q2, q3, max);
    msg_report(buf);

    if (defn().range_valid)
    {
	getRangedDPDF().getQuartile(min, q1, q2, q3, max);
	sprintf(buf, "Ranged Weapon: %s (%d..%d..%d..%d..%d), Range %d\n",
		    defn().range_name,
		    min, q1, q2, q3, max,
		    getRangedRange());
	msg_report(buf);
    }
    // Dump the text.txt if present.
    msg_report(text_lookup("mob", getRawName()));
}

AI_NAMES
MOB::getAI() const
{
    return (AI_NAMES) defn().ai;
}

void
MOB::reportSquare(POS t)
{
    if (!isAvatar())
	return;

    if (t.mob() && t.mob() != this)
    {
	formatAndReport("%S <see> %O.", t.mob());
    }
    if (t.item())
    {
	formatAndReport("%S <see> %O.", t.item());
    }

    if (t.defn().describe)
    {
	formatAndReport("%S <see> %O.", t.defn().legend);
    }
}

void
MOB::meditateMove(POS t)
{
    myMeditatePos = t;
    reportSquare(t);
}

bool
MOB::actionBump(int dx, int dy)
{
    MOB		*mob;
    POS		 t;

    // Stand in place.
    if (!dx && !dy)
	return true;

    if (isMeditating())
    {
	// We are free of our body!
	t = myMeditatePos.delta(dx, dy);
	if (t.defn().ispassable)
	{
	    meditateMove(t);
	    return true;
	}
	// Wall slide...
	if (dx && dy && isAvatar())
	{
	    // Try to wall slide, cause we are pretty real time here
	    // and it is frustrating to navigate curvy passages otherwise.
	    t = myMeditatePos.delta(dx, 0);
	    if (!rand_choice(2) && t.defn().ispassable)
	    { meditateMove(t); return true; }

	    t = myMeditatePos.delta(0, dy);
	    if (t.defn().ispassable)
	    { meditateMove(t); return true; }

	    t = myMeditatePos.delta(dx, 0);
	    if (t.defn().ispassable)
	    { meditateMove(t); return true; }
	}
	else if ((dx || dy) && isAvatar())
	{
	    // If we have
	    // ..
	    // @#
	    // ..
	    // Moving right we want to slide to a diagonal.
	    int		sdx, sdy;

	    // This bit of code is too clever for its own good!
	    sdx = !dx;
	    sdy = !dy;

	    t = myMeditatePos.delta(dx+sdx, dy+sdy);
	    if (!rand_choice(2) && t.defn().ispassable)
	    { meditateMove(t); return true; }

	    t = myMeditatePos.delta(dx-sdx, dy-sdy);
	    if (t.defn().ispassable)
	    { meditateMove(t); return true; }

	    t = myMeditatePos.delta(dx+sdx, dy+sdy);
	    if (t.defn().ispassable)
	    { meditateMove(t); return true; }
	}
	return false;
    }

    t = pos().delta(dx, dy);

    // If we are swallowed, we must attack.
    if (isSwallowed())
	return actionMelee(dx, dy);
    
    mob = t.mob();
    if (mob)
    {
	// Either kill or chat!
	if (mob->isFriends(this))
	{
	    if (isAvatar())
		return actionChat(dx, dy);
	    // Otherwise we just walk and bump.
	}
	else
	{
	    return actionMelee(dx, dy);
	}
    }

    // No mob, see if we can move that way.
    // We let actionWalk deal with unable to move notification.
    return actionWalk(dx, dy);
}

bool
MOB::actionMeditate()
{
    if (isMeditating())
    {
	formatAndReport("%S <stop> meditating.");
	myMeditatePos = POS();
	// we don't want the user to be able to use up a fast
	// turn by doing this!
	PHASE_NAMES		phase;

	phase = spd_getphase();
	if (phase == PHASE_FAST || phase == PHASE_QUICK)
	{
	    mySkipNextTurn = true;
	}
    }
    else
    {
	if (!hasItem(ITEM_AMULETTRUESIGHT))
	{
	    formatAndReport("%S <lack> an amulet to meditate with.");
	    return false;
	}
	formatAndReport("%S <start> meditating.");
	myMeditatePos = pos();
    }
    return true;
}

void
MOB::searchOffset(int dx, int dy)
{
    POS		square = pos().delta(dx, dy);

    if (square.isTrap())
    {
	TRAP_NAMES		trap = (TRAP_NAMES) rand_choice(NUM_TRAPS);

	formatAndReport("%S find %O and disarm it.", glb_trapdefs[trap].name);
	square.clearTrap();

	square.postEvent((EVENTTYPE_NAMES)(EVENTTYPE_ALL | EVENTTYPE_LONG), 
			glb_trapdefs[trap].sym,
			(ATTR_NAMES) glb_trapdefs[trap].attr);
    }
    else
    {
	square.postEvent(EVENTTYPE_FOREBACK, ' ', ATTR_SEARCH);
    }
}

bool
MOB::actionSearch()
{
    if (!hasItem(ITEM_SEARCHRING))
    {
	formatAndReport("%S <glance> around briefly, finding nothing.");
	return true;
    }
    formatAndReport("Applying %r %o, %S <search>.", lookupItem(ITEM_SEARCHRING));

    mySearchPower++;

    // Must be some limit as we are stupid and do n^2 for searching.
    if (mySearchPower > 10)
    {
	formatAndReport("%S <reach> the limit of %r ring's power.");
	mySearchPower = 0;
	return true;
    }

    int		ir, r;

    r = mySearchPower;
    // Note the < r to avoid double counting corners.
    for (ir = -r; ir < r; ir++)
    {
	searchOffset(ir, -r);
	searchOffset(r, ir);
	searchOffset(-ir, r);
	searchOffset(-r, -ir);
    }

    return true;
}

bool
MOB::actionDropButOne(ITEM *item)
{
    if (!item)
	return false;

    ITEM		*butone;

    butone = splitStack(item);
    assert(butone != item);

    if (butone == item)
    {
	addItem(butone);
	return false;
    }

    // Swap the two meanings
    butone->setStackCount(item->getStackCount());
    item->setStackCount(1);

    formatAndReport("%S <drop> %O.", butone);

    // And drop
    butone->move(pos());

    return true;
}

bool
MOB::actionDrop(ITEM *item)
{
    if (!item)
	return false;
    // Drop any stuff we have.
    int i;
    bool	fail = true;
    for (i = 0; i < myInventory.entries(); i++)
    {
	// Ignore special case items..
	// We don't want to drop "blindness" :>
	if (myInventory(i)->defn().isflag)
	    continue;

	if (myInventory(i) == item)
	{
	    formatAndReport("%S <drop> %O.", myInventory(i));
	    fail = false;
	    myInventory(i)->move(pos());
	    myInventory.set(i, 0);
	}
    }
    myInventory.collapse();

    if (fail)
	formatAndReport("%S <drop> nothing.");

    return true;
}

bool
MOB::actionQuaff(ITEM *item)
{
    if (!item)
	return false;
    if (!item->isPotion())
    {
	formatAndReport("%S cannot drink %O.", item);
	return false;
    }

    item = splitStack(item);

    postEvent( (EVENTTYPE_NAMES) (EVENTTYPE_SHOUT | EVENTTYPE_LONG),
		    ' ', ATTR_EMOTE,
		    "*quaff*");
    formatAndReport("%S <quaff> %O.", item);
    if (item->getDefinition() == ITEM_HEALPOTION)
    {
	formatAndReport("%S <look> healthier.");
	gainHP(15);
    }
    else if (item->getDefinition() == ITEM_QUICKBOOST)
    {
	formatAndReport("%S <move> faster.");
    }

    delete item;

    return true;
}

bool
MOB::actionEat(ITEM *item)
{
    if (!item)
	return false;

    if (!item->isFood())
    {
	formatAndReport("%S cannot eat %O.", item);
	return false;
    }

    item = splitStack(item);

    formatAndReport("%S <eat> %O.", item);

    delete item;

    return true;
}

bool
MOB::actionBreak(ITEM *item)
{
    if (!item)
	return false;

    if (!item->isRanged())
    {
	formatAndReport("%S cannot break %O.", item);
	return false;
    }

    item = splitStack(item);

    formatAndReport("%S <break> %O.", item);

    postEvent( (EVENTTYPE_NAMES) (EVENTTYPE_SHOUT | EVENTTYPE_LONG),
		    ' ', ATTR_EMOTE,
		    "*snap*");

    delete item;

    return true;
}

const char *
getYellMessage(YELL_NAMES yell)
{
    const char *yellmsg;

    switch (yell)
    {
	case YELL_KEEPOUT:
	{
	    const char *msg[] =
	    {		// KEEPOUT
		"Keep out!",
		"Away!",
		"Leave!",
		"No Farther!",
		0
	    };
	    yellmsg = rand_string(msg);
	    break;
	}
	case YELL_MURDERER:
	{
	    const char *msg[] =
	    {		// MURDERER
		"Killer!",
		"Murderer!",
		"Code Red!",	// For blood.
		0
	    };
	    yellmsg = rand_string(msg);
	    break;
	}
	case YELL_INVADER:
	{
	    const char *msg[] =
	    {		// INVADER
		"Invader!",
		"ALARM!",
		"ALARM!",
		"Code Yellow!",	// For yummy gold.
		0
	    };
	    yellmsg = rand_string(msg);
	    break;
	}
	case YELL_RANGED:
	{
	    const char *msg[] =
	    {		// RANGED
		"Far Threat!",
		"Code Bow!",	// Kobolds not that imaginative after all
		"Incoming!",
		"Archers!",	
		0
	    };
	    yellmsg = rand_string(msg);
	    break;
	}
	case YELL_KILL:
	{
	    const char *msg[] =
	    {		// KILL
		"Kill it now!",
		"Code Gold!",
		"Kill!",
		"Eviscerate It!",
		"Kill!",
		"Kill!",
		0
	    };
	    yellmsg = rand_string(msg);
	    break;
	}
	case YELL_KILLCHASE:
	{
	    const char *msg[] =
	    {		// KILL
		"No Mercy!",
		"Code Blood!",
		"Hunt It Down!",
		0
	    };
	    yellmsg = rand_string(msg);
	    break;
	}
	case YELL_LOCATION:
	{
	    const char *msg[] =
	    {		// LOCATION
		"It's here!",
		"Foe Sighted!",
		"Over here!",
		"I see it!",
		0
	    };
	    yellmsg = rand_string(msg);
	    break;
	}
	case YELL_HEARDLOCATION:
	{
	    const char *msg[] =
	    {		// LOCATION_HEARD 
		"It's there!",
		"Foe Located!",
		"Over there!",
		0
	    };
	    yellmsg = rand_string(msg);
	    break;
	}
	case YELL_TAUNT:
	{
	    const char *msg[] =
	    {		// LOCATION_HEARD 
		"Coward!",
		"You Flea!",
		"Mendicant!",
		"Wimp!",
		"Fool!",
		"Urchin!",
		0
	    };
	    yellmsg = rand_string(msg);
	    break;
	}
	case YELL_VICTORY:
	{
	    const char *msg[] =
	    {		// LOCATION_HEARD 
		"Victory!",
		"The @ is dead!",
		"Huzzah!",
		"Hooray!",
		"Gold!",
		0
	    };
	    yellmsg = rand_string(msg);
	    break;
	}
    }
    return yellmsg;
}

bool
MOB::actionYell(YELL_NAMES yell)
{
    const char	*yellflavour[] =
    {
	"%S <yell>: \"%o\"",
	"%S <shout>: \"%o\"",
	"%S <chirp>: \"%o\"",
	"%S <squeak>: \"%o\"",
	"%S <curse>: \"%o\"",
	0
    };
    const char *yellmsg = getYellMessage(yell);

    if (getDefinition() == MOB_KOBOLDBABY)
    {
	const char *babymsg[] =
	{
	    "Yip!",
	    "Bark!",
	    "Arf!",
	    "Woof!",
	    0
	};
	yellmsg = rand_string(babymsg);
    }

    MOBLIST	hearmobs;

    pos().getAllConnectedRoomMobs(hearmobs);

    for (int i = 0; i < hearmobs.entries(); i++)
    {
	if (hearmobs(i)->isAvatar())
	{
	    // Don't use line of sight, but the same logic we use.
	    msg_format(rand_string(yellflavour), this, yellmsg);
	    postEvent( (EVENTTYPE_NAMES) (EVENTTYPE_SHOUT | EVENTTYPE_LONG),
			    ' ', ATTR_SHOUT,
			    yellmsg);
	}
	if (hearmobs(i) == this)
	    continue;
	hearmobs(i)->myHeardYell[yell] = true;
	if (hearmobs(i)->pos().roomId() == pos().roomId())
	{
	    // Extra flag to avoid spurious re-shouts
	    hearmobs(i)->myHeardYellSameRoom[yell] = true;
	}
    }

    return true;
}

bool
MOB::actionRotate(int angle)
{
    myPos = myPos.rotate(angle);
    if (isMeditating())
	myMeditatePos = myMeditatePos.rotate(angle);
    // This is always a free action.
    return false;
}

bool
MOB::actionClimb()
{
    TILE_NAMES		tile;

    tile = pos().tile();

    if (tile == TILE_DOWNSTAIRS)
    {
	// Climbing is interesting.
	myFleeCount = 0;
	clearBoredom();
 
	msg_format("%S <climb> down... to nowhere!", this);
	return true;
    }
    else if (tile == TILE_UPSTAIRS)
    {
	msg_format("%S <climb> up... to nowhere!", this);
	return true;
    }
    else if (tile == TILE_FOREST)
    {
	msg_format("%S <climb> a tree... and <fall> down!", this);
	return true;
    }

    msg_format("%S <see> nothing to climb here.", this);

    return false;
}

bool
MOB::actionChat(int dx, int dy)
{
    MOB		*victim;

    // This should never occur, but to avoid
    // embarassaments...
    if (!isAvatar())
	return false;
    
    victim = pos().delta(dx, dy).mob();
    if (!victim)
    {
	// Talk to self
	formatAndReport("%S <talk> to %O!", this);
	return false;
    }

    formatAndReport("%S <chat> with %O:\n", victim);

    msg_quote(text_lookup("chat", getRawName(), victim->getRawName()));
    
    return true;
}

bool
MOB::actionMelee(int dx, int dy)
{
    MOB		*victim;

    // If we are swallowed, attack the person on our square.
    if (isSwallowed())
	dx = dy = 0;

    POS		 t = pos().delta(dx, dy);

    victim = t.mob();
    if (!victim)
    {
	// Swing in air!
	msg_format("%S <swing> at empty air!", this);
	return false;
    }

    if (!victim->alive())
    {
	// The end game avatar, just silently ignore
	assert(!isAvatar());
	return false;
    }

    // Shooting is interesting.
    clearBoredom();
 
    int		damage;
    bool	victimdead;

    damage = getMeleeDamage();

    // attempt to kill victim
    if (damage || (defn().melee_item != ITEM_NONE))
    {
	msg_format("%S %v %O.", this, getMeleeVerb(),
				victim);
    }
    else
    {
	msg_format("%S <miss> %O.", this, victim);
    }

    victimdead = victim->applyDamage(this, damage);
    
    // Vampires regenerate!
    if (defn().isvampire)
    {
	myHP += damage;
    }

    // Grant our item...
    if (!victimdead)
    {
	ITEM_NAMES	item;

	item = (ITEM_NAMES) defn().melee_item;
	if (item != ITEM_NONE)
	{
	    ITEM	*i;

	    // Exclusive items you should only get one of so they
	    // don't stack.
	    if (!glb_itemdefs[item].exclusive ||
		!victim->hasItem(item))
	    {
		i = ITEM::create(item);
		if (i)
		    victim->addItem(i);
	    }
	}

	// Swallow the victim
	// Seems silly to do this on a miss.
	if (defn().swallows && damage)
	{
	    victim->setSwallowed(true);
	    victim->move(pos(), true);
	}

	// Steal something
	if (defn().isthief)
	{
	    ITEM		*item;
	    
	    // Get a random item from the victim
	    item = victim->getRandomItem();
	    if (item) 
	    {
		msg_format("%S <steal> %O.", this, item);
		// Theft successful.
		myFleeCount += 30;
		victim->removeItem(item, true);
		addItem(item);
	    }
	}
    }
	
    return true;
}

const char *
MOB::getMeleeVerb() const
{
    ITEM	*w;

    w = lookupWeapon();
    if (!w)
	return defn().melee_verb;

    return w->defn().melee_verb;
}

int
MOB::getMeleeDamage() const
{
    DPDF	damage;

    damage = getMeleeDPDF();

    return damage.evaluate();
}

DPDF
MOB::getMeleeDPDF() const
{
    DPDF	dpdf(0);

    ITEM	*weapon;

    weapon = lookupWeapon();
    if (weapon)
    {
	return weapon->getMeleeDPDF();
    }

    dpdf = defn().melee_damage.buildDPDF();

    dpdf *= (defn().melee_chance) / 100.0;

    return dpdf;
}

int
MOB::getRangedDamage() const
{
    DPDF	damage;

    damage = getRangedDPDF();

    return damage.evaluate();
}

DPDF
MOB::getRangedDPDF() const
{
    ITEM	*weapon;

    weapon = lookupWand();
    if (weapon)
    {
	return weapon->getRangeDPDF();
    }

    DPDF	dpdf(0);

    dpdf = defn().range_damage.buildDPDF();

    return dpdf;
}

int
MOB::getRangedRange() const
{
    int		range;

    ITEM	*weapon;

    weapon = lookupWand();
    if (weapon)
    {
	return weapon->getRangeRange();
    }

    range = defn().range_range;

    return range;
}

void
MOB::getRangedLook(u8 &symbol, ATTR_NAMES &attr) const
{
    symbol = defn().range_symbol;
    attr = (ATTR_NAMES) defn().range_attr;

    // Check out item, if any.
    ITEM		*i;

    i = lookupWand();

    if (i)
    {
	symbol = i->defn().range_symbol;
	attr = (ATTR_NAMES)i->defn().range_attr;
    }
}

bool
MOB::actionFire(int dx, int dy)
{
    MOB		*victim;
    int		rangeleft;
    
    // Check for no ranged weapon.
    if (!defn().range_valid && !lookupWand())
    {
	msg_format("%S <lack> a ranged attack!", this);
	return false;
    }

    // No suicide.
    if (!dx && !dy)
    {
	msg_format("%S <decide> not to aim at %O.", this, this);
	return false;
    }
    
    // If swallowed, rather useless.
    if (isSwallowed())
    {
	msg_format("%S <do> not have enough room inside here.", this);
	return false;
    }

    // Check for friendly kill.
    victim = pos().traceBullet(getRangedRange(), dx, dy, &rangeleft);

    if (victim && victim->isFriends(this))
    {
	// Not a clear shot!
	if (isAvatar())
	{
	    msg_report(text_lookup("fire", getRawName()));
	    // We have special messages.
	    return false;
	}

	// Avoid friendly fire
	return false;
    }

    // Shooting is interesting.
    clearBoredom();
 
    u8		symbol;
    ATTR_NAMES	attr;
    int		area = 1;
    
    getRangedLook(symbol, attr);

    pos().displayBullet(getRangedRange(),
			dx, dy,
			symbol, attr,
			true);
	//		!isAvatar() || area != 1);


    if (!victim)
    {
	// Shoot at air!
	formatAndReport("%S <fire> at empty air!");
	// But, really, the fireball should explode!
	POS		vpos;
	vpos = pos().traceBulletPos(getRangedRange(), dx, dy, true);

	// Apply damage to everyone in range.
	vpos.fireball(this, area, getRangedDPDF(), symbol, attr); 

	return true;
    }

    // Attemp to kill victim
    msg_format("%S %v %O.", this, defn().range_verb,
			    victim);

    // NOTE: Bad idea to call a function on victim->pos() as
    // that likely will be deleted :>
    POS		vpos = victim->pos();

    // Apply damage to everyone in range.
    vpos.fireball(this, area, getRangedDPDF(), symbol, attr); 
    
    // The following code will keep the flame going past the target
#if 0
    while (isAvatar() && rangeleft && area == 1)
    {
	// Ensure our dx/dy is copacetic.
	vpos.setAngle(pos().angle());
	victim = vpos.traceBullet(rangeleft, dx, dy, &rangeleft);

	if (victim)
	{
	    msg_format("%S %v %O.", this, defn().range_verb,
				    victim);
	    vpos = victim->pos();
	    vpos.fireball(this, area, getRangedDPDF(), symbol, attr); 
	}
	else
	    rangeleft = 0;
    }
#endif
    return true;
}

bool
MOB::actionWalk(int dx, int dy)
{
    MOB		*victim;
    POS		 t;

    // If we are swallowed, we can't move.
    if (isSwallowed())
	return false;
    
    t = pos().delta(dx, dy);
    
    victim = t.mob();
    if (victim)
    {
	// If we are the avatar, screw being nice!
	if (isAvatar())
	    return actionMelee(dx, dy);
	// If it is thet avatar we are bumping into, screw being
	// nice.
	if (victim->isAvatar())
	    return actionMelee(dx, dy);

	// formatAndReport("%S <bump> into %O.", victim);
	// Bump into the mob.

	// Nah, tell the map we want a delayed move.
	return pos().map()->requestDelayedMove(this, dx, dy);
    }

    // Check to see if we can move that way.
    if (canMove(t))
    {
	TILE_NAMES	tile;
	
	// Determine if it is a special move..
	tile = t.tile();
	if (t.defn().isdiggable &&
	    !t.defn().ispassable &&	
	    defn().candig)
	{
	    return t.digSquare();
	}

	POS		opos = pos();
	
	// Move!
	move(t);

	// If we are a breeder, chance to reproduce!
	if (defn().breeder)
	{
	    static  int		lastbreedtime = -10;

	    // Only breed every 7 phases at most so one can
	    // always kill off the breeders.
	    if ((lastbreedtime < spd_gettime() - 7) &&
		!rand_choice(50 + pos().map()->getNumMobs()))
	    {
		MOB		*child;
		
		formatAndReport("%S <reproduce>!");
		child = MOB::create(getDefinition());
		child->move(opos);
		lastbreedtime = spd_gettime();
	    }
	}
	
	// If we are the avatar and something is here, pick it up.
	if (isAvatar() && t.item())
	    actionPickup();

	// Avatars set off traps because they are clumsy
	if (isAvatar() && t.isTrap())
	{
	    t.clearTrap();
	    TRAP_NAMES		trap = (TRAP_NAMES) rand_choice(NUM_TRAPS);

	    t.postEvent((EVENTTYPE_NAMES)(EVENTTYPE_ALL | EVENTTYPE_LONG), 
			    glb_trapdefs[trap].sym,
			    (ATTR_NAMES) glb_trapdefs[trap].attr);
	    DPDF	dpdf(0);

	    formatAndReport("%S <set> off %O!", glb_trapdefs[trap].name);

	    dpdf = glb_trapdefs[trap].damage.buildDPDF();

	    applyDamage(0, dpdf.evaluate());
	}
	
	return true;
    }
    else
    {
	if (dx && dy && isAvatar())
	{
	    // Try to wall slide, cause we are pretty real time here
	    // and it is frustrating to navigate curvy passages otherwise.
	    if (!rand_choice(2) && canMoveDir(dx, 0, false))
		if (actionWalk(dx, 0))
		    return true;
	    if (canMoveDir(0, dy, false))
		if (actionWalk(0, dy))
		    return true;
	    if (canMoveDir(dx, 0, false))
		if (actionWalk(dx, 0))
		    return true;
	}
	else if ((dx || dy) && isAvatar())
	{
	    // If we have
	    // ..
	    // @#
	    // ..
	    // Moving right we want to slide to a diagonal.
	    int		sdx, sdy;

	    // This bit of code is too clever for its own good!
	    sdx = !dx;
	    sdy = !dy;

	    if (!rand_choice(2) && canMoveDir(dx+sdx, dy+sdy, false))
		if (actionWalk(dx+sdx, dy+sdy))
		    return true;
	    if (canMoveDir(dx-sdx, dy-sdy, false))
		if (actionWalk(dx-sdx, dy-sdy))
		    return true;
	    if (canMoveDir(dx+sdx, dy+sdy, false))
		if (actionWalk(dx+sdx, dy+sdy))
		    return true;
	    
	}

	formatAndReport("%S <be> blocked by %O.", t.defn().legend);
	// Bump into a wall.
	return false;
    }
}

bool
MOB::actionPickup()
{
    ITEM		*item;

    item = pos().item();

    if (item->getDefinition() != ITEM_CORPSE)
	return actionPickup(item);
    return false;
}

bool
MOB::actionPickup(ITEM *item)
{
    if (!item)
    {
	formatAndReport("%S <grope> the ground foolishly.");
	return false;
    }

    if (item->pos() != pos())
    {
	formatAndReport("%S <be> too far away to pick up %O.", item);
	return false;
    }

    // Pick up the item!
    formatAndReport("%S <pick> up %O.", item);

    // No longer coveted!
    item->setInterestedUID(INVALID_UID);
    item->move(POS());

    addItem(item);
    return true;
}

void
MOB::addItem(ITEM *item)
{
    int			 i;
    
    // Alert the world to our acquirement.
    if (isAvatar())
	if (item->defn().gaintxt)
	    msg_format(item->defn().gaintxt,
			this, item);

    // First, check to see if we can merge...
    for (i = 0; i < myInventory.entries(); i++)
    {
	if (item->canStackWith(myInventory(i)))
	{
	    myInventory(i)->combineItem(item);
	    delete item;
	    return;
	}
    }

    // Brand new item.
    myInventory.append(item);
}

ITEM *
MOB::splitStack(ITEM *item)
{
    if (item->getStackCount() > 1)
    {
	ITEM	*result;

	item->decStackCount();
	result = item->createCopy();
	result->setStackCount(1);

	return result;
    }
    else
    {
	removeItem(item);
	return item;
    }
}

void
MOB::removeItem(ITEM *item, bool quiet)
{
    // Alert the world to our acquirement.
    if (isAvatar() && !quiet)
	if (item->defn().losetxt)
	    msg_format(item->defn().losetxt,
			this, item);

    myInventory.removePtr(item);
}

void
MOB::loseAllItems()
{
    int		i;
    ITEM	*item;

    for (i = myInventory.entries(); i --> 0;)
    {
	item = myInventory(i);
	removeItem(item, true);
	delete item;
    }
}

void
MOB::loseTempItems()
{
    int		i;
    ITEM	*item;

    for (i = myInventory.entries(); i --> 0;)
    {
	item = myInventory(i);
	if (item->getTimer() >= 0)
	{
	    // All timed items are temporary.
	    removeItem(item, true);
	    delete item;
	}
    }
}

void
MOB::save(ostream &os) const
{
    int		val;
    u8		c;

    val = myDefinition;
    os.write((const char *) &val, sizeof(int));

    myPos.save(os);
    myMeditatePos.save(os);
    myTarget.save(os);
    myHome.save(os);

    os.write((const char *) &myHP, sizeof(int));
    os.write((const char *) &myMP, sizeof(int));
    os.write((const char *) &myAIState, sizeof(int));
    os.write((const char *) &myFleeCount, sizeof(int));
    os.write((const char *) &myBoredom, sizeof(int));
    os.write((const char *) &myYellHystersis, sizeof(int));
    os.write((const char *) &myNumDeaths, sizeof(int));
    os.write((const char *) &myUID, sizeof(int));
    val = myStrategy;
    os.write((const char *) &val, sizeof(int));
    os.write((const char *) &mySearchPower, sizeof(int));

    val = isSwallowed();
    os.write((const char *) &val, sizeof(int));

    int			 numitem;
    int			 i;

    c = mySawMurder;
    os.write((const char *) &c, 1);
    c = mySawMeanMurder;
    os.write((const char *) &c, 1);
    c = mySawVictory;
    os.write((const char *) &c, 1);
    c = myAvatarHasRanged;
    os.write((const char *) &c, 1);

    YELL_NAMES		yell;
    FOREACH_YELL(yell)
    {
	c = myHeardYell[yell];
	os.write((const char *) &c, 1);
	c = myHeardYellSameRoom[yell];
	os.write((const char *) &c, 1);
    }

    numitem = myInventory.entries();
    os.write((const char *) &numitem, sizeof(int));

    for (i = 0; i < myInventory.entries(); i++)
    {
	myInventory(i)->save(os);
    }
}

MOB *
MOB::load(istream &is)
{
    int		 val, num, i;
    u8		 c;
    MOB		*mob;

    mob = new MOB();

    is.read((char *)&val, sizeof(int));
    mob->myDefinition = (MOB_NAMES) val;

    mob->myPos.load(is);
    mob->myMeditatePos.load(is);
    mob->myTarget.load(is);
    mob->myHome.load(is);

    is.read((char *)&mob->myHP, sizeof(int));
    is.read((char *)&mob->myMP, sizeof(int));
    is.read((char *)&mob->myAIState, sizeof(int));
    is.read((char *)&mob->myFleeCount, sizeof(int));
    is.read((char *)&mob->myBoredom, sizeof(int));
    is.read((char *)&mob->myYellHystersis, sizeof(int));
    is.read((char *)&mob->myNumDeaths, sizeof(int));
    is.read((char *)&mob->myUID, sizeof(int));
    is.read((char *)&val, sizeof(int));
    mob->myStrategy = (STRATEGY_STATE) val;
    is.read((char *)&mob->mySearchPower, sizeof(int));
    glb_reportUID(mob->myUID);

    is.read((char *)&val, sizeof(int));
    mob->setSwallowed(val ? true : false);

    is.read((char *)&c, 1);
    mob->mySawMurder = (c ? true : false);
    is.read((char *)&c, 1);
    mob->mySawMeanMurder = (c ? true : false);
    is.read((char *)&c, 1);
    mob->mySawVictory = (c ? true : false);
    is.read((char *)&c, 1);
    mob->myAvatarHasRanged = (c ? true : false);
    
    YELL_NAMES		yell;
    FOREACH_YELL(yell)
    {
	is.read((char *)&c, 1);
	mob->myHeardYell[yell] = (c ? true : false);
	is.read((char *)&c, 1);
	mob->myHeardYellSameRoom[yell] = (c ? true : false);
    }

    is.read((char *)&num, sizeof(int));

    for (i = 0; i < num; i++)
    {
	mob->myInventory.append(ITEM::load(is));
    }

    return mob;
}

bool
MOB::hasVisibleEnemies() const
{
    PTRLIST<MOB *> list;

    getVisibleEnemies(list);
    if (list.entries())
	return true;
    return false;
}

bool
MOB::actionDropSurplus()
{
    ITEM		*weap, *wand;

    weap = lookupWeapon();
    wand = lookupWand();
    for (int i = 0; i < myInventory.entries(); i++)
    {
	if (myInventory(i) == weap)
	    continue;
	if (myInventory(i) == wand)
	    continue;
	if (myInventory(i)->isPotion())
	    continue;

	if (myInventory(i)->defn().isflag)
	    continue;

	return actionDrop(myInventory(i));
    }
    // Check if our wand/weap is redundant
    if (weap && weap->getStackCount() > 1)
    {
	return actionDropButOne(weap);
    }
    if (wand && wand->getStackCount() > 1)
    {
	return actionDropButOne(wand);
    }
    return false;
}

bool
MOB::hasSurplusItems() const
{
    bool	hasrange = false;
    bool	hasmelee = false;

    for (int i = 0; i < myInventory.entries(); i++)
    {
	// We can always drink these.
	if (myInventory(i)->isPotion())
	    continue;
	if (myInventory(i)->defn().isflag)
	    continue;
	if (myInventory(i)->isRanged())
	{
	    if (myInventory(i)->getStackCount() > 1)
		return true;
	    if (hasrange)
		return true;
	    hasrange = true;
	    continue;
	}
	if (myInventory(i)->isMelee())
	{
	    if (myInventory(i)->getStackCount() > 1)
		return true;
	    if (hasmelee)
		return true;
	    hasmelee = true;
	    continue;
	}
	return true;
    }

    return false;
}

int
MOB::numSurplusRange() const
{
    int		numrange = 0;

    for (int i = 0; i < myInventory.entries(); i++)
    {
	if (myInventory(i)->isRanged())
	{
	    numrange += myInventory(i)->getStackCount();
	}
    }

    // Reserve one range for our own use.
    return numrange - 1;
}

void
MOB::getVisibleEnemies(PTRLIST<MOB *> &list) const
{
    // If we are blind, we can't see anyone.
    if (hasItem(ITEM_BLIND))
	return;

    // We only care about the non-avatar case now.
    if (isAvatar())
	return;

    // Check if we are in FOV.  If so, the avatar is visible.
    if (getAvatar())
    {
	// Ignore dead avatars.
	if (!getAvatar()->alive())
	{
	    return;
	}
	// Ignore none hostile
	if (isFriends(getAvatar()))
	    return;

	// We need the double check because if you are meditating
	// and put them in the fov, that doesn't mean that 
	// the avatar is now in fov.
	if (pos().isFOV() && getAvatar()->pos().isFOV())
	{
	    list.append(getAvatar());
	}
    }
}

void
MOB::formatAndReport(const char *msg)
{
    if (isAvatar() || pos().isFOV())
    {
	msg_format(msg, this);
    }
}

void
MOB::formatAndReport(const char *msg, MOB *object)
{
    if (isAvatar() || pos().isFOV())
    {
	msg_format(msg, this, object);
    }
}

void
MOB::formatAndReport(const char *msg, ITEM *object)
{
    if (isAvatar() || pos().isFOV())
    {
	msg_format(msg, this, object);
    }
}

void
MOB::formatAndReport(const char *msg, const char *object)
{
    if (isAvatar() || pos().isFOV())
    {
	msg_format(msg, this, object);
    }
}

int
MOB::numberMeleeHostiles() const
{
    int		dx, dy;
    MOB		*mob;
    int		hostiles = 0;

    FORALL_8DIR(dx, dy)
    {
	mob = pos().delta(dx, dy).mob();

	if (mob && !mob->isFriends(this))
	    hostiles++;
    }
    return hostiles;
}

void
MOB::clearCollision()
{
    myCollisionSources.clear();
    myCollisionTarget = 0;
}

void
MOB::setCollisionTarget(MOB *target)
{
    if (myCollisionTarget == target)
	return;

    assert(!target || !myCollisionTarget);

    myCollisionTarget = target;
    if (target)
	target->collisionSources().append(this);
}
