/*
 * Licensed under BSD license.  See LICENCE.TXT  
 *
 * Produced by:	Jeff Lait
 *
 *      	7DRL Development
 *
 * NAME:        item.cpp ( Live Once Library, C++ )
 *
 * COMMENTS:
 */

#include "item.h"

#include <assert.h>

#include <iostream>
using namespace std;

int glbUniqueId = 0;

int 
glb_allocUID()
{
    return glbUniqueId++;
}

void
glb_reportUID(int uid)
{
    if (uid >= glbUniqueId)
	glbUniqueId = uid+1;
}


//
// Item Definitions
//

ITEM::ITEM()
{
    myDefinition = (ITEM_NAMES) 0;
    myCount = 1;
    myTimer = -1;
    myUID = INVALID_UID;
    myInterestedMobUID = INVALID_UID;
}

ITEM::~ITEM()
{
    pos().removeItem(this);
}

void
ITEM::move(POS pos)
{
    myPos.removeItem(this);
    myPos = pos;
    myPos.addItem(this);
}

ITEM *
ITEM::copy() const
{
    ITEM *item;

    item = new ITEM();

    *item = *this;

    return item;
}

ITEM *
ITEM::createCopy() const
{
    ITEM *item;

    item = copy();

    item->myUID = glb_allocUID();

    return item;
}

ITEM *
ITEM::create(ITEM_NAMES item, int depth)
{
    ITEM	*i;

    assert(item >= ITEM_NONE && item < NUM_ITEMS);

    i = new ITEM();

    i->myDefinition = item;
    i->myTimer = glb_itemdefs[item].timer;

    i->myUID = glb_allocUID();

    return i;
}

ITEM *
ITEM::createRandom(int depth)
{
    ITEM_NAMES		item;

    item = itemFromHash(rand_choice(65536*32767));
    return create(item, depth);
}

ITEM_NAMES
ITEM::itemFromHash(unsigned int hash)
{
    int		totalrarity, rarity;
    int		i;

    // Find total rarity of items
    totalrarity = 0;
    for (i = ITEM_NONE; i < NUM_ITEMS; i++)
    {
	rarity = defn((ITEM_NAMES)i).rarity;
	totalrarity += rarity;
    }

    hash %= totalrarity;
    for (i = ITEM_NONE; i < NUM_ITEMS; i++)
    {
	rarity = defn((ITEM_NAMES)i).rarity;

	if (hash > (unsigned) rarity)
	    hash -= rarity;
	else
	    break;
    }
    return (ITEM_NAMES) i;
}

VERB_PERSON
ITEM::getPerson() const
{
    return VERB_IT;
}

BUF
ITEM::getName() const
{
    BUF		buf, basename;

    basename = getRawName();
    if (!buf.isstring())
    {
	if (getTimer() >= 0)
	{
	    // A timed item...
	    buf.sprintf("%s (%d)", basename.buffer(), getTimer());
	}
	else
	{
	    // A normal item.
	    buf.reference(basename.buffer());
	}
    }
    return gram_createcount(buf, myCount, false);
}

BUF
ITEM::getRawName() const
{
    BUF		buf;

    buf.reference(defn().name);
    return buf;
}

BUF
ITEM::getDetailedDescription() const
{
    BUF		result, buf;

    result.reference("");

    if (isMelee())
    {
	// Build melee description.
	int		p, a, c;
	int		min, q1, q2, q3, max;
	DPDF	damage;

	damage = getMeleeDPDF();
	// Only count hits.
	damage.applyGivenGreaterThan(0);
	damage.getQuartile(min, q1, q2, q3, max);

	getWeaponStats(p, a, c);
	buf.sprintf("Accuracy: %d\nDamage: \n%d..%d..%d..%d..%d\n",
			a, min, q1, q2, q3, max);
	result.strcat(buf);
    }
    if (isRanged())
    {
	// Build ranged description.
	int		p, c, r, a;
	int		min, q1, q2, q3, max;
	DPDF	damage;

	damage = getRangeDPDF();
	// Spells always hit
	damage.getQuartile(min, q1, q2, q3, max);

	getRangeStats(r, p, c, a);
	buf.sprintf(
		    "Area: %d Range: %d\n"
		    "Damage:\n%d..%d..%d..%d..%d\n",
			a, r,
			min, q1, q2, q3, max);
	result.strcat(buf);
    }
    return result;
}

void
ITEM::getLook(u8 &symbol, ATTR_NAMES &attr) const
{
    symbol = defn().symbol;
    attr = (ATTR_NAMES) defn().attr;
}

bool
ITEM::canStackWith(const ITEM *stack) const
{
    if (getDefinition() != stack->getDefinition())
	return false;

    // Disable stacking of weapons/books/armour as we want
    // to be able to delete them easily
    if (defn().unstackable)
	return false;

    // No reason why not...
    return true;
}

void
ITEM::combineItem(const ITEM *item)
{
    // Untimed items stack.
    if (getTimer() < 0)
	myCount += item->getStackCount();
    // Timed items add up charges.
    if (getTimer() >= 0)
    {
	assert(item->getTimer() >= 0);
	myTimer += item->getTimer();
    }

    assert(myCount >= 0);
    if (myCount < 0)
	myCount = 0;
}

bool
ITEM::runHeartbeat()
{
    if (myTimer >= 0)
    {
	if (!myTimer)
	    return true;
	myTimer--;
    }
    return false;
}

void
ITEM::save(ostream &os) const
{
    int		val;

    val = myDefinition;
    os.write((const char *) &val, sizeof(int));

    myPos.save(os);

    os.write((const char *) &myCount, sizeof(int));
    os.write((const char *) &myTimer, sizeof(int));
    os.write((const char *) &myUID, sizeof(int));
    os.write((const char *) &myInterestedMobUID, sizeof(int));
}

ITEM *
ITEM::load(istream &is)
{
    int		val;
    ITEM	*i;

    i = new ITEM();

    is.read((char *)&val, sizeof(int));
    i->myDefinition = (ITEM_NAMES) val;

    i->myPos.load(is);

    is.read((char *)&i->myCount, sizeof(int));
    is.read((char *)&i->myTimer, sizeof(int));

    is.read((char *)&i->myUID, sizeof(int));
    glb_reportUID(i->myUID);
    is.read((char *)&i->myInterestedMobUID, sizeof(int));

    return i;
}

bool
ITEM::isPotion() const
{
    return defn().ispotion;
}

bool
ITEM::isFood() const
{
    return defn().isfood;
}

bool
ITEM::isMelee() const
{
    if (defn().melee_power || defn().melee_consistency)
	return true;
    return false;
}

bool
ITEM::isRanged() const
{
    if (defn().range_power || defn().range_consistency)
	return true;
    return false;
}

void
ITEM::getWeaponStats(int &power, int &accuracy, int &consistency) const
{
    power = defn().melee_power;
    consistency = defn().melee_consistency;
    accuracy = defn().melee_accuracy;
}

DPDF
ITEM::getMeleeDPDF() const
{
    // Compute our power, accuracy, and consistency.
    int		power, accuracy, consistency;

    getWeaponStats(power, accuracy, consistency);

    // Power produces a pure DPDF
    DPDF		damage(0);

    damage += DPDF(0, power*2);

    // Add in the consistency bonus.  This is also comparable to power,
    // but is a constant multiplier rather than a DPDF
    damage += consistency;

    // Now, scale by the accuracy.
    // We make this a straight 0..100.
    double	tohit;

    tohit = (accuracy) / 100.0;
    if (tohit > 1.0)
    {
	DPDF		critical;

	critical = damage;
	critical *= (tohit - 1.0);
	damage += critical;
    }
    else
	damage *= tohit;

    return damage;
}

void
ITEM::getRangeStats(int &range, int &power, int &consistency, int &area) const
{
    range = defn().range_range;
    consistency = defn().range_consistency;
    power = defn().range_power;
    area = defn().range_area;
}

int
ITEM::getRangeRange() const
{
    // Compute our power, accuracy, and consistency.
    int		range, d;

    getRangeStats(range, d, d, d);

    return range;
}

int
ITEM::getRangeArea() const
{
    // Compute our power, accuracy, and consistency.
    int		area, d;

    getRangeStats(d, d, d, area);

    return area;
}
DPDF
ITEM::getRangeDPDF() const
{
    // Compute our power, accuracy, and consistency.
    int		power, consistency, i;

    getRangeStats(i, power, consistency, i);

    // Power produces a pure DPDF
    DPDF		damage(0);

    damage += DPDF(0, power*2);

    // Add in the consistency bonus.  This is also comparable to power,
    // but is the max of three uniform variates.
    DPDF		con(0);
    for (i = 0; i < 3; i++)
    {
	con.max(con, DPDF(0, consistency));
    }

    // Add in consistency bonus.
    damage += con;

    return damage;
}
