/*
 * Licensed under BSD license.  See LICENCE.TXT  
 *
 * Produced by:	Jeff Lait
 *
 *      	Smart Kobold Development
 *
 * NAME:        firefly.cpp ( Jacob's Matrix, C++ )
 *
 * COMMENTS:
 *		Runs a fire simulation on another thread.
 *		Allows you to get pure info dumps of the state.
 */

#include <libtcod.hpp>
#include <math.h>
#include <assert.h>

#include "firefly.h"

#include "rand.h"

#define BOX_DEPTH 5
#define DEFSPEED 30
#define MAXSPEED 60

#define STATE_ALIVE 0
#define STATE_DYING 1
#define STATE_DEAD 2


FIRETEX::FIRETEX(int w, int h)
{
    myW = w;
    myH = h;
    myRGB = new u8[w * h * 3];
    memset(myRGB, 0, w * h * 3);
    myRefCnt.set(0);
}

FIRETEX::~FIRETEX()
{
    delete [] myRGB;
}

void
FIRETEX::decRef()
{
    if (myRefCnt.add(-1) <= 0)
	delete this;
}

void
FIRETEX::incRef()
{
    myRefCnt.add(1);
}

void
FIRETEX::redraw(int sx, int sy) const
{
    int		x, y;
    FORALL_XY(x, y)
    {
	TCODConsole::root->setCharBackground(x+sx, y+sy, TCODColor(getR(x,y), getG(x,y), getB(x,y)));
    }
}

void
FIRETEX::firecurve(FireCurve firetype, float heat,
		    u8 *rgb)
{
    float	r, g, b;

    switch (firetype)
    {
	case FIRE_BLACKBODY:
	    r = heat * 255 * 3;
	    g = (heat - 1/3.0f) * 255 * 2;
	    b = (heat - 2/3.0f) * 255 * 1;
	    break;
	case FIRE_ICE:
	    b = heat * 255 * 3;
	    g = (heat - 1/3.0f) * 255 * 2;
	    r = (heat - 2/3.0f) * 255 * 1;
	    break;
	case FIRE_MONO:
	    r = g = b = heat * 255;
	    break;
    }

    if (r > 255) r = 255;
    if (g > 255) g = 255;
    if (b > 255) b = 255;
    if (r < 0) r = 0;
    if (g < 0) g = 0;
    if (b < 0) b = 0;

    rgb[0] = (u8) r;
    rgb[1] = (u8) g;
    rgb[2] = (u8) b;
}

void
FIRETEX::buildFromConstant(float cutoff, FireCurve firetype)
{
    int		x, y;
    int		ycut = height() - height() * cutoff;
    float	heat;

    for (y = 0; y < height(); y++)
    {
	heat = y / (float)height();
	heat *= 1.5;
	if (y < ycut)
	    heat = 0;

	u8		rgb[3];

	firecurve(firetype, heat, rgb);

	for (x = 0; x < width(); x++)
	{
	    setRGB(x, y, rgb[0], rgb[1], rgb[2]);
	}
    }
}

void
FIRETEX::buildFromParticles(const PARTICLELIST &particles, FireCurve firetype)
{
    int		x, y;
    float	heat;

    FORALL_XY(x, y)
    {
	// Clear to black.
	setRGB(x, y, 0, 0, 0);
    }

    for (int i = 0; i < particles.entries(); i++)
    {
	x = particles(i).myP.x();
	y = particles(i).myP.y();
	heat = particles(i).myP.z();

	if (x >= 0 && x < width() &&
	    y >= 0 && y < height())
	{
	    u8		rgb[3];

	    if (particles(i).myState == STATE_ALIVE)
	    {
		firecurve(firetype, (heat+BOX_DEPTH)/(2*BOX_DEPTH), rgb);
	    }
	    else
	    {
		// Modulate according to the particle id.
		float		bright = (rand_wanginthash(i) & 255) / 255.;

		bright = 0.25 + 0.5 * bright;

		rgb[0] = 255 * bright;
		rgb[1] = 192 * bright;
		rgb[2] = 128 * bright;
	    }
	    setRGB(x, y, rgb[0], rgb[1], rgb[2]);
	}
    }
}

static void *
fire_threadstarter(void *vdata)
{
    FIREFLY	*fire = (FIREFLY *) vdata;

    fire->mainLoop();

    return 0;
}


FIREFLY::FIREFLY(int count, int tw, int th, FireCurve firetype)
{
    myTexW = tw;
    myTexH = th;
    myFireType = firetype;

    myTex = new FIRETEX(myTexW, myTexH);
    myTex->incRef();

    myRatioLiving = 1.0;
void setType (TCOD_noise_type_t TCOD_NOISE_WAVELET);
    for (int n = 0; n < 3; n++)
    {
	myNoise[n] = new TCODNoise(3, 0.5, 2.0);

	// Force the noise to init in case it does JIT as we must be on
	// main thread to use its RNG>
	float	f[3] = { 0, 0 };
	myNoise[n]->getTurbulence(f, 4);
    }

    myThread = THREAD::alloc();
    myThread->start(fire_threadstarter, this);

    setParticleCount(count);
}

FIREFLY::~FIREFLY()
{
    myTex->decRef();

    for (int n = 0; n < 3; n++)
    {
	delete myNoise[n];
    }
}

void
FIREFLY::updateTex(FIRETEX *tex)
{
    AUTOLOCK	a(myTexLock);

    tex->incRef();
    myTex->decRef();
    myTex = tex;
}

FIRETEX *
FIREFLY::getTex()
{
    AUTOLOCK	a(myTexLock);

    myTex->incRef();

    return myTex;
}

void
FIREFLY::setRatioLiving(float size)
{
    // LEt us hope atomic :>
    myRatioLiving = size;
}

// Zany short circuit evaluation for the win!
#define FORALL_PARTICLES(ppart)		\
    for (int lcl_index = 0; lcl_index < myParticles.entries() && (ppart = myParticles.rawptr(lcl_index)); lcl_index++)

void
FIREFLY::mainLoop()
{
    int		lastms, ms;
    float	tinc, t;
    float	ratioliving;
    PARTICLE	*ppt;
    u8		*partcount;
    PARTICLELIST sortedlist;
    
    partcount = new u8[width()*height()];

    lastms = TCOD_sys_elapsed_milli();

    t = 0.0;

    while (1)
    {
	ms = TCOD_sys_elapsed_milli();

	// Clamp this at 100fps.
	if (ms - lastms < 10)
	{
	    TCOD_sys_sleep_milli(1);
	    continue;
	}

	// Rebuild particle list if requested.
	if (myPendingCountPending)
	{
	    myPendingCountPending = 0;
	    myParticles.resize(myPendingCount);

	    FORALL_PARTICLES(ppt)
	    {
		ppt->myP.x() = rand_double() * width();
		ppt->myP.y() = rand_double() * height();
		ppt->myP.z() = rand_double() * BOX_DEPTH;

		ppt->myV.x() = rand_double() - 0.5;
		ppt->myV.y() = rand_double() - 0.5;
		ppt->myV.z() = rand_double() - 0.5;

		ppt->myV *= DEFSPEED;

		ppt->myState = 0;
	    }
	}

	tinc = (ms - lastms) / 1000.0F;
	t += tinc;

	// Noise functions get crappy to far from 0.
	// This adds a "beat" of one minute.  Yeah!  That is why!
	if (t > 60) t -= 60;

	lastms = ms;

	// Read only once as unlocked..
	ratioliving = myRatioLiving;

	if (myParticles.entries() == 0)
	{
	    // Rebuild our texture...
	    FIRETEX		*tex;
	    tex = new FIRETEX(myTexW, myTexH);
	    tex->buildFromConstant(ratioliving, myFireType);

	    // Publish result.
	    updateTex(tex);
	}
	else
	{
	    // Update live/dead.
	    int		goalalive = (myParticles.entries() * ratioliving + 0.5);

	    bool	repiledead = false;

	    if (goalalive > myParticles.entries()) goalalive = myParticles.entries();

	    VEC3	centroid;

	    centroid = VEC3(width()/2, height()/2, BOX_DEPTH/2);
	    
	    int		numalive = 0;
	    FORALL_PARTICLES(ppt)
	    {
		if (ppt->myState == STATE_ALIVE)
		{
		    centroid += ppt->myP;
		    numalive++;
		}
	    }

	    centroid /= (numalive + 1);

	    // Keep near center.
	    centroid += VEC3(width()/2, height()/2, BOX_DEPTH/2);
	    centroid /= 2;

	    if (numalive != goalalive)
	    {
		// Kill random particles
		PTRLIST<PARTICLE *>	listtochange;

		if (numalive > goalalive)
		{
		    int		 numtokill = numalive - goalalive;
		    listtochange.clear();
		    FORALL_PARTICLES(ppt)
		    {
			if (ppt->myState == STATE_ALIVE)
			{
			    if (listtochange.entries() < numtokill)
			    {
				listtochange.append(ppt);
			    }
			    else
			    {
				// Replace an existing point at random
				listtochange( rand_choice(listtochange.entries()) ) = ppt;
			    }
			}
		    }

		    for (int i = 0; i < listtochange.entries(); i++)
		    {
			listtochange(i)->myState = STATE_DYING;
			listtochange(i)->myV.y() *= 0.5;
			numalive--;
		    }
		}

		// Resurrect random particles
		if (numalive < goalalive)
		{
		    int		 numtolive = goalalive - numalive;
		    listtochange.clear();
		    FORALL_PARTICLES(ppt)
		    {
			if (ppt->myState != STATE_ALIVE)
			{
			    if (listtochange.entries() < numtolive)
			    {
				listtochange.append(ppt);
			    }
			    else
			    {
				// Replace an existing point at random
				listtochange( rand_choice(listtochange.entries()) ) = ppt;
			    }
			}
		    }

		    for (int i = 0; i < listtochange.entries(); i++)
		    {
			listtochange(i)->myState = STATE_ALIVE;
			listtochange(i)->myV.x() = rand_double() - 0.5;
			listtochange(i)->myV.y() = rand_double() - 0.5;
			listtochange(i)->myV.z() = rand_double() - 0.5;

			listtochange(i)->myV *= DEFSPEED;
			numalive++;
		    }
		}
	    }

	    // Advect the particles.
	    FORALL_PARTICLES(ppt)
	    {
		if (ppt->myState != STATE_DEAD)
		    ppt->myP += ppt->myV * tinc;
	    }

	    // Forces on live particles.
	    FORALL_PARTICLES(ppt)
	    {
		if (ppt->myState == STATE_ALIVE)
		{
		    float		np[3];

		    // Towards center of flock
		    ppt->myV += (centroid - ppt->myP) * 1 * tinc;

		    // Random crap to keep it moving.

		    // NOTE: We have to inject a lcl_index effect to
		    // ensure particles don't fall onto the same noise lines!

		    np[0] = (5.0F*ppt->myP.x()) / width();
		    np[1] = (5.0F*ppt->myP.y()) / width();	// Yes, width.
		    np[2] = (ppt->myP.z() / BOX_DEPTH);

		    // Yummy salt.
		    np[0] += fmod((2.71828182845904523536 * lcl_index), 10.0);
		    np[1] += fmod((3.141593 * lcl_index), 13.0);
		    np[2] += t;	

		    VEC3		noise;

		    noise.x() = myNoise[0]->get(np) - 0.5;
		    noise.y() = myNoise[1]->get(np) - 0.5;
		    noise.z() = myNoise[2]->get(np) - 0.5;

		    ppt->myV += noise * tinc * 0.5;

		    // Speed limit
		    if (ppt->myV.length() > MAXSPEED)
		    {
			ppt->myV.normalize();
			ppt->myV *= MAXSPEED;
		    }
		}
	    }

	    // Gravity on dying particles
	    FORALL_PARTICLES(ppt)
	    {
		if (ppt->myState == STATE_DYING)
		{
		    ppt->myV.y() += 20 * tinc;
		}
	    }

	    // Bounce the particles
	    FORALL_PARTICLES(ppt)
	    {
		float	xbounce = 1.0;
		float	ybounce = 1.0;
		float	zbounce = 1.0;

		if (ppt->myState == STATE_DYING)
		{
		    ybounce = 0.0;
		}
		if (ppt->myState == STATE_DEAD)
		    continue;

		// Bounce.
		if (ppt->myP.x() < 0)
		{
		    ppt->myP.x() = -xbounce*ppt->myP.x()+.1;
		    ppt->myV.x() = -xbounce*ppt->myV.x();
		}
		if (ppt->myP.y() < 0)
		{
		    ppt->myP.y() = -ybounce*ppt->myP.y()+.1;
		    ppt->myV.y() = -ybounce*ppt->myV.y();
		}
		if (ppt->myP.z() < 0)
		{
		    ppt->myP.z() = -zbounce*ppt->myP.z()+.1;
		    ppt->myV.z() = -zbounce*ppt->myV.z();
		}

		if (ppt->myP.x() > width())
		{
		    ppt->myP.x() = (1+xbounce)*width()-xbounce*ppt->myP.x()-.1;
		    ppt->myV.x() = -xbounce*ppt->myV.x();
		}
		if (ppt->myP.y() > height())
		{
		    ppt->myP.y() = (1+ybounce)*height()-ybounce*ppt->myP.y()-.1;
		    ppt->myV.y() = -ybounce*ppt->myV.y();

		    if (ppt->myState == STATE_DYING)
		    {
			ppt->myState = STATE_DEAD;
			repiledead = true;
		    }
		}
		if (ppt->myP.z() > BOX_DEPTH)
		{
		    ppt->myP.z() = (1+zbounce)*BOX_DEPTH-zbounce*ppt->myP.z()-.1;
		    ppt->myV.z() = -zbounce*ppt->myV.z();
		}
	    }

	    if (true)
	    {
		sortedlist.clear();

		// This is the last day, I'm not doing anything pretty here!
		FORALL_PARTICLES(ppt)
		{
		    if (ppt->myState != STATE_DEAD)
			continue;

		    PARTICLE	part = *ppt;
		    part.myState = lcl_index;
		    sortedlist.append(part);
		}
		sortedlist.stableSort();

		memset(partcount, 0, width()*height());
		int			 id;

		int			partperbucket;

		// 20% capacity, approx :>
		partperbucket = MAX(1, (5.0 * myParticles.entries()) / (width()*height()));

		for (int i = sortedlist.entries(); i-->0; )
		{
		    id = sortedlist(i).myState;

		    int	x = sortedlist(i).myP.x();
		    int	y = sortedlist(i).myP.y();
		    x = BOUND(x, 0, width()-1);
		    y = BOUND(y, 0, height()-1);

		    while (y > 0 && (partcount[x+y*width()] >= partperbucket))
		    {
			// Bubble up
			myParticles(id).myP.y() -= 1;
			y--;
		    }

		    if (y < 0)
		    {
			// We filled up!
			myParticles(id).myP.y() = 0;
			continue;
		    }

		    // Bubble down!
		    while (y < height()-1)
		    {
			if (partcount[x+(y+1)*width()] < partperbucket)
			{
			    // Fall straight.
			    y++;
			    myParticles(id).myP.y() += 1;
			    continue;
			}
			int		dx = 0;
			int		nfound = 0;
			if ((x < width()-1) 
			    && (partcount[x+1+(y+1)*width()] < partperbucket))
			{
			    // Down right
			    dx = 1;
			    nfound = 1;
			}
			if ((x > 0)
			    && (partcount[x-1+(y+1)*width()] < partperbucket))
			{
			    // Down left
			    if (!nfound || !rand_choice(2))
			    {
				dx = -1;
				nfound++;
			    }
			}

			if (nfound)
			{
			    y++;
			    x += dx;
			    myParticles(id).myP.x() += dx;
			    myParticles(id).myP.y() += 1;
			}
			else
			{
			    // Hit supporting structure
			    break;
			}
		    }

		    // Write out our new particle
		    partcount[x + y * width()]++;
		    if (partcount[x+y*width()] > partperbucket)
			partcount[x + y *width()] = partperbucket;
		}

		// Kill any dying particles that have hit a dead particle
		// Bounce any living particles out.
		FORALL_PARTICLES(ppt)
		{
		    int	x = ppt->myP.x();
		    int	y = ppt->myP.y();

		    x = BOUND(x, 0, width()-1);
		    y = BOUND(y, 0, height()-1);

		    if (ppt->myState == STATE_DYING)
		    {
			if (partcount[x + y * width()])
			    ppt->myState = STATE_DEAD;
		    }
		    if (ppt->myState == STATE_ALIVE)
		    {
			if (partcount[x + y * width()])
			{
			    if (ppt->myV.y() > 0)
				ppt->myV.y() = -ppt->myV.y();
			}
		    }
		}
	    }

	    // Rebuild our texture...
	    FIRETEX		*tex;
	    tex = new FIRETEX(myTexW, myTexH);
	    tex->buildFromParticles(myParticles, myFireType);

	    // Publish result.
	    updateTex(tex);
	}
    }

    // HAHA!  As if this is ever reached.
    delete [] partcount;
}
