/* This file is part of the KMPlayer project
 *
 * Copyright (C) 2005 Koos Vriezen <koos.vriezen@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * until boost gets common, a more or less compatable one ..
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <gtk/gtkmain.h>
#include <gio/gio.h>

#include "kmplayertypes.h"

using namespace KMPlayer;

KMPLAYER_NO_CDTOR_EXPORT ByteArrayImpl::ByteArrayImpl (const ByteArrayImpl & b)
 : m_data (b.m_size ? (char *) malloc (b.m_size + 1) : 0L),
   m_size (b.m_size),
   m_capacity (b.m_size),
   data_owner (true) {
    if (m_size) {
        memcpy (m_data, b.m_data, m_size);
        m_data[m_size] = 0;
    }
}

KMPLAYER_NO_EXPORT void ByteArrayImpl::resize (unsigned ns) {
    if (ns > m_capacity) {
        char * tmp = (char *) malloc (ns + 1);
        if (m_size > 0)
            memcpy (tmp, m_data, m_size);
        m_capacity = m_size = ns;
        if (m_data && data_owner)
            free (m_data);
        m_data = tmp;
        m_data[ns] = 0;
        data_owner = true;
    } else
        m_size = ns;
}

KMPLAYER_NO_EXPORT
ByteArrayImpl & ByteArrayImpl::operator = (const ByteArrayImpl & b) {
    if (m_data && data_owner)
        free (m_data);
    m_data = (b.m_size ? (char*) malloc (b.m_size + 1) : 0L);
    m_size = b.m_size;
    m_capacity = b.m_size;
    data_owner = true;
    if (m_size) {
        memcpy (m_data, b.m_data, m_size);
        m_data[m_size] = 0;
    }
}

KMPLAYER_NO_EXPORT ByteArrayImpl &ByteArrayImpl::operator = (const char *b) {
    int len = strlen (b);
    resize (len);
    memcpy (m_data, b, len);
    return *this;
}

//-----------------------------------------------------------------------------

void *FixedSizeAllocator::alloc () {
    return count ? pool[--count] : malloc (size);
}

void FixedSizeAllocator::dealloc (void *p) {
    if (count < 10)
        pool[count++] = p;
    else
        free (p);
}

//-----------------------------------------------------------------------------

static FixedSizeAllocator *shared_data_pool;

namespace KMPlayer {

void *sharedDataAlloc (size_t s) {
    if (!shared_data_pool)
        shared_data_pool = new FixedSizeAllocator (s);

    return shared_data_pool->alloc ();
}

void sharedDataFree (void *p) {
    shared_data_pool->dealloc (p);
}

}

//-----------------------------------------------------------------------------

static FixedSizeAllocator *shared_chars_pool;

void *SharedChars::operator new (size_t s) {
    if (!shared_chars_pool)
        shared_chars_pool = new FixedSizeAllocator (sizeof (SharedChars));

    return shared_chars_pool->alloc ();
}

void SharedChars::operator delete (void *p) {
    shared_chars_pool->dealloc (p);
}

template <>
void Shared<SharedChars>::attach (const Shared<SharedChars> &other) {
    if (data != other.data) {
        if (other.data)
            other.data->ref ();
        if (data)
            data->unref ();
        data = other.data;
    }
}

GLibString::GLibString (const Char & ch) {
    gchar buf [8];
    int nr = g_unichar_to_utf8 (ch.unicode (), buf);
    buf [nr] = 0;
    takeUtf8 (g_strdup (buf), nr);
}

void GLibString::takeUtf8 (char *p, int buf_len) {
    if (!data || data->ptr != p) {
        if (data && 1 == data->ref_count) {
            g_free (data->ptr);
            data->ptr = p;
            data->buf_len = buf_len;
        } else {
            SharedChars *tmp = p ? new SharedChars (p, buf_len) : 0L;
            if (data)
                data->unref ();
            data = tmp;
        }
    }
}

int GLibString::compare (const char *str) const {
    if (!str && !data)
        return 0;
    if (str && data)
        return strcmp (str, data->ptr);
    return data ? 1 : -1;
}

int GLibString::compare (const GLibString & str) const {
    if (str.data == data)
        return 0;
    if (str.data && data)
        return strcmp (str.data->ptr, data->ptr);
    return data ? 1 : -1;
}

void GLibString::take (char * s) {
    const gchar *end;
    if (g_utf8_validate (s, -1, &end)) {
        takeUtf8 (s, end - s);
    } else {
        gsize bytes_read, bytes_written;
        takeUtf8 (g_locale_to_utf8 (s, -1, &bytes_read, &bytes_written, 0L));
    }
}

GLibString & GLibString::operator = (const char * s) {
    if (!s)
        clear ();
    else
        take (g_strdup (s));
    return *this;
}

GLibString & GLibString::operator = (const Char & ch) {
    gchar buf [8];
    int nr = g_unichar_to_utf8 (ch.unicode (), buf);
    buf [nr] = 0;
    takeUtf8 (g_strdup (buf), nr);
    return *this;
}

static char *strCat (const gchar *p1, int len1, const gchar *p2, int len2) {
    char *p = (char *) g_malloc (len1 + len2 + 1);
    memcpy (p, p1, len1);
    memcpy (p + len1, p2, len2);
    p [len1 + len2] = 0;
    return p;
}

void GLibString::append (const char *p, int length) {
    if (!data) {
        takeUtf8 (g_strndup (p, length), length);
    } else {
        if (data->buf_len < 0)
            data->buf_len = strlen (data->ptr);
        if (1 == data->ref_count) {
            data->ptr = (gchar *) g_realloc (data->ptr,
                    data->buf_len + length + 1);
            memcpy (data->ptr + data->buf_len, p, length);
            data->buf_len += length;
            data->ptr [data->buf_len] = 0;
        } else {
            takeUtf8 (strCat (data->ptr, data->buf_len, p, length),
                    data->buf_len + length);
        }
    }
}

GLibString & GLibString::operator += (const GLibString &str) {
    if (str.data) {
        if (!data) {
            attach (str);
        } else {
            int slen = str.data->buf_len;
            if (slen < 0)
                slen = strlen (str.data->ptr);
            append (str.data->ptr, slen);
        }
    }
    return *this;
}

KMPLAYER_NO_EXPORT GLibString &GLibString::operator += (const char *str) {
    append (str, strlen (str));
    return *this;
}

KMPLAYER_NO_EXPORT GLibString &GLibString::operator += (const Char &ch) {
    gchar buf [8];
    int nr = g_unichar_to_utf8 (ch.unicode (), buf);
    buf [nr] = 0;
    append (buf, nr);
    return *this;
}

GLibString GLibString::fill (Char ch, int length) {
    GLibString str;
    str.take (g_strnfill (length, (const char) ch)); //FIME
    return str;
}

GLibString GLibString::stripWhiteSpace () const {
    String str;
    if (data && data->ptr [0]) {
        gchar *iter = data->ptr;
        gchar *first = iter;
        gchar *end = iter;
        Char ch (g_utf8_get_char (iter));
        iter = g_utf8_next_char (iter);
        while (ch.isSpace () && iter && *iter) { // skip leading ws
            ch = Char (g_utf8_get_char (iter));
            first = iter;
            iter = g_utf8_next_char (iter);
        }
        if (ch.isSpace ())
            return str;
        end = first + 1;
        do {
            if (!iter || !*iter)
                break;
            ch = Char (g_utf8_get_char (iter));
            iter = g_utf8_next_char (iter);
            while (ch.isSpace () && iter && *iter) {
                ch = Char (g_utf8_get_char (iter));
                iter = g_utf8_next_char (iter);
            }
            if (ch.isSpace ())
                break;  // skip ending ws
            end = iter;
        } while (iter && *iter);
        if (end > first)
            str.append (first, end - first);
    }
    return str;
}

bool GLibString::startsWith (const GLibString & prefix) const {
    if (!prefix.data && !data)
        return true;
    if (prefix.data && data)
        return g_str_has_prefix (data->ptr, prefix.data->ptr);
    return !prefix.data;
}

bool GLibString::endsWith (const GLibString & suffix) const {
    if (!suffix.data && !data)
        return true;
    if (suffix.data && data)
        return g_str_has_suffix (data->ptr, suffix.data->ptr);
    return !suffix.data;
}

int GLibString::indexOf (const GLibString & needle) const {
    if (!data)
        return needle.data ? -1 : 0;
    if (!needle.data)
        return 0;
    gchar * str = strstr (data->ptr, needle.data->ptr);
    if (!str)
        return -1;
    return int (str - data->ptr);
}

int GLibString::lastIndexOf (const GLibString & needle) const {
    if (!data)
        return needle.data ? -1 : 0;
    if (!needle.data)
        return length ();
    gchar * str = g_strrstr (data->ptr, needle.data->ptr);
    if (!str)
        return -1;
    return int (str - data->ptr);
}

GLibString GLibString::replace (Char c1, Char c2) const {
    GLibString str;
    if (data) {
        gchar cstr1 [8];
        int nr1 = g_unichar_to_utf8 (c1.unicode (), cstr1);
        cstr1 [nr1] = 0;
        gchar cstr2 [8];
        int nr2 = g_unichar_to_utf8 (c2.unicode (), cstr2);
        cstr2 [nr2] = 0;
        gchar *s = data->ptr;
        gchar *p = strstr (s, cstr1);
        while (p) {
            if (p != s)
                str.append (s, p - s);
            str.append (cstr2, nr2);
            s = p + nr1;
            p = strstr (s, cstr1);
        }
        if (s[0])
            str.append (s, strlen (s));
    }
    return str;
}

int GLibString::toInt (bool * valid, int base) const {
    int i = 0;
    if (data) {
        char *endptr;
        i = (int) strtol (data->ptr, &endptr, base);
        if (valid)
            *valid = (data->ptr != endptr);
    } else if (valid)
        *valid = false;
    return i;
}

KMPLAYER_NO_EXPORT double GLibString::toDouble (bool * valid) const {
    if (data) {
        gchar * endptr;
        double d = g_ascii_strtod (data->ptr, &endptr);
        if (valid)
            *valid = (endptr != data->ptr);
        return d;
    } else {
        if (valid)
            *valid = false;
        return 0.0;
    }
}

GLibString GLibString::mid (const int index, unsigned len) const {
    if (!data)
        return GLibString ();
    gchar * off = g_utf8_offset_to_pointer (data->ptr, index);
    gchar * eos = off;
    if (!off)
        return GLibString ();
    while (eos && *eos && len--)
        eos = g_utf8_next_char (eos);
    GLibString str;
    if (!eos || !*eos)
        str.takeUtf8 (g_strdup (off));
    else
        str.takeUtf8 (g_strndup (off, eos-off), eos - off);
    return str;
}

GLibString GLibString::left (int index) const {
    if (!data)
        return GLibString ();
    if (index < 0)
        return GLibString (*this);
    gchar * off = g_utf8_offset_to_pointer (data->ptr, index);
    if (!off)
        return GLibString ();
    GLibString str;
    str.takeUtf8 (g_strndup (data->ptr, off - data->ptr));
    return str;
}

void GLibString::truncate (int pos) {
    if (data) {
        if (pos == 0) {  // speed up clearing string
            clear ();
        } else {
            gchar *epos = g_utf8_offset_to_pointer (data->ptr, pos);
            if (epos)
                takeUtf8 (g_strndup (data->ptr, epos - data->ptr),
                          epos - data->ptr);
        }
    }
}

//-----------------------------------------------------------------------------

namespace KMPlayer {

class GIOFilePrivate {
public:
    GIOFilePrivate () : file (NULL), fin (NULL), fout (NULL) {}
    ~GIOFilePrivate () {}
    GFile *file;
    GInputStream *fin;
    GOutputStream *fout;
    String m_path;
};

}

KMPLAYER_NO_CDTOR_EXPORT
GIOFile::GIOFile (const String & p) : d (new GIOFilePrivate) {
    d->m_path = p;
    d->file = g_file_new_for_path ((const char *) d->m_path);
}

KMPLAYER_NO_CDTOR_EXPORT GIOFile::~GIOFile () {
    close ();
    g_object_unref (G_OBJECT (d->file));
    d->file = NULL;
    delete d;
}

KMPLAYER_NO_EXPORT bool GIOFile::open (FileMode mode) {
    switch (mode) {
        case IO_ReadOnly:
            d->fin = G_INPUT_STREAM (g_file_read (d->file, NULL, NULL));
            if (d->fin)
                return true;
            break;
        case IO_WriteOnly: // fixme truncate
            d->fout = G_OUTPUT_STREAM (g_file_replace (d->file, NULL, false,
                    G_FILE_CREATE_NONE, NULL, NULL));
            if (d->fout)
                return true;
            break;
        default:
            warningLog() << "open " << int (mode) << " not implemented" << endl;
    }
    return false;
}

KMPLAYER_NO_EXPORT void GIOFile::close () {
    if (d->fin) {
        g_object_unref (G_OBJECT (d->fin));
        d->fin = NULL;
    }
    if (d->fout) {
        g_object_unref (G_OBJECT (d->fout));
        d->fout = NULL;
    }
}

KMPLAYER_NO_EXPORT ByteArray GIOFile::readAll () {
    ByteArray str;
    char buf [2048];
    gsize nr;
    if (d->fin)
        close (); // seek 0
    if (!open (IO_ReadOnly))
        return str;
    do {
        nr = g_input_stream_read (d->fin, buf, sizeof (buf), NULL, NULL);
        if (nr <= 0)
            break;
        off_t size = str.size ();
        str.resize (size + nr);
        memcpy (str.data () + size, buf, nr);
    } while (true);
    return str;
}

KMPLAYER_NO_EXPORT off_t GIOFile::readBlock (char * buf, off_t max) {
    if (!d->fin && !open (IO_ReadOnly))
        return 0;
    gsize nr = g_input_stream_read (d->fin, buf, max, NULL, NULL);
    return nr < 0 ? 0 : nr;
}

KMPLAYER_NO_EXPORT bool GIOFile::exists () {
    return g_file_query_exists (d->file, NULL);
}

KMPLAYER_NO_EXPORT off_t GIOFile::size () {
    goffset sz = 0;
    GFileInfo *info = g_file_query_info (d->file,
            G_FILE_ATTRIBUTE_STANDARD_SIZE,
            G_FILE_QUERY_INFO_NONE, NULL, NULL);
    if (info) {
        sz = g_file_info_get_size (info);
        g_object_unref (info);
    }
    return sz;
}

KMPLAYER_NO_EXPORT off_t GIOFile::writeBlock (const char * data, off_t length) {
    gsize nr;
    g_output_stream_write_all (d->fout, data, length, &nr, NULL, NULL);
    return nr;
}

//-----------------------------------------------------------------------------

GioUrl::GioUrl (const GioUrl &base, const String &rel) {
    if (base.m_file) {
        GFile *p = g_file_get_parent (base.m_file);
        m_file = g_file_resolve_relative_path (p ? p : base.m_file, rel);
        if (p)
            g_object_unref (p);
    } else {
        m_file = g_file_new_for_uri ((const char *) rel);
    }
}

KMPLAYER_NO_CDTOR_EXPORT GioUrl::GioUrl (const String &url) : m_file (NULL) {
    if (url.startsWith (Char ('/')))
        m_file = g_file_new_for_path (url);
    else if (!url.isEmpty ())
        m_file = g_file_new_for_uri ((const char *) url);
}

GioUrl::GioUrl (const File &file)
{
    m_file = g_file_dup (file.d->file);
}

KMPLAYER_NO_CDTOR_EXPORT GioUrl::GioUrl (const GioUrl & url) {
    m_file = (url.m_file ? g_file_dup (url.m_file) : 0L);
}

KMPLAYER_NO_CDTOR_EXPORT GioUrl::~GioUrl () {
    if (m_file)
        g_object_unref (m_file);
}

KMPLAYER_NO_EXPORT String GioUrl::url () const {
    if (m_file) {
        GLibString s;
        s.take (g_file_get_uri (m_file));
        return String (s);
    }
    return String ();
}

GioUrl & GioUrl::operator = (const String &url) {
    if (m_file)
        g_object_unref (m_file);
    m_file = g_file_new_for_uri ((const char *) url);
    return *this;
}

GioUrl & GioUrl::operator = (const GioUrl & url) {
    if (m_file)
        g_object_unref (m_file);
    m_file = url.m_file ? g_file_dup (url.m_file) : 0L;
    return *this;
}

KMPLAYER_NO_EXPORT String GioUrl::path () const {
    if (m_file)
        return String (g_file_get_path (m_file));
    return String ();
}

KMPLAYER_NO_EXPORT String GioUrl::protocol () const {
    if (m_file)
        return String (g_file_get_uri_scheme (m_file));
    return String ();
}

KMPLAYER_NO_EXPORT String GioUrl::decode_string (const String & s) {
    if (s.isEmpty ())
        return String ();
    String str;
    GLibString &gs = str;
    gs.take (g_uri_unescape_string ((const char *) s, ""));
    return str;
}

KMPLAYER_NO_EXPORT String GioUrl::encode_string (const String & s) {
    if (s.isEmpty ())
        return String ();
    String str;
    GLibString &gs = str;
    gs.take(g_uri_escape_string ((const char *) s, "", TRUE));
    return str;
}

KMPLAYER_NO_EXPORT bool GioUrl::isLocalFile () const {
    if (m_file)
        return g_file_is_native (m_file) ||
            protocol () == "file";
    return false;
}

//-----------------------------------------------------------------------------

String GMimeType::findByContent (const char *data, int nr) {
    if (nr <= 0)
        return String();
    const gchar *end;
    if (g_utf8_validate ((gchar *) data, nr, &end))
        return "text/plain";
    if (g_utf8_get_char_validated (end, nr - (end - data)) == (gunichar)-2)
        return "text/plain";
    gboolean pb;
    return String (g_content_type_guess (NULL, (const guchar *)data, nr, &pb));
}

String GMimeType::findByURL (const URL &url) {
    gboolean pb;
    if (url.isLocalFile ())
        return String (g_content_type_guess (url.path (), NULL, 200, &pb));
    warningLog () << "GMimeType::findByURL not supported on remote links\n";
    return String ();
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT
GLibTextStream::GLibTextStream (const String *buffer)
    : m_gstring (NULL), m_utf8_str (NULL), m_iter ((const char *) *buffer) {
}

GLibTextStream & GLibTextStream::operator >> (Char & ch) {
    if (m_gstring) {
        m_utf8_str = g_string_free (m_gstring, false);
        m_gstring = 0L;
        m_iter = m_utf8_str;
    }
    if (m_iter) {
        ch = Char (g_utf8_get_char (m_iter));
        m_iter = g_utf8_next_char (m_iter);
    }
}

#if 0
KMPLAYER_NO_EXPORT
GLibTextStream & GLibTextStream::operator = (const GLibTextStream & s) {
    if (m_gstring)
        g_string_free (m_gstring, true);
    else if (m_utf8_str)
        g_free (m_utf8_str);
    m_utf8_str = 0L;
    m_iter = 0L;
    m_gstring = g_string_new (s.m_gstring ? s.m_gstring->str :
            (s.m_utf8_str ? s.m_utf8_str : ""));
    return *this;
}
#endif

KMPLAYER_NO_EXPORT String GLibTextStream::release () {
    GLibString str;
    if (m_gstring) {
        str.takeUtf8 (g_string_free (m_gstring, false));
        m_gstring = 0L;
    }
    return str;
}

String GLibTextStream::readLine () {
    String str;
    Char ch;
    while (!atEnd ()) {
        *this >> ch;
        if (ch == '\r' || ch == '\n')
            break;
        str += ch;
    }
    const gchar *iter = m_iter;
    while (!atEnd ()) { // FIXME: current is skipping emty lines as well
        *this >> ch;
        if (ch != '\r' && ch != '\n') {
            m_iter = iter;
            break;
        }
        iter = m_iter;
    }
    return str;
}

//-----------------------------------------------------------------------------
#if 0
namespace KMPlayer {

    class GnomeVFSGetJob : public IOJob {
    public:
        GnomeVFSGetJob (IOJobListener * rec, const String & u);
        ~GnomeVFSGetJob ();
        void kill (bool quiet = true);
        void finished ();
        bool error ();

        GnomeVFSAsyncHandle * network_handle;
        IOJobListener * m_receiver;
        bool job_closed;
        bool job_error;
        bool keep_quiet;
        bool continuing;
        char read_buf [2048];
    };

    IOJob * asyncGet (IOJobListener * receiver, const String & url) {
        return new GnomeVFSGetJob (receiver, url);
    }
}

//static GTree *uri_info;

//void getRedirection (gpointer data, gpointer d) {
//    debugLog() << " getRedirection: " << (char *)data << endl;
//}

//static void globalVFSAsyncModuleCallback (gconstpointer in,
//        gsize in_size,
//        gpointer out,
//        gsize out_size,
//        gpointer callback_data/*,
//        GnomeVFSModuleCallbackResponse response,
//        gpointer response_data*/) {
//    GnomeVFSModuleCallbackReceivedHeadersIn *in_args =
//        (GnomeVFSModuleCallbackReceivedHeadersIn *)in;
//    debugLog() << "globalVFSAsyncModuleCallback " << 
//        in_args->uri->text << endl;
//    g_list_foreach (in_args->headers, getRedirection, NULL);
//    //response (response_data);
//}


static void cb_async_get_close (GnomeVFSAsyncHandle *handle, GnomeVFSResult result, gpointer p) {
    //debugLog () << "cb_async_get_close " << endl;
    if (!p)
        return;
    GnomeVFSGetJob * get = static_cast <GnomeVFSGetJob *> (p);
    get->finished ();
}

static void cb_async_get_read (GnomeVFSAsyncHandle *handle, GnomeVFSResult result, gpointer buffer, GnomeVFSFileSize bytes_requested, GnomeVFSFileSize bytes_read, gpointer p) {
    GnomeVFSGetJob * get = static_cast <GnomeVFSGetJob *> (p);
    if (get->job_closed)
        return;
    get->continuing = true;
    if (bytes_read > 0) {
    debugLog () << "cb_async_get_read " << handle << endl;
        ByteArray data (bytes_read);
        memcpy (data.data (), (const char *) buffer, bytes_read);
        get->m_receiver->jobData (get, data);
    }
    //debugLog () << "cb_async_get_read " << int (bytes_read) << " canceled:" << get->job_closed << " not ok:" << (result != GNOME_VFS_OK) << " less:" << (bytes_read < bytes_requested) << " killed:" << !get->continuing << endl;
    if (result != GNOME_VFS_OK || bytes_read <= 0 || !get->continuing) {
        // || bytes_read < bytes_requested 
        get->job_closed = true;
        get->job_error = result != GNOME_VFS_ERROR_EOF;
        gnome_vfs_async_cancel (handle);
        gnome_vfs_async_close (handle, cb_async_get_close, p);
    } else {
        gnome_vfs_async_read (handle, get->read_buf, sizeof (get->read_buf) -1, cb_async_get_read, p);
    }
    get->continuing = false;
}

static void cb_async_get_open (GnomeVFSAsyncHandle *handle, GnomeVFSResult result, gpointer p) {
    debugLog () << "cb_async_get_open " << handle << endl;
    GnomeVFSGetJob * get = static_cast <GnomeVFSGetJob *> (p);
    if (get->job_closed)
        return;
    if (result != GNOME_VFS_OK)
        get->finished ();  // kills get
    else
        gnome_vfs_async_read (handle, get->read_buf, sizeof (get->read_buf) -1, cb_async_get_read, p);
}

GnomeVFSGetJob::GnomeVFSGetJob (IOJobListener * rec, const String &url)
 : m_receiver (rec), network_handle (0L),
   job_closed (false), job_error (false),
   keep_quiet (false), continuing (false) {

    //if (!uri_info) {
    //    uri_info = g_tree_new ((GCompareFunc)::strcmp);
    //    gnome_vfs_module_callback_set_default
    //        (GNOME_VFS_MODULE_CALLBACK_HTTP_RECEIVED_HEADERS,
    //         globalVFSAsyncModuleCallback,
    //         uri_info,
    //         NULL);
    //}
    gnome_vfs_async_open (&network_handle, (const char *) url, GNOME_VFS_OPEN_READ, 0, cb_async_get_open , static_cast <void *> (this));
    debugLog() << "GnomeVFSGetJob " << network_handle << " " << url << endl;
}

GnomeVFSGetJob::~GnomeVFSGetJob () {
    if (network_handle && !job_closed) {
        gnome_vfs_async_cancel (network_handle);
        gnome_vfs_async_close (network_handle, cb_async_get_close, 0L);
    }
}

void GnomeVFSGetJob::kill (bool quiet) {
    keep_quiet = quiet;
    if (network_handle && !job_closed) {
        if (!continuing) {
            job_closed = true;
            gnome_vfs_async_cancel (network_handle);
            gnome_vfs_async_close (network_handle, cb_async_get_close, static_cast <void *> (this));//
        } else
            continuing = false;  // killed from jobData receiver callback
    } else if (!quiet)
        m_receiver->jobResult (this);
}

void GnomeVFSGetJob::finished () {
    network_handle = 0L;
    if (!keep_quiet)
        m_receiver->jobResult (this);
    delete this;
}

bool GnomeVFSGetJob::error () {
    return job_error;
}
#endif
//-----------------------------------------------------------------------------

#include <conic.h>
#include <curl/curl.h>

static gint easyCompare (gconstpointer a, gconstpointer b) {
    return (long)a - (long)b;
}

namespace KMPlayer {
    class CurlGetJob {
    public:
        CurlGetJob (IOJobListener * rec, const String & u);
        ~CurlGetJob ();

        void init (IOJob *job);
        void start ();

        static size_t headerLine( void *p, size_t sz, size_t n, void *strm);
        static size_t dataReceived (void *p, size_t sz, size_t n, void *strm);
        static size_t writePost (void *p, size_t sz, size_t n, void *strm);
        static gboolean findKilledJob (gpointer key, gpointer val, gpointer d);

        IOJobListener * m_receiver;

        static bool global_init;
        static CURLM *multi;

        String url;
        String content_type;
        ByteArray post_data;
        curl_slist *headers;
        CURL *easy;
        bool quiet;
        bool killed;
        bool redirected;
        int error_code;
        unsigned long content_length;
    };
}

namespace {

    struct FdWatch {
        int fd;
        int chan_read_watch;
        int chan_write_watch;
    };

}

namespace KMPlayer {

    IOJob *asyncGet (IOJobListener *receiver, const String &url) {
        return new IOJob (new CurlGetJob (receiver, url));
    }
}

inline gboolean
CurlGetJob::findKilledJob (gpointer key, gpointer value, gpointer data) {
    IOJob *job = (IOJob *)value;
    CurlGetJob *self = job->d;

    if (self->killed) {
        *(IOJob **)data = job;
        return true;
    }
    return false;
}

static gboolean curlChannelRead (GIOChannel *src, GIOCondition c, gpointer d);
static gboolean curlChannelWrite (GIOChannel *src, GIOCondition c, gpointer d);
static gboolean curlTimeOut (void *);
static void curlPerformMulti ();
bool CurlGetJob::global_init = false;
CURLM *CurlGetJob::multi = NULL;
static GTree *easy_list = NULL;
static FdWatch *fd_watch_array;
static int fd_watch_array_len;
static gint curl_timeout_timer;
static bool in_perform_multi;
static bool curl_job_killed;
static ConIcConnection *con_ic_connection;
#ifdef __ARMEL__
static ConIcConnectionStatus ic_status = CON_IC_STATUS_DISCONNECTED;
#else
static ConIcConnectionStatus ic_status = CON_IC_STATUS_CONNECTED;
#endif
static GSList *ic_connect_waiters;
static bool ic_connect_request;

void icConnection (ConIcConnection *conn, ConIcConnectionEvent *ev, gpointer) {
    ic_connect_request = false;
    ic_status = con_ic_connection_event_get_status (ev);
    if (CON_IC_STATUS_CONNECTED == ic_status && ic_connect_waiters) {
        ic_connect_waiters = g_slist_reverse (ic_connect_waiters);
        for (GSList *sl = ic_connect_waiters; sl; sl = sl->next)
            ((CurlGetJob *)sl->data)->start ();
        g_slist_free (ic_connect_waiters);
        ic_connect_waiters = NULL;
        curlPerformMulti ();
    }
}

static void curlChannelDestroy (gpointer) {
    //debugLog() << "channel destoyed" << endl;
}

static void updateReadWatches() {
    fd_set fd_read, fd_write, fd_excep;
    FD_ZERO (&fd_read);
    FD_ZERO (&fd_write);
    FD_ZERO (&fd_excep);
    int maxfd = 0;
    curl_multi_fdset (CurlGetJob::multi, &fd_read, &fd_write, &fd_excep, &maxfd);
    ++maxfd;
    if (maxfd > 0 && !fd_watch_array) {
        fd_watch_array = new FdWatch[maxfd];
        memset (fd_watch_array, 0, maxfd * sizeof (FdWatch));
        fd_watch_array_len = maxfd;
    } else if (maxfd > fd_watch_array_len) {
        FdWatch *tmp = fd_watch_array;
        fd_watch_array = new FdWatch[maxfd];
        memset (fd_watch_array, 0, maxfd * sizeof (FdWatch));
        memcpy (fd_watch_array, tmp, fd_watch_array_len * sizeof (FdWatch));
        delete[] tmp;
        fd_watch_array_len = maxfd;
    }
    for (int i = 3; i < fd_watch_array_len; i++) {
        if (FD_ISSET (i, &fd_read)) {
            //debugLog () << "update watches for read fd=" << i << endl;
            if (!fd_watch_array[i].chan_read_watch) {
                fd_watch_array[i].fd = i;
                GIOChannel *channel = g_io_channel_unix_new (i);
                g_io_channel_set_encoding (channel, NULL, NULL);
                fd_watch_array[i].chan_read_watch = g_io_add_watch_full (channel, G_PRIORITY_DEFAULT, G_IO_IN, curlChannelRead, NULL, curlChannelDestroy);
                g_io_channel_unref (channel);
            }
        } else if (fd_watch_array[i].chan_read_watch) {
            g_source_remove (fd_watch_array[i].chan_read_watch);
            fd_watch_array[i].chan_read_watch = 0;
        }
        if (!fd_watch_array[i].chan_read_watch && FD_ISSET (i, &fd_write)) {
            // connecting ..
            //debugLog () << "update watches for write fd=" << i << endl;
            if (!fd_watch_array[i].chan_write_watch) {
                fd_watch_array[i].fd = i;
                GIOChannel *channel = g_io_channel_unix_new (i);
                g_io_channel_set_encoding (channel, NULL, NULL);
                fd_watch_array[i].chan_write_watch = g_io_add_watch_full (channel, G_PRIORITY_DEFAULT, G_IO_OUT, curlChannelWrite, NULL, curlChannelDestroy);
                g_io_channel_unref (channel);
            }
        } else if (fd_watch_array[i].chan_write_watch) {
            g_source_remove (fd_watch_array[i].chan_write_watch);
            fd_watch_array[i].chan_write_watch = 0;
        }
    }
}

static void curlPerformMulti () {

    in_perform_multi = true;

    int running_handles = 0;
    CURLMcode curlCode;

    do {
        curl_job_killed = false;
        curlCode = curl_multi_perform (CurlGetJob::multi, &running_handles);
	if (curl_job_killed)
            do {
                IOJob *job = NULL;
                g_tree_traverse (easy_list, CurlGetJob::findKilledJob, G_IN_ORDER, &job);
                if (job)
                    delete job;
                else
                    break;
            } while (true);
    } while (CURLM_CALL_MULTI_PERFORM == curlCode && running_handles > 0);

    int messages;
    CURLMsg* msg = curl_multi_info_read (CurlGetJob::multi, &messages);
    while (msg) {
        if (CURLMSG_DONE == msg->msg) {
            IOJob *job = (IOJob *)g_tree_lookup (easy_list, msg->easy_handle);
            if (job)
                delete job;
        }
        msg = curl_multi_info_read (CurlGetJob::multi, &messages);
    }
    updateReadWatches();

    in_perform_multi = false;

    if (!curl_timeout_timer && g_tree_height (easy_list))
        curl_timeout_timer = g_timeout_add (250, curlTimeOut, NULL);
}

static gboolean curlTimeOut (void *) {
    curlPerformMulti ();
    if (g_tree_height (easy_list))
        return 1; // continue
    curl_timeout_timer = 0;
    debugLog() << "clearing polling" << endl;
    return 0; // stop
}

static gboolean curlChannelRead (GIOChannel *src, GIOCondition c, gpointer d) {
    curlPerformMulti ();
}

static gboolean curlChannelWrite (GIOChannel *src, GIOCondition c, gpointer d) {
    //debugLog() << "curl connected" << endl;
    curlPerformMulti ();
}
size_t CurlGetJob::headerLine( void *ptr, size_t size, size_t nmemb, void *stream) {
    IOJob *job = (IOJob *) stream;
    CurlGetJob *self = job->d;
    //char buf[256];
    //snprintf (buf, sizeof (buf), "%s", (char *)ptr);
    //debugLog() << "header: " << buf << endl;
    // Last-Modified:
    if (!self->killed) {
        if (!strncmp ((char *)ptr, "HTTP/", 5)) {
            char *p = strchr ((char *)ptr + 5, ' ');
            if (p) {
                int rc = strtol (p +1, NULL, 10);
                if (rc >= 400) {
                    self->error_code = rc;
                    gchar *buf = g_strndup (p + 1, size * nmemb);
                    debugLog() << "CurlGetJob err: " << buf << endl;
                    g_free (buf);
                }  else if (rc >= 300)
                    self->redirected = true;
            }
        } else if (self->redirected && !strncasecmp ((char *)ptr, "location:", 9)) {
            String nuri ((char *) ptr + 9, size * nmemb - 9);
            self->redirected = false;
            self->m_receiver->redirected (job, nuri.stripWhiteSpace ());
        } else if (!strncasecmp ((char *)ptr, "content-length:", 15)) {
            self->content_length = strtol ((char *)ptr + 15, NULL, 10);
        } else if (!strncasecmp ((char *)ptr, "content-type:", 13)) {
            self->content_type = String (
                    (char *)ptr + 13, size * nmemb - 13).stripWhiteSpace ();
            int pos = self->content_type.indexOf (Char (';'));
            if (pos > 0)
                self->content_type = self->content_type.left (pos);
        }
    }
    return size * nmemb;
}

size_t CurlGetJob::dataReceived (void *ptr, size_t size, size_t nmemb, void *stream) {
    IOJob *job = (IOJob *) stream;
    CurlGetJob *self = job->d;
    ByteArray data ((char *)ptr, size * nmemb);
    //debugLog() << "dataReceived " << (size * nmemb) << endl;
    if (!self->killed)
        self->m_receiver->jobData (job, data);
    return size * nmemb;
}

size_t CurlGetJob::writePost (void *p, size_t sz, size_t n, void *stream) {
    IOJob *job = (IOJob *) stream;
    CurlGetJob *self = job->d;
    int plen = self->post_data.size ();
    if (!plen)
        return 0;
    int count = plen;
    if (count > sz * n)
        count = sz * n;
    memcpy (p, self->post_data.data (), count);
    if (plen > count) {
        memmove (self->post_data.data(), self->post_data.data() + count, plen - count);
    }
    self->post_data.resize (plen - count);
    return count;
}

static int job_count;

KMPLAYER_NO_CDTOR_EXPORT
CurlGetJob::CurlGetJob (IOJobListener * rec, const String & u)
 : m_receiver (rec), url (u), headers (NULL), easy (NULL),
   quiet (false), killed (false), redirected (false),
   error_code (0), content_length (0)
{}

void CurlGetJob::init (IOJob *job) {
    if (!global_init) {
        global_init = !curl_global_init (CURL_GLOBAL_ALL);
        easy_list = g_tree_new (easyCompare);
        con_ic_connection = con_ic_connection_new ();
        g_signal_connect (G_OBJECT (con_ic_connection), "connection-event",
                G_CALLBACK (icConnection), NULL);
    }
    if (!multi)
        multi = curl_multi_init ();
    easy = curl_easy_init ();
    //debugLog() << "CurlGetJob::CurlGetJob " << easy << " " << ++job_count << endl;
    g_tree_insert (easy_list, easy, job);
    //curl_easy_setopt (easy, CURLOPT_VERBOSE, true);
    curl_easy_setopt (easy, CURLOPT_NOPROGRESS, true);
    curl_easy_setopt (easy, CURLOPT_WRITEFUNCTION, CurlGetJob::dataReceived);
    curl_easy_setopt (easy, CURLOPT_WRITEDATA, job);
    curl_easy_setopt (easy, CURLOPT_HEADERFUNCTION, CurlGetJob::headerLine);
    curl_easy_setopt (easy, CURLOPT_WRITEHEADER, job);
    if (post_data.size ()) {
        curl_easy_setopt (easy, CURLOPT_POST, 1);
        curl_easy_setopt (easy, CURLOPT_POSTFIELDSIZE, post_data.size ());
        curl_easy_setopt (easy, CURLOPT_READFUNCTION, CurlGetJob::writePost);
        curl_easy_setopt (easy, CURLOPT_READDATA, job);
    }
    curl_easy_setopt (easy, CURLOPT_FOLLOWLOCATION, 1);
    curl_easy_setopt (easy, CURLOPT_MAXREDIRS, 10);
    //CURLSH *curl_share_init( );
    //curl_easy_setopt (easy, CURLOPT_SHARE, m_curlShareHandle);
    //curl_share_setopt(share_handle,CURL_LOCK_DATA_DNS);
    //curl_share_cleanup(CURLSH * share_handle );
    curl_easy_setopt (easy, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V4);
    curl_easy_setopt (easy, CURLOPT_DNS_CACHE_TIMEOUT, 60 * 5);
    curl_easy_setopt (easy, CURLOPT_ENCODING, "");
    curl_easy_setopt (easy, CURLOPT_USERAGENT, "KMPlayer/" VERSION);
    curl_easy_setopt (easy, CURLOPT_SSL_VERIFYPEER, 0);
    curl_easy_setopt (easy, CURLOPT_URL, (const char *)url);
    if (headers)
        curl_easy_setopt (easy, CURLOPT_HTTPHEADER, headers);
    //if (!strncmp (u, "http", 4))
    //    curl_easy_setopt (easy, CURLOPT_HTTPGET, true);
    if (CON_IC_STATUS_CONNECTED == ic_status || URL (url).isLocalFile ()) {
        start ();
    } else {
        if (!ic_connect_request)
            ic_connect_request = con_ic_connection_connect (con_ic_connection,
                    CON_IC_CONNECT_FLAG_NONE);
        if (ic_connect_request)
            ic_connect_waiters = g_slist_prepend (ic_connect_waiters, this);
        else
            job->kill (false);
    }
}

void CurlGetJob::start () {
    CURLMcode type = curl_multi_add_handle (multi, easy);
    if (type && type != CURLM_CALL_MULTI_PERFORM)
        debugLog() << "CurlGetJob add to multi failure" << endl;
    else if (!ic_connect_waiters)
        curlPerformMulti ();
}

KMPLAYER_NO_CDTOR_EXPORT CurlGetJob::~CurlGetJob () {
    //debugLog() << "CurlGetJob::~CurlGetJob " << easy << " " << --job_count << endl;
    if (ic_connect_waiters)
        ic_connect_waiters = g_slist_remove (ic_connect_waiters, this);
    if (easy) {
        curl_multi_remove_handle (multi, easy);
        curl_easy_cleanup (easy);
        g_tree_remove (easy_list, easy);
    }
    if (headers)
        curl_slist_free_all (headers);
    if (0) {
        curl_multi_cleanup (multi);
        multi = NULL;
    }
}

KMPLAYER_NO_EXPORT IOJob::IOJob (IOJobPrivate *priv) : d (priv) {
}

KMPLAYER_NO_EXPORT IOJob::~IOJob () {
    IOJobListener *listener = d->easy && !d->quiet ? d->m_receiver : NULL;

    if (listener)
        listener->jobResult (this);

    delete d;
}

KMPLAYER_NO_EXPORT void IOJob::setHeader (const String &header) {
    d->headers = curl_slist_append (d->headers, (const char *) header);
}

KMPLAYER_NO_EXPORT void IOJob::setHttpPostData (const ByteArray &data) {
    d->post_data = data;
}

KMPLAYER_NO_EXPORT void IOJob::start () {
    d->init (this);
}

KMPLAYER_NO_EXPORT void IOJob::kill (bool q) {
    d->quiet = q;
    d->killed = true;
    curl_job_killed = true;
    if (!in_perform_multi) {
        delete this;
	curlPerformMulti ();
    }
}

KMPLAYER_NO_EXPORT bool IOJob::error () {
    return d->error_code;
}

unsigned long IOJob::contentLength() const {
    return d->content_length;
}

String IOJob::contentType () const {
    return d->content_type;
}

//-----------------------------------------------------------------------------

static bool gconf_initialized = false;

KMPLAYER_NO_CDTOR_EXPORT GConf::GConf () {
    if (!gconf_initialized) {
        gconf_initialized = true;
        g_type_init ();
    }
    client = gconf_client_get_default ();
}

KMPLAYER_NO_CDTOR_EXPORT GConf::~GConf () {
    gconf_client_clear_cache (client);
    g_object_unref (client);
}

KMPLAYER_NO_EXPORT
String GConf::readEntry (const String & key, const String & def) {
    gchar *val = gconf_client_get_string (client, (const char *) key, 0L);
    if (!val)
        return def;
    String s;
    s.take (val);
    return s;
}

KMPLAYER_NO_EXPORT int GConf::readNumEntry (const String & key, int def) {
    GConfValue *val = gconf_client_get (client, (const char *) key, 0L);
    if (val) {
        int i = gconf_value_get_int (val);
        gconf_value_free (val);
        return i;
    }
    return def;
}

KMPLAYER_NO_EXPORT double GConf::readNumEntry (const String & key, double def) {
    GError * err;
    double val = gconf_client_get_float (client, (const char *) key, &err);
    bool has_err = !!err;
    g_error_free (err);
    return has_err ? def : val;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_EXPORT Log & Log::operator << (const char * s) {
    fprintf (stderr, "%s", s);
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (int i) {
    fprintf (stderr, "%d", i);
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (unsigned int i) {
    fprintf (stderr, "%u", i);
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (double d) {
    fprintf (stderr, "%f", d);
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (const Single &s) {
    fprintf (stderr, "%.3f", (double)s);
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (const SRect &r) {
    fprintf (stderr, "SRect(%.1f, %.1f, %.1f, %.1f)",
           (double)r.x(), (double)r.y(), (double)r.width(), (double)r.height());
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (const IRect &r) {
    fprintf (stderr, "SRect(%d, %d, %d, %d)",
            r.x (), r.y (), r.width (), r.height ());
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (void * d) {
    fprintf (stderr, "0x%x", d);
    return *this;
}

//-----------------------------------------------------------------------------

#include <glib/gprintf.h>

static GdkColor * colorFromRGB (int rgb) {
    GdkColormap * cmap = gdk_colormap_get_system ();
    GdkColor *color = g_new (GdkColor, 1);
    GdkColor *rcolor = NULL;
    char buf [16];
    g_sprintf (buf, "#%02X%02X%02X", (rgb & 0x00FF0000) >> 16, (rgb & 0x0000FF00) >> 8, (rgb & 0x000000FF));
    if (gdk_color_parse (buf, color) &&
                (gdk_colormap_alloc_color (cmap, color, 0, 0) ||
                 gdk_colormap_alloc_color (cmap, color, 0, 1)))
        rcolor = gdk_color_copy (color);
    g_free (color);
    return rcolor;
}

KMPLAYER_NO_CDTOR_EXPORT GColor::GColor (const String & desc) : m_color (NULL) {
    GdkColormap * cmap = gdk_colormap_get_system ();
    GdkColor *color = g_new (GdkColor, 1);
    if (!(gdk_color_parse ((const char *) desc, color) &&
                gdk_colormap_alloc_color (cmap, color, 0, 1))) {
        if (desc.lower () == "silver")
            m_color = colorFromRGB (0xbebebe);
        // TODO add more unsupported real live colors
        if (!m_color)
            errorLog () << "Could not parse " << desc << " color" << endl;
    } else {
        m_color = gdk_color_copy (color);
    }
    g_free (color);
    //debugLog () << desc << " -> " << int (m_color ? m_color->pixel : 0) << endl;
}

KMPLAYER_NO_CDTOR_EXPORT GColor::GColor (int rgb)
    : m_color (colorFromRGB(rgb)) {
    //debugLog () << "GColor::GColor (int " << rgb << ") -> " << int (m_color ? m_color->pixel : -1) << " str:" << buf << endl;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::GPixbuf (int w, int h) {
    GdkVisual * visual = gdk_visual_get_system ();//gdk_rgb_get_visual
    m_pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, false, visual->depth, w, h);
}

KMPLAYER_NO_CDTOR_EXPORT
GPixbuf::GPixbuf (const ByteArray & data) : m_pixbuf (0L) {
    GdkPixbufLoader * pix_loader = gdk_pixbuf_loader_new ();
    GError *gerror = 0L;
    if (gdk_pixbuf_loader_write (pix_loader, (const guchar *) data.data (), data.size (), &gerror) && gdk_pixbuf_loader_close (pix_loader, 0L)) {
        m_pixbuf = gdk_pixbuf_loader_get_pixbuf (pix_loader);
        g_object_ref (m_pixbuf);
    } else
        g_free (gerror);
    g_object_unref (pix_loader);
}

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::GPixbuf (GdkPixbuf * pb, bool skip_addref)
  : m_pixbuf (pb) {
    if (!skip_addref && m_pixbuf)
        g_object_ref (m_pixbuf);
}

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::GPixbuf (const GPixbuf & other)
 : m_pixbuf (other.m_pixbuf) {
    if (m_pixbuf)
        g_object_ref (m_pixbuf);
}

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::~GPixbuf () {
    reset ();
}

KMPLAYER_NO_EXPORT GPixbuf & GPixbuf::operator = (const GPixbuf & other) {
    if (&other != this) {
        reset ();
        m_pixbuf = other.m_pixbuf;
        if (m_pixbuf)
            g_object_ref (m_pixbuf);
    }
    return *this;
}

KMPLAYER_NO_EXPORT void GPixbuf::reset () {
    if (m_pixbuf) {
        g_object_unref (m_pixbuf);
        m_pixbuf = 0L;
    }
}

KMPLAYER_NO_EXPORT void GPixbuf::fill (const Color & c) {
    if (m_pixbuf)
        gdk_pixbuf_fill (m_pixbuf, c.rgb ());
}

KMPLAYER_NO_EXPORT GPixbuf GPixbuf::scale (int w, int h) {
    if (m_pixbuf)
        return GPixbuf (gdk_pixbuf_scale_simple (m_pixbuf, w, h, w > width () ? GDK_INTERP_NEAREST : GDK_INTERP_BILINEAR), true);
}


/*
void        gdk_pixbuf_xlib_init            (Display *display,
                                             int screen_num);


GdkPixbuf*  gdk_pixbuf_new_from_file        (const char *filename,
                                             GError **error);
// gdk_pixbuf_new_from_data gdk_pixbuf_new_from_xpm_data
void        gdk_pixbuf_copy_area            (const GdkPixbuf *src_pixbuf,
                                             int src_x,
                                             int src_y,
                                             int width,
                                             int height,
                                             GdkPixbuf *dest_pixbuf,
                                             int dest_x,
                                             int dest_y);

void        gdk_pixbuf_scale                (const GdkPixbuf *src,
                                             GdkPixbuf *dest,
                                             int dest_x,
                                             int dest_y,
                                             int dest_width,
                                             int dest_height,
                                             double offset_x,
                                             double offset_y,
                                             double scale_x,
                                             double scale_y,
                                             GdkInterpType interp_type);

g_object_unref() 

// http://developer.gnome.org/doc/API/2.0/gdk-pixbuf/gdk-pixbuf-animation.html

GdkPixbuf*  gdk_pixbuf_add_alpha            (const GdkPixbuf *pixbuf,
                                             gboolean substitute_color,
                                             guchar r,
                                             guchar g,
                                             guchar b);


typedef struct {
  gint x;
  gint y;
  gint width;
  gint height;
} GdkRectangle;

void        gdk_rectangle_union             (GdkRectangle *src1,
                                             GdkRectangle *src2,
                                             GdkRectangle *dest);

*/
