/* This file is part of the KMPlayer project
 *
 * Copyright (C) 2010 Koos Vriezen <koos.vriezen@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * until boost gets common, a more or less compatable one ..
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <gtk/gtkmain.h>

#include "kmplayerstring.h"

using namespace KMPlayer;


KMPLAYER_NO_CDTOR_EXPORT ByteArrayImpl::ByteArrayImpl (const ByteArrayImpl & b)
 : m_data (b.m_size ? (char *) malloc (b.m_size + 1) : 0L),
   m_size (b.m_size),
   m_capacity (b.m_size),
   data_owner (true) {
    if (m_size) {
        memcpy (m_data, b.m_data, m_size);
        m_data[m_size] = 0;
    }
}

KMPLAYER_NO_EXPORT void ByteArrayImpl::resize (unsigned ns) {
    if (ns > m_capacity) {
        char * tmp = (char *) malloc (ns + 1);
        if (m_size > 0)
            memcpy (tmp, m_data, m_size);
        m_capacity = m_size = ns;
        if (m_data && data_owner)
            free (m_data);
        m_data = tmp;
        m_data[ns] = 0;
        data_owner = true;
    } else
        m_size = ns;
}

KMPLAYER_NO_EXPORT
ByteArrayImpl & ByteArrayImpl::operator = (const ByteArrayImpl & b) {
    if (m_data && data_owner)
        free (m_data);
    m_data = (b.m_size ? (char*) malloc (b.m_size + 1) : 0L);
    m_size = b.m_size;
    m_capacity = b.m_size;
    data_owner = true;
    if (m_size) {
        memcpy (m_data, b.m_data, m_size);
        m_data[m_size] = 0;
    }
}

KMPLAYER_NO_EXPORT ByteArrayImpl &ByteArrayImpl::operator = (const char *b) {
    int len = strlen (b);
    resize (len);
    memcpy (m_data, b, len);
    return *this;
}

//-----------------------------------------------------------------------------

void *FixedSizeAllocator::alloc () {
    return count ? pool[--count] : malloc (size);
}

void FixedSizeAllocator::dealloc (void *p) {
    if (count < 10)
        pool[count++] = p;
    else
        free (p);
}

//-----------------------------------------------------------------------------

static FixedSizeAllocator *shared_data_pool;

namespace KMPlayer {

void *sharedDataAlloc (size_t s) {
    if (!shared_data_pool)
        shared_data_pool = new FixedSizeAllocator (s);

    return shared_data_pool->alloc ();
}

void sharedDataFree (void *p) {
    shared_data_pool->dealloc (p);
}

}

//-----------------------------------------------------------------------------

static FixedSizeAllocator *shared_chars_pool;

void *SharedChars::operator new (size_t s) {
    if (!shared_chars_pool)
        shared_chars_pool = new FixedSizeAllocator (sizeof (SharedChars));

    return shared_chars_pool->alloc ();
}

void SharedChars::operator delete (void *p) {
    shared_chars_pool->dealloc (p);
}

template <>
void Shared<SharedChars>::attach (const Shared<SharedChars> &other) {
    if (data != other.data) {
        if (other.data)
            other.data->ref ();
        if (data)
            data->unref ();
        data = other.data;
    }
}

GLibString::GLibString (const Char & ch) {
    gchar buf [8];
    int nr = g_unichar_to_utf8 (ch.unicode (), buf);
    buf [nr] = 0;
    takeUtf8 (g_strdup (buf), nr);
}

void GLibString::takeUtf8 (char *p, int buf_len) {
    if (!data || data->ptr != p) {
        if (data && 1 == data->ref_count) {
            g_free (data->ptr);
            data->ptr = p;
            data->buf_len = buf_len;
        } else {
            SharedChars *tmp = p ? new SharedChars (p, buf_len) : 0L;
            if (data)
                data->unref ();
            data = tmp;
        }
    }
}

int GLibString::compare (const char *str) const {
    if (!str && !data)
        return 0;
    if (str && data)
        return strcmp (data->ptr, str);
    return data ? 1 : -1;
}

int GLibString::compare (const GLibString & str) const {
    if (str.data == data)
        return 0;
    if (str.data && data)
        return strcmp (data->ptr, str.data->ptr);
    return data ? 1 : -1;
}

void GLibString::take (char * s) {
    const gchar *end;
    if (g_utf8_validate (s, -1, &end)) {
        takeUtf8 (s, end - s);
    } else {
        gsize bytes_read, bytes_written;
        takeUtf8 (g_locale_to_utf8 (s, -1, &bytes_read, &bytes_written, 0L));
    }
}

GLibString & GLibString::operator = (const char * s) {
    if (!s)
        clear ();
    else
        take (g_strdup (s));
    return *this;
}

GLibString & GLibString::operator = (const Char & ch) {
    gchar buf [8];
    int nr = g_unichar_to_utf8 (ch.unicode (), buf);
    buf [nr] = 0;
    takeUtf8 (g_strdup (buf), nr);
    return *this;
}

static char *strCat (const gchar *p1, int len1, const gchar *p2, int len2) {
    char *p = (char *) g_malloc (len1 + len2 + 1);
    memcpy (p, p1, len1);
    memcpy (p + len1, p2, len2);
    p [len1 + len2] = 0;
    return p;
}

void GLibString::append (const char *p, int length) {
    if (!data) {
        takeUtf8 (g_strndup (p, length), length);
    } else {
        if (data->buf_len < 0)
            data->buf_len = strlen (data->ptr);
        if (1 == data->ref_count) {
            data->ptr = (gchar *) g_realloc (data->ptr,
                    data->buf_len + length + 1);
            memcpy (data->ptr + data->buf_len, p, length);
            data->buf_len += length;
            data->ptr [data->buf_len] = 0;
        } else {
            takeUtf8 (strCat (data->ptr, data->buf_len, p, length),
                    data->buf_len + length);
        }
    }
}

GLibString & GLibString::operator += (const GLibString &str) {
    if (str.data) {
        if (!data) {
            attach (str);
        } else {
            int slen = str.data->buf_len;
            if (slen < 0)
                slen = strlen (str.data->ptr);
            append (str.data->ptr, slen);
        }
    }
    return *this;
}

KMPLAYER_NO_EXPORT GLibString &GLibString::operator += (const char *str) {
    append (str, strlen (str));
    return *this;
}

KMPLAYER_NO_EXPORT GLibString &GLibString::operator += (const Char &ch) {
    gchar buf [8];
    int nr = g_unichar_to_utf8 (ch.unicode (), buf);
    buf [nr] = 0;
    append (buf, nr);
    return *this;
}

GLibString GLibString::fill (Char ch, int length) {
    GLibString str;
    str.take (g_strnfill (length, (const char) ch)); //FIME
    return str;
}

GLibString GLibString::stripWhiteSpace () const {
    String str;
    if (data && data->ptr [0]) {
        gchar *iter = data->ptr;
        gchar *first = iter;
        gchar *end = iter;
        Char ch (g_utf8_get_char (iter));
        iter = g_utf8_next_char (iter);
        while (ch.isSpace () && iter && *iter) { // skip leading ws
            ch = Char (g_utf8_get_char (iter));
            first = iter;
            iter = g_utf8_next_char (iter);
        }
        if (ch.isSpace ())
            return str;
        end = first + 1;
        do {
            if (!iter || !*iter)
                break;
            ch = Char (g_utf8_get_char (iter));
            iter = g_utf8_next_char (iter);
            while (ch.isSpace () && iter && *iter) {
                ch = Char (g_utf8_get_char (iter));
                iter = g_utf8_next_char (iter);
            }
            if (ch.isSpace ())
                break;  // skip ending ws
            end = iter;
        } while (iter && *iter);
        if (end > first)
            str.append (first, end - first);
    }
    return str;
}

bool GLibString::startsWith (const GLibString & prefix) const {
    if (!prefix.data && !data)
        return true;
    if (prefix.data && data)
        return g_str_has_prefix (data->ptr, prefix.data->ptr);
    return !prefix.data;
}

bool GLibString::endsWith (const GLibString & suffix) const {
    if (!suffix.data && !data)
        return true;
    if (suffix.data && data)
        return g_str_has_suffix (data->ptr, suffix.data->ptr);
    return !suffix.data;
}

int GLibString::indexOf (const GLibString &needle, int offset) const {
    if (!data)
        return needle.data ? -1 : 0;
    if (!needle.data)
        return 0;
    char *p = data->ptr;
    if (offset > -1) {
        if (data->buf_len < 0)
            data->buf_len = strlen (data->ptr);
        if (offset >= data->buf_len)
            return -1;
        p = g_utf8_offset_to_pointer (data->ptr, offset);
        if (!p)
            return -1;
    }
    gchar * str = strstr (p, needle.data->ptr);
    if (!str)
        return -1;
    return g_utf8_strlen (p, str - p) + (offset > -1 ? offset : 0);
}

int GLibString::lastIndexOf (const GLibString & needle) const {
    if (!data)
        return needle.data ? -1 : 0;
    if (!needle.data)
        return length ();
    gchar * str = g_strrstr (data->ptr, needle.data->ptr);
    if (!str)
        return -1;
    return g_utf8_strlen (data->ptr, str - data->ptr);
}

GLibString GLibString::replace (Char c1, Char c2) const {
    GLibString str;
    if (data) {
        gchar cstr1 [8];
        int nr1 = g_unichar_to_utf8 (c1.unicode (), cstr1);
        cstr1 [nr1] = 0;
        gchar cstr2 [8];
        int nr2 = g_unichar_to_utf8 (c2.unicode (), cstr2);
        cstr2 [nr2] = 0;
        gchar *s = data->ptr;
        gchar *p = strstr (s, cstr1);
        while (p) {
            if (p != s)
                str.append (s, p - s);
            str.append (cstr2, nr2);
            s = p + nr1;
            p = strstr (s, cstr1);
        }
        if (s[0])
            str.append (s, strlen (s));
    }
    return str;
}

int GLibString::toInt (bool * valid, int base) const {
    int i = 0;
    if (data) {
        char *endptr;
        i = (int) strtol (data->ptr, &endptr, base);
        if (valid)
            *valid = (data->ptr != endptr);
    } else if (valid)
        *valid = false;
    return i;
}

KMPLAYER_NO_EXPORT double GLibString::toDouble (bool * valid) const {
    if (data) {
        gchar * endptr;
        double d = g_ascii_strtod (data->ptr, &endptr);
        if (valid)
            *valid = (endptr != data->ptr);
        return d;
    } else {
        if (valid)
            *valid = false;
        return 0.0;
    }
}

GLibString GLibString::mid (const int index, unsigned len) const {
    if (!data)
        return GLibString ();
    gchar * off = g_utf8_offset_to_pointer (data->ptr, index);
    gchar * eos = off;
    if (!off)
        return GLibString ();
    while (eos && *eos && len--)
        eos = g_utf8_next_char (eos);
    GLibString str;
    if (!eos || !*eos)
        str.takeUtf8 (g_strdup (off));
    else
        str.takeUtf8 (g_strndup (off, eos-off), eos - off);
    return str;
}

GLibString GLibString::left (int index) const {
    if (!data)
        return GLibString ();
    if (index < 0)
        return GLibString (*this);
    gchar * off = g_utf8_offset_to_pointer (data->ptr, index);
    if (!off)
        return GLibString ();
    GLibString str;
    str.takeUtf8 (g_strndup (data->ptr, off - data->ptr));
    return str;
}

void GLibString::truncate (int pos) {
    if (data) {
        if (pos == 0) {  // speed up clearing string
            clear ();
        } else {
            gchar *epos = g_utf8_offset_to_pointer (data->ptr, pos);
            if (epos)
                takeUtf8 (g_strndup (data->ptr, epos - data->ptr),
                          epos - data->ptr);
        }
    }
}

