/* This file is part of the KMPlayer project
 *
 * Copyright (C) 2010 Koos Vriezen <koos.vriezen@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * until boost gets common, a more or less compatable one ..
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <gtk/gtkmain.h>
#include <gio/gio.h>

#include "io.h"
#include "log.h"

using namespace KMPlayer;


namespace KMPlayer {

class GIOFilePrivate {
public:
    GIOFilePrivate () : file (NULL), fin (NULL), fout (NULL) {}
    ~GIOFilePrivate () {}
    GFile *file;
    GInputStream *fin;
    GOutputStream *fout;
    String m_path;
};

}

KMPLAYER_NO_CDTOR_EXPORT
GIOFile::GIOFile (const String & p) : d (new GIOFilePrivate) {
    d->m_path = p;
    d->file = g_file_new_for_path ((const char *) d->m_path);
}

KMPLAYER_NO_CDTOR_EXPORT GIOFile::~GIOFile () {
    close ();
    g_object_unref (G_OBJECT (d->file));
    d->file = NULL;
    delete d;
}

KMPLAYER_NO_EXPORT bool GIOFile::open (FileMode mode) {
    switch (mode) {
        case IO_ReadOnly:
            d->fin = G_INPUT_STREAM (g_file_read (d->file, NULL, NULL));
            if (d->fin)
                return true;
            break;
        case IO_WriteOnly: // fixme truncate
            d->fout = G_OUTPUT_STREAM (g_file_replace (d->file, NULL, false,
                    G_FILE_CREATE_NONE, NULL, NULL));
            if (d->fout)
                return true;
            break;
        default:
            warningLog() << "open " << int (mode) << " not implemented" << endl;
    }
    return false;
}

KMPLAYER_NO_EXPORT void GIOFile::close () {
    if (d->fin) {
        g_object_unref (G_OBJECT (d->fin));
        d->fin = NULL;
    }
    if (d->fout) {
        g_object_unref (G_OBJECT (d->fout));
        d->fout = NULL;
    }
}

KMPLAYER_NO_EXPORT ByteArray GIOFile::readAll () {
    ByteArray str;
    char buf [2048];
    gsize nr;
    if (d->fin)
        close (); // seek 0
    if (!open (IO_ReadOnly))
        return str;
    do {
        nr = g_input_stream_read (d->fin, buf, sizeof (buf), NULL, NULL);
        if (nr <= 0)
            break;
        off_t size = str.size ();
        str.resize (size + nr);
        memcpy (str.data () + size, buf, nr);
    } while (true);
    return str;
}

KMPLAYER_NO_EXPORT off_t GIOFile::readBlock (char * buf, off_t max) {
    if (!d->fin && !open (IO_ReadOnly))
        return 0;
    gsize nr = g_input_stream_read (d->fin, buf, max, NULL, NULL);
    return nr < 0 ? 0 : nr;
}

KMPLAYER_NO_EXPORT bool GIOFile::exists () {
    return g_file_query_exists (d->file, NULL);
}

KMPLAYER_NO_EXPORT off_t GIOFile::size () {
    goffset sz = 0;
    GFileInfo *info = g_file_query_info (d->file,
            G_FILE_ATTRIBUTE_STANDARD_SIZE,
            G_FILE_QUERY_INFO_NONE, NULL, NULL);
    if (info) {
        sz = g_file_info_get_size (info);
        g_object_unref (info);
    }
    return sz;
}

KMPLAYER_NO_EXPORT off_t GIOFile::writeBlock (const char * data, off_t length) {
    gsize nr;
    g_output_stream_write_all (d->fout, data, length, &nr, NULL, NULL);
    return nr;
}

//-----------------------------------------------------------------------------

GioUrl::GioUrl (const GioUrl &base, const String &rel) {
    GFile *file = g_file_new_for_uri ((const char *) rel);
    gchar *scheme = g_file_get_uri_scheme (file);
    if (base.m_file && (!scheme || !scheme[0])) {
        GFile *p = g_file_get_parent (base.m_file);
        m_file = g_file_resolve_relative_path (p ? p : base.m_file, rel);
        if (p)
            g_object_unref (p);
    } else {
        m_file = file;
    }
    g_free (scheme);
}

KMPLAYER_NO_CDTOR_EXPORT GioUrl::GioUrl (const String &url) : m_file (NULL) {
    if (url.startsWith (Char ('/')))
        m_file = g_file_new_for_path (url);
    else if (!url.isEmpty ())
        m_file = g_file_new_for_uri ((const char *) url);
}

GioUrl::GioUrl (const File &file)
{
    m_file = g_file_dup (file.d->file);
}

KMPLAYER_NO_CDTOR_EXPORT GioUrl::GioUrl (const GioUrl & url) {
    m_file = (url.m_file ? g_file_dup (url.m_file) : 0L);
}

KMPLAYER_NO_CDTOR_EXPORT GioUrl::~GioUrl () {
    if (m_file)
        g_object_unref (m_file);
}

KMPLAYER_NO_EXPORT String GioUrl::url () const {
    if (m_file) {
        GLibString s;
        s.take (g_file_get_uri (m_file));
        return String (s);
    }
    return String ();
}

GioUrl & GioUrl::operator = (const String &url) {
    if (m_file)
        g_object_unref (m_file);
    m_file = g_file_new_for_uri ((const char *) url);
    return *this;
}

GioUrl & GioUrl::operator = (const GioUrl & url) {
    if (m_file)
        g_object_unref (m_file);
    m_file = url.m_file ? g_file_dup (url.m_file) : 0L;
    return *this;
}

KMPLAYER_NO_EXPORT String GioUrl::path () const {
    String str;
    if (m_file) {
        char *p = g_file_get_path (m_file);
        if (p)
            str.take (p);
    }
    return str;
}

KMPLAYER_NO_EXPORT String GioUrl::protocol () const {
    String str;
    if (m_file) {
        char *scheme = g_file_get_uri_scheme (m_file);
        if (scheme)
            str.take (scheme);
    }
    return str;
}

KMPLAYER_NO_EXPORT String GioUrl::decode_string (const String & s) {
    if (s.isEmpty ())
        return String ();
    String str;
    GLibString &gs = str;
    gs.take (g_uri_unescape_string ((const char *) s, ""));
    return str;
}

KMPLAYER_NO_EXPORT String GioUrl::encode_string (const String & s) {
    if (s.isEmpty ())
        return String ();
    String str;
    GLibString &gs = str;
    gs.take(g_uri_escape_string ((const char *) s, "", TRUE));
    return str;
}

KMPLAYER_NO_EXPORT bool GioUrl::isLocalFile () const {
    if (m_file)
        return g_file_is_native (m_file) ||
            protocol () == "file";
    return false;
}

//-----------------------------------------------------------------------------

#include <conic.h>
#include <curl/curl.h>

static gint easyCompare (gconstpointer a, gconstpointer b) {
    return (long)a - (long)b;
}

namespace KMPlayer {
    class CurlGetJob {
    public:
        CurlGetJob (IOJobListener * rec, const String & u);
        ~CurlGetJob ();

        void init (IOJob *job, off_t pos);
        void start ();

        static size_t headerLine( void *p, size_t sz, size_t n, void *strm);
        static size_t dataReceived (void *p, size_t sz, size_t n, void *strm);
        static size_t writePost (void *p, size_t sz, size_t n, void *strm);
        static gboolean findKilledJob (gpointer key, gpointer val, gpointer d);

        IOJobListener * m_receiver;

        static bool global_init;
        static CURLM *multi;

        String url;
        String content_type;
        ByteArray post_data;
        curl_slist *headers;
        CURL *easy;
        bool quiet;
        bool killed;
        bool redirected;
        int error_code;
        unsigned long content_length;
    };
}

namespace {

    struct FdWatch {
        int fd;
        int chan_read_watch;
        int chan_write_watch;
    };

}

namespace KMPlayer {

    IOJob *asyncGet (IOJobListener *receiver, const String &url) {
        return new IOJob (new CurlGetJob (receiver, url));
    }
}

inline gboolean
CurlGetJob::findKilledJob (gpointer key, gpointer value, gpointer data) {
    IOJob *job = (IOJob *)value;
    CurlGetJob *self = job->d;

    if (self->killed) {
        *(IOJob **)data = job;
        return true;
    }
    return false;
}

static gboolean curlChannelRead (GIOChannel *src, GIOCondition c, gpointer d);
static gboolean curlChannelWrite (GIOChannel *src, GIOCondition c, gpointer d);
static gboolean curlTimeOut (void *);
static void curlPerformMulti ();
bool CurlGetJob::global_init = false;
CURLM *CurlGetJob::multi = NULL;
static GTree *easy_list = NULL;
static FdWatch *fd_watch_array;
static int fd_watch_array_len;
static gint curl_timeout_timer;
static bool in_perform_multi;
static bool curl_job_killed;
static ConIcConnection *con_ic_connection;
#ifdef __ARMEL__
static ConIcConnectionStatus ic_status = CON_IC_STATUS_DISCONNECTED;
#else
static ConIcConnectionStatus ic_status = CON_IC_STATUS_CONNECTED;
#endif
static GSList *ic_connect_waiters;
static bool ic_connect_request;

static
void icConnection (ConIcConnection *conn, ConIcConnectionEvent *ev, gpointer) {
    ic_connect_request = false;
    ic_status = con_ic_connection_event_get_status (ev);
    if (CON_IC_STATUS_CONNECTED == ic_status && ic_connect_waiters) {
        ic_connect_waiters = g_slist_reverse (ic_connect_waiters);
        for (GSList *sl = ic_connect_waiters; sl; sl = sl->next)
            ((CurlGetJob *)sl->data)->start ();
        g_slist_free (ic_connect_waiters);
        ic_connect_waiters = NULL;
        curlPerformMulti ();
    } else if (CON_IC_STATUS_DISCONNECTED == ic_status) {
        ConIcConnectionError err = con_ic_connection_event_get_error (ev);
        if (CON_IC_CONNECTION_ERROR_NONE != err)
            for (GSList *sl = ic_connect_waiters; sl; sl = sl->next) {
                CurlGetJob *curl_job = (CurlGetJob *)ic_connect_waiters->data;
                curl_job->error_code = 1;
                curl_job->killed = true;
            }
        curlPerformMulti ();
    }
}

static void curlChannelDestroy (gpointer) {
#ifdef DEBUG_TIMERS
    debugLog() << "curlChannelDestroy" << endl;
#endif
}

static void updateReadWatches() {
    fd_set fd_read, fd_write, fd_excep;
    FD_ZERO (&fd_read);
    FD_ZERO (&fd_write);
    FD_ZERO (&fd_excep);
    int maxfd = 0;
    curl_multi_fdset (CurlGetJob::multi, &fd_read, &fd_write, &fd_excep, &maxfd);
    ++maxfd;
    if (maxfd > 0 && !fd_watch_array) {
        fd_watch_array = new FdWatch[maxfd];
        memset (fd_watch_array, 0, maxfd * sizeof (FdWatch));
        fd_watch_array_len = maxfd;
    } else if (maxfd > fd_watch_array_len) {
        FdWatch *tmp = fd_watch_array;
        fd_watch_array = new FdWatch[maxfd];
        memset (fd_watch_array, 0, maxfd * sizeof (FdWatch));
        memcpy (fd_watch_array, tmp, fd_watch_array_len * sizeof (FdWatch));
        delete[] tmp;
        fd_watch_array_len = maxfd;
    }
    for (int i = 3; i < fd_watch_array_len; i++) {
        if (FD_ISSET (i, &fd_read)) {
            //debugLog () << "update watches for read fd=" << i << endl;
            if (!fd_watch_array[i].chan_read_watch) {
                fd_watch_array[i].fd = i;
                GIOChannel *channel = g_io_channel_unix_new (i);
                g_io_channel_set_encoding (channel, NULL, NULL);
                fd_watch_array[i].chan_read_watch = g_io_add_watch_full (channel, G_PRIORITY_DEFAULT, G_IO_IN, curlChannelRead, NULL, curlChannelDestroy);
                g_io_channel_unref (channel);
            }
        } else if (fd_watch_array[i].chan_read_watch) {
            g_source_remove (fd_watch_array[i].chan_read_watch);
            fd_watch_array[i].chan_read_watch = 0;
        }
        if (!fd_watch_array[i].chan_read_watch && FD_ISSET (i, &fd_write)) {
            // connecting ..
            //debugLog () << "update watches for write fd=" << i << endl;
            if (!fd_watch_array[i].chan_write_watch) {
                fd_watch_array[i].fd = i;
                GIOChannel *channel = g_io_channel_unix_new (i);
                g_io_channel_set_encoding (channel, NULL, NULL);
                fd_watch_array[i].chan_write_watch = g_io_add_watch_full (channel, G_PRIORITY_DEFAULT, G_IO_OUT, curlChannelWrite, NULL, curlChannelDestroy);
                g_io_channel_unref (channel);
            }
        } else if (fd_watch_array[i].chan_write_watch) {
            g_source_remove (fd_watch_array[i].chan_write_watch);
            fd_watch_array[i].chan_write_watch = 0;
        }
    }
}

static void curlPerformMulti () {

    in_perform_multi = true;

    int running_handles = 0;
    CURLMcode curlCode;

    do {
        curl_job_killed = false;
        curlCode = curl_multi_perform (CurlGetJob::multi, &running_handles);
	if (curl_job_killed)
            do {
                IOJob *job = NULL;
                g_tree_traverse (easy_list, CurlGetJob::findKilledJob, G_IN_ORDER, &job);
                if (job)
                    delete job;
                else
                    break;
            } while (true);
    } while (CURLM_CALL_MULTI_PERFORM == curlCode && running_handles > 0);

    int messages;
    CURLMsg* msg = curl_multi_info_read (CurlGetJob::multi, &messages);
    while (msg) {
        if (CURLMSG_DONE == msg->msg) {
            IOJob *job = (IOJob *)g_tree_lookup (easy_list, msg->easy_handle);
            if (job) {
                if (CURLE_OK != msg->data.result &&
                        CURLE_PARTIAL_FILE != msg->data.result) {
                    debugLog() << "curl error " << msg->data.result << endl;
                    job->setError ();
                }
                delete job;
            }
        }
        msg = curl_multi_info_read (CurlGetJob::multi, &messages);
    }
    updateReadWatches();

    in_perform_multi = false;

    if (!curl_timeout_timer && g_tree_height (easy_list))
        curl_timeout_timer = g_timeout_add (250, curlTimeOut, NULL);
}

static gboolean curlTimeOut (void *) {
#ifdef DEBUG_TIMERS
    debugLog() << "curlTimeOut" << endl;
#endif
    curlPerformMulti ();
    if (g_tree_height (easy_list))
        return 1; // continue
    curl_timeout_timer = 0;
    debugLog() << "clearing polling" << endl;
    return 0; // stop
}

static gboolean curlChannelRead (GIOChannel *src, GIOCondition c, gpointer d) {
#ifdef DEBUG_TIMERS
    debugLog() << "curlChannelRead" << endl;
#endif
    curlPerformMulti ();
}

static gboolean curlChannelWrite (GIOChannel *src, GIOCondition c, gpointer d) {
#ifdef DEBUG_TIMERS
    debugLog() << "curl connected" << endl;
#endif
    curlPerformMulti ();
}

size_t CurlGetJob::headerLine( void *ptr, size_t size, size_t nmemb, void *stream) {
    IOJob *job = (IOJob *) stream;
    CurlGetJob *self = job->d;
    //char buf[256];
    //snprintf (buf, sizeof (buf), "%s", (char *)ptr);
    //debugLog() << "header: " << buf << endl;
    // Last-Modified:
    if (!self->killed) {
        if (!strncmp ((char *)ptr, "HTTP/", 5)) {
            char *p = strchr ((char *)ptr + 5, ' ');
            if (p) {
                int rc = strtol (p +1, NULL, 10);
                if (rc >= 400) {
                    self->error_code = rc;
                    gchar *buf = g_strndup (p + 1, size * nmemb);
                    debugLog() << "CurlGetJob err: " << buf << endl;
                    g_free (buf);
                }  else if (rc >= 300)
                    self->redirected = true;
            }
        } else if (self->redirected && !strncasecmp ((char *)ptr, "location:", 9)) {
            String nuri ((char *) ptr + 9, size * nmemb - 9);
            self->redirected = false;
            self->m_receiver->redirected (job, nuri.stripWhiteSpace ());
        } else if (!strncasecmp ((char *)ptr, "content-length:", 15)) {
            self->content_length = strtol ((char *)ptr + 15, NULL, 10);
        } else if (!strncasecmp ((char *)ptr, "content-type:", 13)) {
            self->content_type = String (
                    (char *)ptr + 13, size * nmemb - 13).stripWhiteSpace ();
            int pos = self->content_type.indexOf (Char (';'));
            if (pos > 0)
                self->content_type = self->content_type.left (pos);
        }
    }
    return size * nmemb;
}

size_t CurlGetJob::dataReceived (void *ptr, size_t size, size_t nmemb, void *stream) {
    IOJob *job = (IOJob *) stream;
    CurlGetJob *self = job->d;
    ByteArray data ((char *)ptr, size * nmemb);
    //debugLog() << "dataReceived " << (size * nmemb) << endl;
    if (!self->killed)
        self->m_receiver->jobData (job, data);
    return size * nmemb;
}

size_t CurlGetJob::writePost (void *p, size_t sz, size_t n, void *stream) {
    IOJob *job = (IOJob *) stream;
    CurlGetJob *self = job->d;
    int plen = self->post_data.size ();
    if (!plen)
        return 0;
    int count = plen;
    if (count > sz * n)
        count = sz * n;
    memcpy (p, self->post_data.data (), count);
    if (plen > count) {
        memmove (self->post_data.data(), self->post_data.data() + count, plen - count);
    }
    self->post_data.resize (plen - count);
    return count;
}

static int job_count;

KMPLAYER_NO_CDTOR_EXPORT
CurlGetJob::CurlGetJob (IOJobListener * rec, const String & u)
 : m_receiver (rec), url (u), headers (NULL), easy (NULL),
   quiet (false), killed (false), redirected (false),
   error_code (0), content_length (0)
{}

void CurlGetJob::init (IOJob *job, off_t pos) {
    if (!global_init) {
        global_init = !curl_global_init (CURL_GLOBAL_ALL);
        easy_list = g_tree_new (easyCompare);
    }
    if (!multi)
        multi = curl_multi_init ();
    easy = curl_easy_init ();
    //debugLog() << "CurlGetJob::CurlGetJob " << easy << " " << ++job_count << endl;
    g_tree_insert (easy_list, easy, job);
    //curl_easy_setopt (easy, CURLOPT_VERBOSE, true);
    curl_easy_setopt (easy, CURLOPT_NOPROGRESS, true);
    curl_easy_setopt (easy, CURLOPT_WRITEFUNCTION, CurlGetJob::dataReceived);
    curl_easy_setopt (easy, CURLOPT_WRITEDATA, job);
    curl_easy_setopt (easy, CURLOPT_HEADERFUNCTION, CurlGetJob::headerLine);
    curl_easy_setopt (easy, CURLOPT_WRITEHEADER, job);
    if (pos)
        curl_easy_setopt (easy, CURLOPT_RESUME_FROM_LARGE, (unsigned long long) pos);
    if (post_data.size ()) {
        curl_easy_setopt (easy, CURLOPT_POST, 1);
        curl_easy_setopt (easy, CURLOPT_POSTFIELDSIZE, post_data.size ());
        curl_easy_setopt (easy, CURLOPT_READFUNCTION, CurlGetJob::writePost);
        curl_easy_setopt (easy, CURLOPT_READDATA, job);
    }
    curl_easy_setopt (easy, CURLOPT_FOLLOWLOCATION, 1);
    curl_easy_setopt (easy, CURLOPT_MAXREDIRS, 10);
    //CURLSH *curl_share_init( );
    //curl_easy_setopt (easy, CURLOPT_SHARE, m_curlShareHandle);
    //curl_share_setopt(share_handle,CURL_LOCK_DATA_DNS);
    //curl_share_cleanup(CURLSH * share_handle );
    curl_easy_setopt (easy, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V4);
    curl_easy_setopt (easy, CURLOPT_DNS_CACHE_TIMEOUT, 60 * 5);
    curl_easy_setopt (easy, CURLOPT_ENCODING, "");
    curl_easy_setopt (easy, CURLOPT_USERAGENT, "KMPlayer/" VERSION);
    curl_easy_setopt (easy, CURLOPT_SSL_VERIFYPEER, 0);
    curl_easy_setopt (easy, CURLOPT_URL, (const char *)url);
    if (headers)
        curl_easy_setopt (easy, CURLOPT_HTTPHEADER, headers);
    //if (!strncmp (u, "http", 4))
    //    curl_easy_setopt (easy, CURLOPT_HTTPGET, true);
    if (CON_IC_STATUS_CONNECTED == ic_status || URL (url).isLocalFile ()) {
        start ();
    } else if (ic_connect_request) {
        ic_connect_waiters = g_slist_prepend (ic_connect_waiters, this);
    } else {
        if (!con_ic_connection) {
            con_ic_connection = con_ic_connection_new ();
            g_signal_connect (G_OBJECT (con_ic_connection), "connection-event",
                    G_CALLBACK (icConnection), NULL);
        }
        ic_connect_request = true;
        if (con_ic_connection_connect (con_ic_connection,
                CON_IC_CONNECT_FLAG_NONE)) {
            if (!ic_connect_request)
                start ();
            else
                ic_connect_waiters = g_slist_prepend (ic_connect_waiters, this);
        } else {
            ic_connect_request = false;
            job->kill (false);
        }
    }
}

void CurlGetJob::start () {
    CURLMcode type = curl_multi_add_handle (multi, easy);
    if (type && type != CURLM_CALL_MULTI_PERFORM)
        debugLog() << "CurlGetJob add to multi failure" << endl;
    else if (!ic_connect_waiters)
        curlPerformMulti ();
}

KMPLAYER_NO_CDTOR_EXPORT CurlGetJob::~CurlGetJob () {
    //debugLog() << "CurlGetJob::~CurlGetJob " << easy << " " << --job_count << endl;
    if (ic_connect_waiters)
        ic_connect_waiters = g_slist_remove (ic_connect_waiters, this);
    if (easy) {
        curl_multi_remove_handle (multi, easy);
        curl_easy_cleanup (easy);
        g_tree_remove (easy_list, easy);
    }
    if (headers)
        curl_slist_free_all (headers);
    if (0) {
        curl_multi_cleanup (multi);
        multi = NULL;
    }
}

KMPLAYER_NO_EXPORT IOJob::IOJob (IOJobPrivate *priv) : d (priv) {
}

KMPLAYER_NO_EXPORT IOJob::~IOJob () {
    IOJobListener *listener = d->easy && !d->quiet ? d->m_receiver : NULL;

    if (listener)
        listener->jobResult (this);

    delete d;
}

KMPLAYER_NO_EXPORT void IOJob::setHeader (const String &header) {
    d->headers = curl_slist_append (d->headers, (const char *) header);
}

KMPLAYER_NO_EXPORT void IOJob::setHttpPostData (const ByteArray &data) {
    d->post_data = data;
}

KMPLAYER_NO_EXPORT void IOJob::start (off_t pos) {
    d->init (this, pos);
}

KMPLAYER_NO_EXPORT void IOJob::kill (bool q) {
    d->quiet = q;
    d->killed = true;
    curl_job_killed = true;
    if (!in_perform_multi) {
        delete this;
	curlPerformMulti ();
    }
}

KMPLAYER_NO_EXPORT bool IOJob::error () {
    return d->error_code;
}

KMPLAYER_NO_EXPORT void IOJob::setError () {
    d->error_code = 1;
}

unsigned long IOJob::contentLength() const {
    return d->content_length;
}

String IOJob::contentType () const {
    return d->content_type;
}

