//#include <config.h>

#include <string.h>

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <pango/pango-font.h>
#include <hildon/hildon-defines.h>
#include <hildon/hildon-gtk.h>

#include "kmplayerplaylist.h"
#include "kmplayer.h"
#include "kmplayercontrol.h"
#include "kmplayer_lists.h"

using namespace KMPlayer;

static GdkPixbuf *folder_pix;
static GdkPixbuf *auxiliary_pix;
static GdkPixbuf *auxiliary_fld_pix;
static GdkPixbuf *unknown_pix;
static GdkPixbuf *config_pix;
static GdkPixbuf *music_pix;
static GdkPixbuf *video_pix;
static GdkPixbuf *video_link_pix;
static GdkPixbuf *list_pix;

/*
#define PLAYLIST_TYPE            (playlist_get_type ())
#define PLAYLIST(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), PLAYLIST_TYPE, PlayList))
#define PLAYLIST_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass),  PLAYLIST_TYPE, PlayListClass))
#define IS_PLAYLIST(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), PLAYLIST_TYPE))
#define IS_PLAYLIST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass),  PLAYLIST_TYPE))
#define PLAYLIST_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj),  PLAYLIST_TYPE, PlayListClass))

// The data columns that we export via the tree model interface

enum
{
  PLAYLIST_COL_ICON = 0,
  PLAYLIST_COL_TEXT
  PLAYLIST_N_COLUMNS
} ;


typedef struct _PlayList       PlayList;
typedef struct _PlayListClass  PlayListClass;

class PlaylistTree {
    PlaylistTree (NodePtr root, int _id, PlaylistTree * n)
        : node (root), id (_id), show_all (false), next (n) {}
    NodePtrW node;
    int id;
    bool show_all;
    PlaylistTree * next;
};

struct _PlayList {
    GObject parent;      // this MUST be the first member

    int num_trees;
    PlaylistTree  *trees; 

    gint            n_columns;
    GType           column_types[PLAYLIST_N_COLUMNS];
};

struct _PlayListClass {
  GObjectClass parent_class;
};


GType  playlist_get_type (void);
PlayList *playlist_new (void);
int playlistAddTree (PlayList *playlist, NodePtr root);
*/
enum {
    COL_ICON = 0,
    COL_TEXT,
    COL_ID,
    NUM_COLS
};

static GdkPixbuf *iconForNode (Node *n, int child_count) {
    if (n->auxiliaryNode ())
        return child_count > 0 ? auxiliary_fld_pix : auxiliary_pix;
    return n->isPlayable()
        ? n->mrl ()->resolved
            ? n->mrl ()->audio_only
                ? music_pix
                : video_pix
            : video_link_pix
        : child_count > 0
            ? folder_pix
            : unknown_pix;
}

static
int populate (GtkTreeStore * tree_store, PlayListData * pld,
        Node *e, GtkTreeIter * riter, GtkTreeIter * piter,
        GtkTreePath ** fpath)
{
    GtkTreeIter iter, citer, cciter;
    bool m_have_dark_nodes = false;
    int child_count = 0;
    PlaylistRole *title = (PlaylistRole *) e->role (RolePlaylist);
    if (!pld->show_all && !title && piter) {
        if (pld->current.ptr () == e)
            *fpath = gtk_tree_model_get_path (GTK_TREE_MODEL (tree_store), piter);
        for (Node *c = e->firstChild (); c; c = c->nextSibling ())
            child_count += populate (tree_store, pld, c, riter, piter, fpath);
        return child_count;
    }
    if (piter)
        gtk_tree_store_append (tree_store, &iter, piter);
    else
        iter = *riter;
    String text (title ? title->caption () : "");
    if (text.isEmpty ()) {
        Mrl *mrl = e->mrl ();
        text = e->nodeName ();
        if (!pld->show_all && e->isDocument ())
            text = e->hasChildNodes () ? "unnamed" : "none";
    }
    gtk_tree_store_set (tree_store, &iter,
            COL_TEXT, (const char *) text, COL_ID, pld->id, -1);
    if (pld->current.ptr () == e)
        *fpath = gtk_tree_model_get_path (GTK_TREE_MODEL (tree_store), &iter);
    //if (e->active ())
    //    ; //ensureItemVisible (item);
    if (e->isElementNode ()) {
        Attribute *a = static_cast <Element *> (e)->attributes ().first ();
        if (a) {
            m_have_dark_nodes = true;
            if (pld->show_all) {
                gtk_tree_store_append (tree_store, &citer, &iter);
                gtk_tree_store_set (tree_store, &citer,
                        COL_ICON, list_pix,
                        COL_TEXT, "[attributes]",
                        COL_ID, pld->id, -1);
                for (; a; a = a->nextSibling ()) {
                    gtk_tree_store_append (tree_store, &cciter, &citer);
                    gchar * s = g_strdup_printf ("%s=%s",
                            (const char *) a->name ().toString (),
                            (const char *) a->value ());
                    gtk_tree_store_set (tree_store, &cciter,
                            COL_ICON, config_pix,
                            COL_TEXT, s,
                            COL_ID, pld->id, -1);
                    g_free (s);
                }
                child_count = 1;
            }
        }
    }
    for (Node *c = e->firstChild (); c; c = c->nextSibling ())
        child_count += populate (tree_store, pld, c, riter, &iter, fpath);
    GdkPixbuf *icon = piter ? iconForNode (e, child_count) : pld->icon;
    gtk_tree_store_set (tree_store, &iter, COL_ICON, icon, -1);
    return child_count + 1;
}

/*static void dumpIter (tkTreeModel *mode, GtkTreeIter *iter) {
    GValue value = { 0 , };
    gtk_tree_model_get_value (model, &iter, COL_TEXT, &value);
    const gchar *val = g_value_get_string (&value);
    debugLog () << "found iter: " << val << endl;
    g_value_unset (&value);
}*/

static int countFirstExposed (Node *node, bool recursive)
{
    if (node->role (RolePlaylist))
        return 1;
    int nr = 0;
    if (recursive)
        for (Node *c = node->firstChild (); c; c = c->nextSibling ())
            nr += countFirstExposed (c, true);
    return nr;
}

static gboolean findNodeImpl (PlayListData *pld, GtkTreeModel *model, Node *node, GtkTreeIter *iter, bool &expand) {
    Node *p = node->parentNode ();
    if (p) {
        if (!findNodeImpl (pld, model, p, iter, expand))
            return false;
        for (Node *n = p->firstChild (); n; n = n->nextSibling ()) {
            int exposed = pld->show_all ? 1 : countFirstExposed (n, n != node);
            for (int i = exposed; i > 0; --i) {
                if (expand) {
                    expand = false;
                    GtkTreeIter piter = *iter;
                    if (!gtk_tree_model_iter_children (model, iter, &piter))
                        return false;
                    if (pld->show_all && n->isElementNode () &&
                            static_cast<Element*>(n)->attributes().first() &&
                            !gtk_tree_model_iter_next (model, iter))
                        return false;
                } else if (!gtk_tree_model_iter_next (model, iter)) {
                    return false;
                }
            }
            if (n == node) {
                if (exposed)
                    expand = true;
                return true;
            }
        }
        errorLog() << "findNode node not found" << endl;
        return false;
    } else {
        expand = true;
        return true;
    }
}

static gboolean findNode (PlayListData *pld, GtkTreeModel *model, Node *node, GtkTreeIter *iter)
{
    bool b = true;
    return findNodeImpl (pld, model, node, iter, b);
}

void KMPlayer::updatePlayTree (Application *app) {
    app->in_tree_update = true;
    PlayListData *pld = CONTROL(app)->playlist_data;
    GtkTreeView *treeview = GTK_TREE_VIEW (app->playlist_view);

    Node *node = pld->current.ptr ();
    if (!pld->show_all)
        while (node && !node->role (RolePlaylist))
            node = node->parentNode ();
    if (!node) {
        // undo selection
        return;
    }
    GtkTreeModel *model = gtk_tree_view_get_model (treeview);
    GtkTreeIter iter;
    if (!gtk_tree_model_get_iter_first (model, &iter ))
        return;
    if (findNode (pld, model, node, &iter)) {
        //GdkPixbuf *icon = piter ? iconForNode (e.ptr (), 0) : pld->icon;
        GdkPixbuf *icon = iconForNode (node, 0);
        gtk_tree_store_set (GTK_TREE_STORE (model), &iter, COL_ICON, icon, -1);
    } else {
        errorLog() << "findNode return false" << endl;
    }
    app->in_tree_update = false;
}

void KMPlayer::updateTree (Application * app, short id) {
    GtkTreeView *treeview = GTK_TREE_VIEW (app->playlist_view);
    GtkTreeIter valid_iter, *iter;
    GtkTreePath *fpath = 0L;
    GtkTreeIter oldriter, riter;
    bool valid_tree = false;
    PlayListData *pld = CONTROL(app)->playlist_data;
    GtkTreeModel *model = gtk_tree_view_get_model (treeview);
    Document *doc;

    for (valid_tree = gtk_tree_model_get_iter_first (model, &oldriter );
            valid_tree && pld;
            valid_tree = gtk_tree_model_iter_next (model, &oldriter )) {
        GValue value = { 0 , };
        gtk_tree_model_get_value (model, &oldriter, COL_ID, &value);
        int ival = g_value_get_int (&value);
        g_value_unset (&value);
        if (ival == id)
            break;
        pld = pld->next;
    }

    if (!valid_tree || !pld || !pld->document) {
        if (pld)
            pld->updated_document = NULL;
        return;
    }

    app->in_tree_update = true;
    doc = pld->document->document ();

    if (!pld->id &&
            pld->updated_document == doc &&
            pld->updated_document_version == doc->m_tree_version &&
            gtk_tree_model_get_iter_first (model, &riter )) {
        if (pld->current &&
                findNode (pld, model, pld->current.ptr (), &riter))
            fpath = gtk_tree_model_get_path (model, &riter);
        else
            gtk_tree_selection_unselect_all (gtk_tree_view_get_selection (treeview));

    } else {

        valid_tree = gtk_tree_store_remove (GTK_TREE_STORE (model), &oldriter);
        if (valid_tree)
            gtk_tree_store_insert_before(GTK_TREE_STORE(model),&riter,0L,&oldriter);
        else
            gtk_tree_store_append (GTK_TREE_STORE(model), &riter, 0L);
        populate (GTK_TREE_STORE (model), pld,
                pld->document.ptr (), &riter, 0L, &fpath);

        while (!gtk_tree_model_get_iter (model, &valid_iter, fpath) &&
                gtk_tree_path_up (fpath));
    }

    if (fpath) {
        if (gtk_tree_path_get_depth (fpath) > 1)
            gtk_tree_view_expand_to_path (treeview, fpath);

        gtk_tree_view_set_cursor (treeview, fpath, NULL, false);
        gtk_tree_view_scroll_to_cell (treeview, fpath, NULL, false, 0, 0);
        gtk_tree_path_free (fpath);
    }

    pld->updated_document = doc;
    pld->updated_document_version = doc->m_tree_version;
    app->in_tree_update = false;
}

static PlayListData *
getListData (Application * app, GtkTreeModel * model, GtkTreeIter *iter) {
    GValue value = { 0 , };
    gtk_tree_model_get_value (model, iter, COL_ID, &value);
    int id = g_value_get_int (&value);
    g_value_unset (&value);
    PlayListData * pld = CONTROL(app)->playlist_data;
    while (pld && pld->id != id)
        pld = pld->next;
    return pld;
}

static PlayListData *getListData (Control *ctr, GtkTreePath * path) {
    if (gtk_tree_path_get_depth (path) < 1)
        return 0L;
    gint id = gtk_tree_path_get_indices (path) [0];
    PlayListData * pld = ctr->playlist_data;
    while (pld && id-- > 0)
        pld = pld->next;
    return pld;
}

static PlayListData * getListDataFromSelection (Application * app, GtkTreePath ** path = 0L) {
    GtkTreeIter iter;
    GtkTreeSelection *sel = gtk_tree_view_get_selection
        (GTK_TREE_VIEW (app->playlist_view));
    GtkTreeModel * model = gtk_tree_view_get_model
        (GTK_TREE_VIEW (app->playlist_view));
    if (gtk_tree_selection_get_selected (sel, 0L, &iter)) {
        if (path)
            *path = gtk_tree_model_get_path (model, &iter);
        return getListData (app, model, &iter);
    }
    return 0L;
}

void collapseSelectedList (Application *app)
{
    GtkTreeIter iter;
    GtkTreeView *tree = GTK_TREE_VIEW (app->playlist_view);
    GtkTreeSelection *sel = gtk_tree_view_get_selection (tree);
    if (gtk_tree_selection_get_selected (sel, 0L, &iter)) {
        GtkTreeModel *model = gtk_tree_view_get_model (tree);
        GtkTreePath *path = gtk_tree_model_get_path (model, &iter);
        if (gtk_tree_model_iter_n_children (model, &iter) &&
                gtk_tree_view_row_expanded (tree, path)) {
            gtk_tree_view_collapse_row (tree, path);
        } else if (gtk_tree_path_get_depth (path) > 1) {
            gtk_tree_path_up (path);
            gtk_tree_view_collapse_row (tree, path);
        }
        gtk_tree_view_set_cursor (tree, path, NULL, false);
        gtk_tree_path_free (path);
        app_selection_changed (app);
    }
}

void expandSelectedList (Application *app)
{
    GtkTreeIter iter;
    GtkTreeView *tree = GTK_TREE_VIEW (app->playlist_view);
    GtkTreeSelection *sel = gtk_tree_view_get_selection (tree);
    if (gtk_tree_selection_get_selected (sel, 0L, &iter)) {
        GtkTreeModel *model = gtk_tree_view_get_model (tree);
        GtkTreePath *path = gtk_tree_model_get_path (model, &iter);
        if (gtk_tree_model_iter_n_children (model, &iter) &&
                !gtk_tree_view_row_expanded (tree, path)) {
            gtk_tree_view_expand_row (tree, path, FALSE);
        }
        gtk_tree_path_free (path);
        app_selection_changed (app);
    }
}

void cb_treerow_expanded (GtkTreeView *treeview, GtkTreeIter *arg1, GtkTreePath *arg2, gpointer user_data) {
    GtkTreeIter iter;
    GtkTreeModel *model = gtk_tree_view_get_model (GTK_TREE_VIEW (treeview));
    if (gtk_tree_model_iter_n_children (model, arg1) == 1 &&
            gtk_tree_model_iter_children (model, &iter, arg1)) {
        GtkTreePath * path = gtk_tree_model_get_path (model, &iter);
        gtk_tree_view_expand_row (treeview, path, FALSE);
        gtk_tree_path_free (path);
    }
}
//gboolean    gtk_tree_view_row_expanded      (GtkTreeView *tree_view,
//                                             GtkTreePath *path);
static Node *findFirstLevelExposed (Node *n, int & idx) {
    if (!n)
        return n;
    if (n->role (RolePlaylist)) {
        if (--idx < 0)
            return n;
    } else { // not exposed node, check deeper ..
        Node *c = findFirstLevelExposed (n->firstChild (), idx);
        while (c) {
            if (idx < 0)
                return c;
            Node *c2 = c;
            c = findFirstLevelExposed (c->nextSibling (), idx);
            while (!c) { // backtracking ..
                Node *p = c2->parentNode ();
                while (!c && p && p != n) {
                    if (p->nextSibling ())
                        c = p->nextSibling ();
                    else
                        p = p->parentNode ();
                }
            }
        }
    }
    return findFirstLevelExposed (n->nextSibling (), idx);
}

static Node *getNode (PlayListData * pld, GtkTreePath * path) {
    Node *node = pld->document;
    gint depth = gtk_tree_path_get_depth (path);
    gint * idxs = gtk_tree_path_get_indices (path);
    if (pld->show_all) {
        for (int i = 1; node && i < depth; i++) {
            bool has_attrs = (node->isElementNode () &&
                    static_cast<Element *> (node)->attributes ().first ());
            int index = idxs [i] - (has_attrs ? 1 : 0);
            //    g_print ("cb_selection_changed %d->%d\n", idxs [i], index);
            node = index < 0 ? 0L : node->childNodes ().item (index);
        }
    } else {
        for (int i = 1; node && i < depth; i++) {
            int idx = idxs [i];
            node = findFirstLevelExposed (node->firstChild (), idx);
        }
    }
    return node;
}

static void setInfoContent (Application *app, Node *node, bool edit_mode) {
    app_set_info_content (app, node
            ? edit_mode
                ? node->innerXML ()
                : node->description ()
            : "", edit_mode);
}

static gboolean delayedUpdateTree (Application *app) {
#ifdef DEBUG_TIMERS
    debugLog() << "delayedUpdateTree" << endl;
#endif
    for (PlayListData *pd = CONTROL(app)->playlist_data; pd; pd = pd->next) {
        if (pd->document &&
                (!pd->updated_document ||
                 pd->updated_document_version !=
                 pd->document->document ()->m_tree_version))
            updateTree (app, pd->id);
    }
    return false;
}

static void
setPlaylistEdit (Application *app, PlayListData *pld, bool editable)
{
    if (editable) {
        gtk_widget_show (GTK_WIDGET (app->tb_item_playlist_commit));
        gtk_widget_hide (GTK_WIDGET (app->tb_save));
    } else {
        gtk_widget_hide (GTK_WIDGET (app->tb_item_playlist_commit));
        if (!GTK_WIDGET_VISIBLE (app->tb_stop))
            gtk_widget_show (GTK_WIDGET (app->tb_save));
    }
    g_object_set (G_OBJECT (app->playlist_cell_renderer),
            "editable", editable, 0L);
    g_object_set (G_OBJECT (app->infopanel_view),
            "editable", editable, "cursor-visible", editable, 0L);
    pld->edit_mode = editable;
    pld->updated_document = NULL;
    //gtk_paned_set_position (app->vsplit_view, editable ? 0 : VSPLIT);
    g_idle_add ((GSourceFunc) delayedUpdateTree, (gpointer) app);
}

static bool had_selection_change;

static gboolean cb_tapTimeout (gpointer)
{
#ifdef DEBUG_TIMERS
    debugLog() << "cb_tapTimeout" << endl;
#endif
    had_selection_change = false;
    return false;
}

static void cb_selection_changed (GtkTreeSelection *selection, Application *app)
{
    if (app->in_tree_update || !selection)
        return;
    GtkTreeView *treeview = GTK_TREE_VIEW (app->playlist_view);
    GtkTreeIter iter;
    PlayListData * pld = 0L;
    //g_print ("cb_selection_changed %d\n", app->playlist_data->show_all);
    if (gtk_tree_selection_get_selected (selection, 0L, &iter)) {
        GtkTreeModel * model = gtk_tree_view_get_model (treeview);
        PlayListData * pld = getListData (app, model, &iter);
        GtkTreePath* path = gtk_tree_model_get_path (model, &iter);

        // click outside editable tree
        for (PlayListData *pd = CONTROL(app)->playlist_data; pd; pd = pd->next)
            if (pd != pld && (pd->edit_mode || pd->show_all)) {
                pd->show_all = false;
                if (pd->edit_mode) {
                    setPlaylistEdit (app, pd, false);
                } else {
                    pd->updated_document = NULL;
                    g_idle_add ((GSourceFunc) delayedUpdateTree, (gpointer)app);
                }
                break;
            }

        gtk_tree_view_scroll_to_cell (treeview, path, NULL, false, 0, 0);
        Node *node = getNode (pld, path);
        Mrl *mrl = node ? node->mrl () : NULL;
        gtk_tree_path_free (path);
        if (mrl && pld->id > 0 && node == pld->document.ptr() && !mrl->resolved)
            mrl->activate ();
        if (mrl && pld->updated_document_version !=
                mrl->document ()->m_tree_version)
            g_idle_add ((GSourceFunc) delayedUpdateTree, (gpointer) app);
        setInfoContent (app, node, pld->edit_mode);
        if (mrl && pld->id > 0) //FIXME, hack to become less jumpy when changing trees
            pld->current = node;

        if (CONTROL(app)->current_selected.ptr () != node) {
            CONTROL(app)->setCurrentSelected (node);
            had_selection_change = true;
            g_timeout_add (200, (GSourceFunc) cb_tapTimeout, NULL);
        }
    }
}

static gboolean delayedJump (Application *app) {
    GtkTreePath *path;
#ifdef DEBUG_TIMERS
    debugLog() << "delayedJump" << endl;
#endif
    PlayListData * pld = getListDataFromSelection (app, &path);
    if (pld) {
        jump (pld->current.ptr (), CONTROL(app));
        gtk_tree_path_free (path);
    }
    return false;
}

static void
cb_hildon_row_tapped (GtkTreeView *view, GtkTreePath *path, Application *app)
{
    PlayListData *pld = getListData (CONTROL(app), path);
    if (pld) {
        Node *node = getNode (pld, path);
        if (node) {
            debugLog() << "row activated" << endl;
            if (gtk_tree_path_get_depth (path) != 1 || pld->id == 0) {
                if (pld->id == 0 || pld->edit_mode)
                    setInfoContent (app, node, pld->edit_mode);
                if (!had_selection_change &&
                        !pld->edit_mode &&
                        (pld->id == 0 || (node && node->mrl ()))) {
                    if (pld->id > 0) {
                        pld->current = node;
                        g_idle_add ((GSourceFunc) delayedJump, (gpointer) app);
                    } else {
                        jump (node, CONTROL(app));
                    }
                } else {
                    had_selection_change = false;
                }
            }
        }
    }
}

void KMPlayer::rowSelected (Application *app) {
    GtkTreeIter iter;
    GtkTreeView *view = GTK_TREE_VIEW (app->playlist_view);
    GtkTreeSelection *sel = gtk_tree_view_get_selection (view);
    GtkTreeModel *model = gtk_tree_view_get_model (view);
    bool expand_collapse = false;
    Node *node = NULL;

    if (!gtk_tree_selection_get_selected (sel, 0L, &iter))
        return;

    PlayListData *pld = getListData (app, model, &iter);
    if (!pld)
        return;

    GtkTreePath *path = gtk_tree_model_get_path (model, &iter);
    if (gtk_tree_path_get_depth (path) == 1) {
        expand_collapse = true;
        if (pld->id && !gtk_tree_model_iter_has_child (model, &iter)) {
            pld->document->activate (); //enable lazy loading
            if (!pld->document->isPlayable ())
                updateTree (app, pld->id);
        }
    } else {
        node = getNode (pld, path);
        if (node && node->id == id_node_group_node)
            expand_collapse = true;
    }
    if (expand_collapse) {
        if (gtk_tree_model_iter_has_child (model, &iter)) {
            if (!gtk_tree_view_row_expanded (view, path)) {
                GtkTreeIter next;
                gtk_tree_view_expand_row (view, path, FALSE);
                if (gtk_tree_model_iter_children (model, &next, &iter))
                    gtk_tree_selection_select_iter (sel, &next);
            } else {
                gtk_tree_view_collapse_row (view, path);
            }
        }
    } else if (node) {
        setInfoContent (app, node, pld->edit_mode);
        if (!pld->edit_mode && (pld->id == 0 || node->mrl ()))
            jump (node, CONTROL(app));
    }
    gtk_tree_path_free (path);
}

void KMPlayer::nextSelected (Application *app) {
    GtkTreeIter iter;
    GtkTreeIter next;
    GtkTreeView *view = GTK_TREE_VIEW (app->playlist_view);
    GtkTreeSelection *selection = gtk_tree_view_get_selection (view);
    GtkTreeModel *model = gtk_tree_view_get_model (view);

    if (gtk_tree_selection_get_selected (selection, NULL, &iter)) {
        GtkTreePath *path = gtk_tree_model_get_path (model, &iter);
        if (gtk_tree_model_iter_has_child (model, &iter) &&
            gtk_tree_view_row_expanded (view, path)) {
            if (gtk_tree_model_iter_children (model, &next, &iter))
                gtk_tree_selection_select_iter (selection, &next);
        } else if (gtk_tree_model_iter_next (model, &iter)) {
            gtk_tree_selection_select_iter (selection, &iter);
        } else {
            while (gtk_tree_model_iter_parent (model, &next, &iter)) {
                if (gtk_tree_model_iter_next (model, &next)) {
                    gtk_tree_selection_select_iter (selection, &next);
                    break;
                }
                iter = next;
            }
        }
        gtk_tree_path_free (path);
        app_selection_changed (app);
    }
}

void KMPlayer::previousSelected (Application *app) {
    GtkTreeIter iter;
    GtkTreeView *view = GTK_TREE_VIEW (app->playlist_view);
    GtkTreeSelection *selection = gtk_tree_view_get_selection (view);
    GtkTreeModel *model = gtk_tree_view_get_model (view);

    if (gtk_tree_selection_get_selected (selection, NULL, &iter)) {
        GtkTreePath *path = gtk_tree_model_get_path (model, &iter);
        if (gtk_tree_path_prev (path))
            gtk_tree_selection_select_path (selection, path);
        else if (gtk_tree_path_up (path))
            gtk_tree_selection_select_path (selection, path);
        gtk_tree_path_free (path);
        app_selection_changed (app);
    }
}

gboolean playlist_selected (Application *app, GValue *icon, GValue *text) {
    GtkTreeIter iter;
    GtkTreeView *view = GTK_TREE_VIEW (app->playlist_view);
    GtkTreeSelection *selection = gtk_tree_view_get_selection (view);
    GtkTreeModel *model = gtk_tree_view_get_model (view);

    if (gtk_tree_selection_get_selected (selection, NULL, &iter)) {
        gtk_tree_model_get_value (model, &iter, COL_ICON, icon);
        gtk_tree_model_get_value (model, &iter, COL_TEXT, text);
        return true;
    }
    return false;
}

void cb_treeCellDataFunc (GtkTreeViewColumn *tree_column, GtkCellRenderer *cell,
        GtkTreeModel *tree_model, GtkTreeIter *iter, Application *app) {
    GtkTreeIter parent;
    PlayListData * pld = getListData (app, tree_model, iter);
    if (!gtk_tree_model_iter_parent (tree_model, &parent, iter) ) {
        g_object_set (G_OBJECT (cell), "underline",
                PANGO_UNDERLINE_SINGLE, 0L);
        //g_object_set (G_OBJECT (cell), "scale", 1.2, 0L);
        g_object_set (G_OBJECT (cell), "editable", false, 0L);
        g_object_set (G_OBJECT (cell), "mode",
                GTK_CELL_RENDERER_MODE_ACTIVATABLE, 0L);
    } else if (pld) {
        g_object_set (G_OBJECT (cell), "underline",
                PANGO_UNDERLINE_NONE, 0L);
        //g_object_set (G_OBJECT (cell), "scale", 1.0, 0L);
        g_object_set (G_OBJECT (cell), "mode", pld->edit_mode
                ? GTK_CELL_RENDERER_MODE_EDITABLE
                : GTK_CELL_RENDERER_MODE_INERT, 0L);
        g_object_set (G_OBJECT (cell), "editable", pld->edit_mode, 0L);
    }
}

short KMPlayer::addTree (Application * app, Mrl *doc, GdkPixbuf * ico) {
    GtkTreeIter iter;
    PlayListData ** prev = &CONTROL(app)->playlist_data;
    short id = 0;
    while (*prev) {
        id = (*prev)->id + 1;
        prev = &(*prev)->next;
    }
    *prev = new PlayListData (id);
    (*prev)->document = doc;
    (*prev)->icon = ico;
    (*prev)->updated_document = doc ? doc->document () : NULL;
    String title = "empty";
    if (doc && !doc->title.isEmpty())
        title = (const char *) doc->title;
    GtkTreeModel *model = gtk_tree_view_get_model (GTK_TREE_VIEW (app->playlist_view));
    gtk_tree_store_append (GTK_TREE_STORE (model), &iter, 0L);
    gtk_tree_store_set (GTK_TREE_STORE (model), &iter,
            COL_ICON, ico, COL_TEXT, (const char *) title, COL_ID, id, -1);
    return id;
}

GtkWidget * createTreeView (Application * app) {
    GtkTreeStore  *store;
    GtkTreeModel      *model;
    GtkTreeViewColumn *col;
    GtkWidget         *view;
    //GValue            prop_value = { TRUE, };

    GtkIconTheme *icon_theme = gtk_icon_theme_get_default();
    folder_pix = gtk_icon_theme_load_icon (icon_theme,
            "filemanager_video_folder", 26, (GtkIconLookupFlags)0, 0L);
    auxiliary_pix = gtk_icon_theme_load_icon (icon_theme,
            "general_profile", 26, (GtkIconLookupFlags)0, 0L);
    auxiliary_fld_pix = gtk_icon_theme_load_icon (icon_theme,
            "filemanager_document_folder", 26, (GtkIconLookupFlags)0, 0L);
    unknown_pix = gtk_icon_theme_load_icon (icon_theme,
            "filemanager_nonreadable_file", 26, (GtkIconLookupFlags)0, 0L);
    config_pix = gtk_icon_theme_load_icon (icon_theme,
            "general_settings", 26, (GtkIconLookupFlags)0, 0L);
    music_pix = gtk_icon_theme_load_icon (icon_theme,
            "general_audio_file", 26, (GtkIconLookupFlags)0, 0L);
    video_pix = gtk_icon_theme_load_icon (icon_theme,
            "general_video_file", 26, (GtkIconLookupFlags)0, 0L);
    video_link_pix = gtk_icon_theme_load_icon (icon_theme,
            "imageviewer_favourite", 26, (GtkIconLookupFlags)0, 0L);
    list_pix = gtk_icon_theme_load_icon (icon_theme,
            "filemanager_playlist", 26, (GtkIconLookupFlags)0, 0L);

    store = gtk_tree_store_new (3, GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_INT);
    view = hildon_gtk_tree_view_new_with_model (HILDON_UI_MODE_EDIT, GTK_TREE_MODEL (store));
    g_object_unref (store);

    col = gtk_tree_view_column_new();
    //gtk_tree_view_column_set_title(col, "Title");

    app->playlist_cell_renderer = gtk_cell_renderer_pixbuf_new();
    //gtk_cell_renderer_set_fixed_size (app->playlist_cell_renderer, 26, 26);
    gtk_tree_view_column_pack_start(col, app->playlist_cell_renderer, FALSE);
    gtk_tree_view_column_set_attributes(col, app->playlist_cell_renderer,
            "pixbuf", COL_ICON,
            NULL);

    PangoContext * pango_cxt = gtk_widget_get_pango_context (view);
    PangoFontDescription * pfd = pango_context_get_font_description (pango_cxt);
    pango_font_description_set_size (pfd, 12);

    app->playlist_cell_renderer = gtk_cell_renderer_text_new();
    gtk_cell_renderer_text_set_fixed_height_from_font (
            GTK_CELL_RENDERER_TEXT(app->playlist_cell_renderer), 1);
    //g_value_init (&prop_value, G_TYPE_BOOLEAN);
    //g_value_set_boolean (&prop_value, TRUE);
    //g_object_set_property (G_OBJECT (renderer), "editable", &prop_value);
    GtkTreeSelection *sel = gtk_tree_view_get_selection(GTK_TREE_VIEW (view));
    g_object_set (G_OBJECT (view), "headers-visible", FALSE, 0L);
    g_object_set (G_OBJECT (app->playlist_cell_renderer),
            "font", "Sans 15", 0L);
    gtk_tree_view_column_pack_start(col, app->playlist_cell_renderer, TRUE);
    gtk_tree_view_column_set_attributes(col, app->playlist_cell_renderer,
            "text", COL_TEXT,
            NULL);
    g_signal_connect (G_OBJECT (view),
            "row-expanded", G_CALLBACK (cb_treerow_expanded), app);
    g_signal_connect (G_OBJECT (view),
            "hildon-row-tapped", G_CALLBACK (cb_hildon_row_tapped), app);
    g_signal_connect(G_OBJECT (sel),
            "changed", G_CALLBACK (cb_selection_changed), app);
    gtk_tree_view_column_set_cell_data_func (col, app->playlist_cell_renderer,
            (GtkTreeCellDataFunc)cb_treeCellDataFunc, app, false);

    gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);
    gtk_tree_view_set_expander_column (GTK_TREE_VIEW (view), col);
    gtk_tree_selection_set_mode (sel, GTK_SELECTION_BROWSE);

    gtk_widget_show_all(view);

    return view;
}

void cb_playlistMenuActivate (GtkWidget *, GdkEvent *, Application * app) {
    bool show_copy = false, show_paste = false, show_cut = false,
         show_title = false;
    GtkTreePath *path;
    PlayListData * pld = getListDataFromSelection (app, &path);
    if (pld) {
        Node *node = getNode (pld, path);
        if (node) {
            show_copy = isListItem (node->id) ||
                id_node_group_node == node->id ||
                (pld->id == 0 &&
                 node->mrl () &&
                 node->id != id_node_playlist_document &&
                 !node->mrl ()->src.isEmpty ());
            show_paste = pld->id == app->playlists_tree_id &&
                (!CONTROL(app)->copyPasteXml.isEmpty () ||
                 CONTROL(app)->copyPasteNode);
            show_cut = pld->id == app->playlists_tree_id &&
                gtk_tree_path_get_depth (path) > 1;
            show_title = pld->id == app->playlists_tree_id &&
                (node->id == id_node_playlist_item ||
                 node->id == id_node_group_node ||
                 node->id == id_node_html_object);
        }
    }
    if (show_copy)
        gtk_widget_show (app->menu_item_playlist_copy);
    else
        gtk_widget_hide (app->menu_item_playlist_copy);
    if (show_paste) {
        gtk_widget_show (app->menu_item_playlist_paste);
        gtk_widget_show (app->menu_item_playlist_paste_in_group);
    } else {
        gtk_widget_hide (app->menu_item_playlist_paste);
        gtk_widget_hide (app->menu_item_playlist_paste_in_group);
    }
    if (show_cut)
        gtk_widget_show (app->menu_item_playlist_cut);
    else
        gtk_widget_hide (app->menu_item_playlist_cut);
}

void cb_playlistCopy (GtkButton * button, Application * app) {
    GtkTreePath *path;
    PlayListData * pld = getListDataFromSelection (app, &path);
    if (pld) {
        Node *node = getNode (pld, path);
        if (node) {
            CONTROL(app)->copyPasteNode = 0L;
            if (isListItem (node->id) || id_node_group_node == node->id) {
                CONTROL(app)->copyPasteXml = node->outerXML (false);
            } else if (node->mrl ()) {
                TextStream txt;
                txt << "<item url=\""
                    << XMLStringlet (node->mrl ()->src)
                    << "\"/>";
                CONTROL(app)->copyPasteXml = txt.release ();
            }
            if (node->mrl ())
                gtk_clipboard_set_text (gtk_clipboard_get_for_display (
                            gtk_widget_get_display (GTK_WIDGET (button)),
                            GDK_SELECTION_CLIPBOARD),
                        (const char *)node->mrl ()->src, -1);
        }
    }
}

void cb_playlistPaste (GtkButton * button, Application * app) {
    GtkTreePath *path;
    PlayListData * pld = getListDataFromSelection (app, &path);
    if (pld) {
        Node *node = getNode (pld, path);
        if (node && pld->id == app->playlists_tree_id) {
            Node *newnode = CONTROL(app)->copyPasteNode;
            if (newnode) {
                if (gtk_tree_path_get_depth (path) == 1)
                    node->insertBefore (newnode, node->firstChild ());
                else
                    node->parentNode ()->insertBefore (newnode, node);
            } else if (!CONTROL(app)->copyPasteXml.isEmpty ()) {
                if (gtk_tree_path_get_depth (path) == 1) {
                    node->insertBefore (
                            CONTROL(app)->copyPasteXml, node->firstChild ());
                    newnode = node->firstChild ();
                } else {
                    node->parentNode ()->insertBefore (
                            CONTROL(app)->copyPasteXml, node);
                    newnode = node->previousSibling ();
                }
            }
            static_cast <Element *> (newnode)->setAttribute (
                    "clipBegin", String ());
            pld->current = newnode;
            updateTree (app, app->playlists_tree_id);
        }
    }
}

String KMPlayer::getUserInput (Application *app,
        const gchar *title, const gchar *desc, const gchar *def) {
    GtkWidget * dialog= gtk_dialog_new_with_buttons (
            title,
            GTK_WINDOW (app->window), GTK_DIALOG_MODAL,
            GTK_STOCK_OK, GTK_RESPONSE_OK,
            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, NULL);
    if (desc) {
        GtkWidget *lbl = gtk_label_new (desc);
        gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), lbl);
    }
    GtkWidget *edit = gtk_entry_new();
    if (def) {
        gtk_entry_set_text (GTK_ENTRY (edit), def);
        gtk_editable_select_region (GTK_EDITABLE (edit), 0, -1);
    }
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), edit);
    gtk_window_set_default_size(GTK_WINDOW(dialog), 400, 150);

    gtk_widget_show_all (dialog);

    String val;
    if (GTK_RESPONSE_OK == gtk_dialog_run (GTK_DIALOG (dialog)))
        val = gtk_entry_get_text (GTK_ENTRY (edit));

    gtk_widget_destroy (dialog);

    return val;
}

void cb_playlistPasteInNewGroup (GtkButton * button, Application * app) {
    GtkTreePath *path;
    PlayListData * pld = getListDataFromSelection (app, &path);
    if (pld) {
        Node *node = getNode (pld, path);
        if (node && pld->id == app->playlists_tree_id) {
            String grp = getUserInput (app,
                    "New Group", "Enter name of the group", NULL);
            if (!grp.isNull ()) {
                NodePtr group = new Group (pld->document, app, grp);
                if (CONTROL(app)->copyPasteNode) {
                    group->appendChild (CONTROL(app)->copyPasteNode);
                } else if (!CONTROL(app)->copyPasteXml.isEmpty ()) {
                    group->insertBefore (CONTROL(app)->copyPasteXml, NULL);
                    if (group->firstChild ())
                        static_cast <Element *> (group->firstChild ())->
                            setAttribute ("clipBegin", String ());
                }
                if (gtk_tree_path_get_depth (path) == 1)
                    node->insertBefore (group, node->firstChild ());
                else
                    node->parentNode ()->insertBefore (group, node);
                pld->current = group;
                updateTree (app, app->playlists_tree_id);
            }
        }
    }
}

void cb_playlistCut (GtkButton * button, Application * app) {
    GtkTreePath *path;
    PlayListData * pld = getListDataFromSelection (app, &path);
    if (pld) {
        Node *node = getNode (pld, path);
        if (node &&
                pld->id == app->playlists_tree_id &&
                gtk_tree_path_get_depth (path) > 1) {
            CONTROL(app)->copyPasteXml.clear ();
            CONTROL(app)->copyPasteNode = node;
            if (node->previousSibling ())
                pld->current = node->previousSibling ();
            else if (node->nextSibling ())
                pld->current = node->nextSibling ();
            else
                pld->current = node->parentNode ();
            node->parentNode ()->removeChild (node);
            updateTree (app, app->playlists_tree_id);
        }
    }
}

void cb_playlistTitle (Application *app) {
    GtkTreePath *path;
    PlayListData *pld = getListDataFromSelection (app, &path);
    if (pld) {
        Node *node = getNode (pld, path);
        PlaylistRole *pr = node ? (PlaylistRole*)node->role(RolePlaylist): NULL;
        if (pr && pld->id == app->playlists_tree_id) {
            String title = getUserInput (app,
                    "New Title", "Enter new title", pr->caption ());
            if (!title.isNull ()) {
                pr->setCaption (title);
                static_cast <Element *> (node)->setAttribute (
                        StringPool::attr_title, title);
                node->document ()->m_tree_version++;
                updateTree (app, app->playlists_tree_id);
            }
        }
        gtk_tree_path_free (path);
    }
}

void togglePlaylistShowAll (Application *app) {
    //g_print ("togglePlaylistShowAll\n");
    PlayListData * pld = getListDataFromSelection (app);
    if (pld) {
        bool selected = !pld->show_all;
        if (!selected && !pld->show_all) { // click outside editable tree
            for (PlayListData * pd = CONTROL(app)->playlist_data; pd; pd = pd->next)
                if (pd->show_all) {
                    pld = pd;
                    break;
                }
        }
        pld->show_all = selected;
        pld->updated_document = NULL;
        g_idle_add ((GSourceFunc) delayedUpdateTree, (gpointer) app);
    }
}

void togglePlaylistEdit (Application *app) {
    //g_print ("togglePlaylistEdit\n");
    PlayListData * pld = getListDataFromSelection (app);
    if (pld) {
        setPlaylistEdit (app, pld, !pld->edit_mode);
        NodePtr node = pld->current;
        if (!node)
            node = pld->document;
        setInfoContent (app, node.ptr (), pld->edit_mode);
    }
}

void cb_playlist_edit_commit (GtkButton * button, Application * app) {
    GtkTextIter start, end;
    gtk_text_buffer_get_bounds (gtk_text_view_get_buffer (GTK_TEXT_VIEW (app->infopanel_view)), &start, &end);
    gchar * str = gtk_text_buffer_get_text (gtk_text_view_get_buffer (GTK_TEXT_VIEW (app->infopanel_view)), &start, &end, TRUE);
    GtkTreePath *path;
    PlayListData * pld = getListDataFromSelection (app, &path);
    if (pld) {
        Node *node = getNode (pld, path);
        if (node) {
            NodePtr tmp = new GenericURL (pld->document, "foo", "bar");
            while (node->firstChild ()) {
                NodePtr c = node->firstChild ();
                node->removeChild (c);
                tmp->appendChild (c.ptr ());
            }
            node->insertBefore (str, NULL);
            updateTree (app, pld->id);
        } else
            warningLog() << "No node selected" << endl;
        gtk_tree_path_free (path);
    }
    g_free (str);
}

/*
// boring declarations of local functions

static void         playlist_init            (PlayList      *pkg_tree);

static void         playlist_class_init      (PlayListClass *klass);

static void         playlist_tree_model_init (GtkTreeModelIface *iface);

static void         playlist_finalize        (GObject           *object);

static GtkTreeModelFlags playlist_get_flags  (GtkTreeModel      *tree_model);

static gint         playlist_get_n_columns   (GtkTreeModel      *tree_model);

static GType        playlist_get_column_type (GtkTreeModel      *tree_model,
                                                 gint               index);

static gboolean     playlist_get_iter        (GtkTreeModel      *tree_model,
                                                 GtkTreeIter       *iter,
                                                 GtkTreePath       *path);

static GtkTreePath *playlist_get_path        (GtkTreeModel      *tree_model,
                                                 GtkTreeIter       *iter);

static void         playlist_get_value       (GtkTreeModel      *tree_model,
                                                 GtkTreeIter       *iter,
                                                 gint               column,
                                                 GValue            *value);

static gboolean     playlist_iter_next       (GtkTreeModel      *tree_model,
                                                 GtkTreeIter       *iter);

static gboolean     playlist_iter_children   (GtkTreeModel      *tree_model,
                                                 GtkTreeIter       *iter,
                                                 GtkTreeIter       *parent);

static gboolean     playlist_iter_has_child  (GtkTreeModel      *tree_model,
                                                 GtkTreeIter       *iter);

static gint         playlist_iter_n_children (GtkTreeModel      *tree_model,
                                                 GtkTreeIter       *iter);

static gboolean     playlist_iter_nth_child  (GtkTreeModel      *tree_model,
                                                 GtkTreeIter       *iter,
                                                 GtkTreeIter       *parent,
                                                 gint               n);

static gboolean     playlist_iter_parent     (GtkTreeModel      *tree_model,
                                                 GtkTreeIter       *iter,
                                                 GtkTreeIter       *child);



static GObjectClass *parent_class = NULL;  // GObject stuff - nothing to worry about


/ *****************************************************************************
 *
 *  playlist_get_type: here we register our new type and its interfaces
 *                        with the type system. If you want to implement
 *                        additional interfaces like GtkTreeSortable, you
 *                        will need to do it here.
 *
 ***************************************************************************** /

GType playlist_get_type (void) {
    static GType playlist_type = 0;

    if (playlist_type)
        return playlist_type;

    // Some boilerplate type registration stuff
    if (1) {
        static const GTypeInfo playlist_info = {
            sizeof (PlayListClass),
            NULL,                                         // base_init
            NULL,                                         // base_finalize
            (GClassInitFunc) playlist_class_init,
            NULL,                                         // class finalize
            NULL,                                         // class_data
            sizeof (PlayList),
            0,                                           // n_preallocs
            (GInstanceInitFunc) playlist_init
        };

        playlist_type = g_type_register_static (G_TYPE_OBJECT, "PlayList",
                &playlist_info, (GTypeFlags)0);
    }

    // Here we register our GtkTreeModel interface with the type system
    if (1) {
    static const GInterfaceInfo tree_model_info = {
        (GInterfaceInitFunc) playlist_tree_model_init,
        NULL,
        NULL
    };

    g_type_add_interface_static (playlist_type, GTK_TYPE_TREE_MODEL, &tree_model_info);
    }
    return playlist_type;
}


/ *****************************************************************************
 *
 *  playlist_class_init: more boilerplate GObject/GType stuff.
 *                          Init callback for the type system,
 *                          called once when our new class is created.
 *
 ***************************************************************************** /

static void playlist_class_init (PlayListClass *klass) {
    GObjectClass *object_class;

    parent_class = (GObjectClass*) g_type_class_peek_parent (klass);
    object_class = (GObjectClass*) klass;

    object_class->finalize = playlist_finalize;
}

/ *****************************************************************************
 *
 *  playlist_tree_model_init: init callback for the interface registration
 *                               in playlist_get_type. Here we override
 *                               the GtkTreeModel interface functions that
 *                               we implement.
 *
 ***************************************************************************** /

static void playlist_tree_model_init (GtkTreeModelIface *iface) {
    iface->get_flags       = playlist_get_flags;
    iface->get_n_columns   = playlist_get_n_columns;
    iface->get_column_type = playlist_get_column_type;
    iface->get_iter        = playlist_get_iter;
    iface->get_path        = playlist_get_path;
    iface->get_value       = playlist_get_value;
    iface->iter_next       = playlist_iter_next;
    iface->iter_children   = playlist_iter_children;
    iface->iter_has_child  = playlist_iter_has_child;
    iface->iter_n_children = playlist_iter_n_children;
    iface->iter_nth_child  = playlist_iter_nth_child;
    iface->iter_parent     = playlist_iter_parent;
}

static void playlist_init (PlayList *playlist) {
    playlist->n_columns       = PLAYLIST_N_COLUMNS;

    playlist->column_types[0] = GDK_TYPE_PIXBUF;
    playlist->column_types[1] = G_TYPE_STRING;

    playlist->num_trees = 0;
    playlist->trees     = NULL;
}

static void playlist_finalize (GObject *object) {
    Playlist *playlist = PLAYLIST (object);
    while (trees) {
        PlaylistTree * tmp = trees;
        trees = trees->next;
        delete tmp;
    }
    // must chain up - finalize parent
    (* parent_class->finalize) (object);
}

static GtkTreeModelFlags playlist_get_flags (GtkTreeModel *tree_model) {
    g_return_val_if_fail (IS_PLAYLIST(tree_model), (GtkTreeModelFlags)0);
    return 0;//(GTK_TREE_MODEL_LIST_ONLY | GTK_TREE_MODEL_ITERS_PERSIST);
}

static gint playlist_get_n_columns (GtkTreeModel *tree_model) {
    g_return_val_if_fail (IS_PLAYLIST(tree_model), 0);
    return PLAYLIST(tree_model)->n_columns;
}

static GType playlist_get_column_type (GtkTreeModel *tree_model, gint index) {
    g_return_val_if_fail (IS_PLAYLIST(tree_model), G_TYPE_INVALID);
    g_return_val_if_fail (index < PLAYLIST(tree_model)->n_columns && index >= 0, G_TYPE_INVALID);
    return PLAYLIST(tree_model)->column_types[index];
}

static NodePtr getFirstExposedNode (NodePtr node, bool / *show_all* /) {
    return node; // TODO account for show_all
}

static NodePtr getNextExposedNode (NodePtr node, bool / *show_all* /) {
    return node->nextSibling (); // TODO account for show_all
}

static PlaylistTree *
findPlaylistTree (GtkTreeModel *tm, GtkTreeIter  *iter, int & i, Node *& n) {
    g_return_val_if_fail (IS_PLAYLIST(tm), NULL);
    g_return_val_if_fail (iter != NULL, NULL);

    PlayList *playlist = PLAYLIST(tm);
    PlaylistTree *tree = plst->trees;
    i = 0;
    n = 0L;
    while (tree != static_cast <PlaylistTree *> (iter->user_data2)) {
        if (!tree)
            return 0L;
        i++;
        tree = tree->next;
    }
    if (!tree || !tree->node ||
            tree->node->document()->tree_version != iter->stamp)
        return 0L;
    n = static_cast <Node *> (iter->user_data);
    return tree;
}

static gboolean
playlist_get_iter (GtkTreeModel *tmodel, GtkTreeIter *iter, GtkTreePath *path) {
    g_assert(IS_PLAYLIST(model));
    g_assert(path!=NULL);

    PlayList * playlist = PLAYLIST(tmodel);

    gint *indices = gtk_tree_path_get_indices(path);
    gint depth   = gtk_tree_path_get_depth(path);

    PlaylistTree  *tree = playlist->trees;
    for (int i = indices[0]; i > 0; i--) {
        if (!tree)
            return false;
        tree = tree->next;
    }
    if (!tree || !tree->node)
        return false;
    NodePtr node = tree->node;
    for (int d = 1; d < depth; d++) {
        node = getFirstExposedNode (node->firstChild (), tree->show_all);
        for (int i = indices[d]; i > 0; i--) {
            if (!node)
                return false;
            node = getNextExposedNode (node, tree->show_all);
        }
    }
    if (!node)
        return false;

    iter->stamp = tree->document()->tree_version;
    iter->user_data  = node.ptr();
    iter->user_data2 = tree;
    iter->user_data3 = 0L;   // unused

    return true;
}

static GtkTreePath *
playlist_get_path (GtkTreeModel *tree_model, GtkTreeIter  *iter) {
    int i = 0;
    Node * n;
    PlaylistTree * tree = findPlaylistTree (tree_model, iter, i, n);
    if (!tree || !n)
        return 0L;

    GtkTreePath  *path = gtk_tree_path_new();
    NodePtr node = n;
    for (NodePtr p = n->parentNode (); p; p = node->parentNode ()) {
        int j = 0;
        for (NodePtr c = p->firstChild (); c; c = c->nextSibling (), ++j)
            if (c == node) {
                gtk_tree_path_prepend_index (path, j);
                node = p;
                break;
            }
    }
    gtk_tree_path_prepend_index (path, i);
    return path;
}

static void
playlist_get_value (GtkTreeModel *tree_model, GtkTreeIter *iter, gint col, GValue *val)
{
    Playlist * playlist = PLAYLIST(tree_model);
    g_return_if_fail (col < (int) PLAYLIST_N_COLUMNS);
    int i = 0;
    Node * n;
    PlaylistTree * tree = findPlaylistTree (tree_model, iter, i, n);
    if (!tree || !n)
        return;

    g_value_init (val, playlist->column_types[col]);

    switch (col) {
        case PLAYLIST_COL_ICON:
            g_value_set_pixmap (n != tree->node ? (n->isPlayable() ? video_pix : (n->firstChild() ? (e->auxiliaryNode () ? auxiliary_pix : folder_pix) : unknown_pix)) : www_pix);
            break;
        case PLAYLIST_COL_TEXT:
            g_value_set_string (val, n->nodeName ());
            break;
    }
}


/ *****************************************************************************
 *
 *  playlist_iter_next: Takes an iter structure and sets it to point
 *                         to the next row.
 *
 ***************************************************************************** /

static gboolean
playlist_iter_next (GtkTreeModel *tree_model, GtkTreeIter *iter) {
  PlaylistTree  *record, *nextrecord;
  PlayList    *playlist;

  g_return_val_if_fail (IS_PLAYLIST (tree_model), FALSE);

  if (iter == NULL || iter->user_data == NULL)
    return FALSE;

  playlist = PLAYLIST(tree_model);

  record = (PlaylistTree *) iter->user_data;

  // Is this the last record in the list?
  if ((record->pos + 1) >= playlist->num_trees)
    return FALSE;

  nextrecord = playlist->rows[(record->pos + 1)];

  g_assert ( nextrecord != NULL );
  g_assert ( nextrecord->pos == (record->pos + 1) );

  iter->stamp     = playlist->stamp;
  iter->user_data = nextrecord;

  return TRUE;
}


/ *****************************************************************************
 *
 *  playlist_iter_children: Returns TRUE or FALSE depending on whether
 *                             the row specified by 'parent' has any children.
 *                             If it has children, then 'iter' is set to
 *                             point to the first child. Special case: if
 *                             'parent' is NULL, then the first top-level
 *                             row should be returned if it exists.
 *
 ***************************************************************************** /

static gboolean
playlist_iter_children (GtkTreeModel *tree_model,
                           GtkTreeIter  *iter,
                           GtkTreeIter  *parent)
{
  PlayList  *playlist;

  g_return_val_if_fail (parent == NULL || parent->user_data != NULL, FALSE);

  // this is a list, nodes have no children
  if (parent)
    return FALSE;

  // parent == NULL is a special case; we need to return the first top-level row

  g_return_val_if_fail (IS_PLAYLIST (tree_model), FALSE);

  playlist = PLAYLIST(tree_model);

  // No rows => no first row
  if (playlist->num_trees == 0)
    return FALSE;

  // Set iter to first item in list
  iter->stamp     = playlist->stamp;
  iter->user_data = playlist->rows[0];

  return TRUE;
}


/ *****************************************************************************
 *
 *  playlist_iter_has_child: Returns TRUE or FALSE depending on whether
 *                              the row specified by 'iter' has any children.
 *                              We only have a list and thus no children.
 *
 ***************************************************************************** /

static gboolean
playlist_iter_has_child (GtkTreeModel *tree_model,
                            GtkTreeIter  *iter)
{
  return FALSE;
}


/ *****************************************************************************
 *
 *  playlist_iter_n_children: Returns the number of children the row
 *                               specified by 'iter' has. This is usually 0,
 *                               as we only have a list and thus do not have
 *                               any children to any rows. A special case is
 *                               when 'iter' is NULL, in which case we need
 *                               to return the number of top-level nodes,
 *                               ie. the number of rows in our list.
 *
 ***************************************************************************** /

static gint
playlist_iter_n_children (GtkTreeModel *tree_model,
                             GtkTreeIter  *iter)
{
  PlayList  *playlist;

  g_return_val_if_fail (IS_PLAYLIST (tree_model), -1);
  g_return_val_if_fail (iter == NULL || iter->user_data != NULL, FALSE);

  playlist = PLAYLIST(tree_model);

  // special case: if iter == NULL, return number of top-level rows
  if (!iter)
    return playlist->num_trees;

  return 0; // otherwise, this is easy again for a list
}


/ *****************************************************************************
 *
 *  playlist_iter_nth_child: If the row specified by 'parent' has any
 *                              children, set 'iter' to the n-th child and
 *                              return TRUE if it exists, otherwise FALSE.
 *                              A special case is when 'parent' is NULL, in
 *                              which case we need to set 'iter' to the n-th
 *                              row if it exists.
 *
 ***************************************************************************** /

static gboolean
playlist_iter_nth_child (GtkTreeModel *tree_model,
                            GtkTreeIter  *iter,
                            GtkTreeIter  *parent,
                            gint          n)
{
  PlaylistTree  *record;
  PlayList    *playlist;

  g_return_val_if_fail (IS_PLAYLIST (tree_model), FALSE);

  playlist = PLAYLIST(tree_model);

  // a list has only top-level rows
  if(parent)
    return FALSE;

  // special case: if parent == NULL, set iter to n-th top-level row 

  if( n >= playlist->num_trees )
    return FALSE;

  record = playlist->rows[n];

  g_assert( record != NULL );
  g_assert( record->pos == n );

  iter->stamp = playlist->stamp;
  iter->user_data = record;

  return TRUE;
}


/ *****************************************************************************
 *
 *  playlist_iter_parent: Point 'iter' to the parent node of 'child'. As
 *                           we have a list and thus no children and no
 *                           parents of children, we can just return FALSE.
 *
 **************************************************************************** /

static gboolean
playlist_iter_parent (GtkTreeModel *tree_model,
                         GtkTreeIter  *iter,
                         GtkTreeIter  *child)
{
  return FALSE;
}

PlayList * playlist_new (void) {
    PlayList *newplaylist = (PlayList*) g_object_new (PLAYLIST_TYPE, NULL);
    g_assert( newplaylist != NULL );
    return newplaylist;
}

int playlistAddTree (PlayList *playlist, NodePtr root) {
    GtkTreeIter   iter;
    GtkTreePath  *path;
    PlaylistTree *newrecord;
    gulong        newsize;
    guint         pos;

    g_return_if_fail (IS_PLAYLIST(playlist));

    if (!playlist->trees) {
        playlist->trees = new PlaylistTree (root, 0, 0L);
    } else {
        PlaylistTree * last_tree = trees;
        int id = last_tree->id;
        while (last_tree->next) {
            last_tree = last_tree->next;
            if (last_tree->id > id)
                id = last_tree->id;
        }
        last_tree->next = new PlaylistTree (root, id+1, 0L);
    }
    playlist->num_trees++;

    / * inform the tree view and other interested objects
     *  (e.g. tree row references) that we have inserted
     *  a new row, and where it was inserted * /

    // FIXME: do this for whole tree ??
    path = gtk_tree_path_new();
    gtk_tree_path_append_index(path, playlist->num_trees-1);
    playlist_get_iter(GTK_TREE_MODEL(playlist), &iter, path);
    gtk_tree_model_row_inserted(GTK_TREE_MODEL(playlist), path, &iter);
    gtk_tree_path_free(path);
}
*/
