#include "kimi.h"
#include <glib-object.h>
#include <check.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h> /* For sleep() */
#define STRING_SIZE 25
#define STRING_QUEUE_SIZE 10

Kimi* kimi = NULL;
GPtrArray* modules = NULL;

static void delay(char* mod)
{
    if (mod == 0 || strcmp(mod, "Google") == 0) {
        sleep(1);
    }
}

void setup()
{
    GPtrArray* a = g_ptr_array_new();
   
    char file[BUF_SIZE];
    char file2[BUF_SIZE];
    char file3[BUF_SIZE];

    sprintf(file, "%s/modules/cb", g_get_current_dir());
    g_ptr_array_add(a, file);

    sprintf(file2, "%s/modules/google/", g_get_current_dir());
    //g_ptr_array_add(a, file2);

    sprintf(file3, "%s/modules/", g_get_current_dir());
    g_ptr_array_add(a, file3);

    KimiUICallbacks back;
    kimi = kimi_init(a, back, NULL);
    g_ptr_array_free(a, TRUE);

    modules = kimi_get_modules(kimi);
}

void teardown()
{
    g_ptr_array_free(modules, TRUE);
    kimi_deinit(kimi);
    kimi = NULL;
}

static void remove_all_events()
{
    int i;
    time_t start = kimi_per_get_time_t("19700101");
    time_t end = kimi_per_get_time_t("20380101");

    kimi_get_by_period(start, end, kimi, NULL);

    GPtrArray* events = kimi_get_date_event_list_array(kimi);

    for (i = 0; i < events->len; i++) {
        Event* ev = g_ptr_array_index(events, i);
        kimi_event_remove(ev->id, ev->service_event_id, kimi, NULL);
        delay(ev->service_event_id);
    }
    g_ptr_array_free(events, TRUE);
}

static Module* get_random_module()
{
    return modules->pdata[rand() % modules->len];
}

static const char* get_random_string()
{
    int i;

    static int q_i = 0;
    static char str[STRING_QUEUE_SIZE][STRING_SIZE];
    static const char graph_symbols[] = "1234567890qwertyuiopasdfghjklzxcvbnmQWERTYUIOPASDFGHJKLZXCVBNM";

    char* current = str[q_i];
    q_i++;
    if (q_i >= STRING_QUEUE_SIZE)
        q_i = 0;
   
    for (i = 0; i < STRING_SIZE - 1; i++) {
        current[i] = graph_symbols[rand() % (sizeof(graph_symbols) - 1)];
    }
    current[STRING_SIZE - 1] = '\0';

    return current;
}
static time_t get_random_time()
{
    time_t a = kimi_per_get_time_t("20080101");
    time_t b = kimi_per_get_time_t("20100101");
    return ((b - a) * ((double)rand() / RAND_MAX) + a);
}

static Event* get_random_event(Module* mod)
{
    Event* ev = kimi_event_new(NULL);
    char* mod_str = mod ? mod->service_string : get_random_module()->service_string;

    time_t s = get_random_time();
    kimi_event_fill(ev,
            mod_str,
            NULL,
            get_random_string(),
            get_random_string(),
            get_random_string(),
            s,
            s + 3600,
            NULL,
            NULL);
    return ev;
}

static int compare_strings(char* s1, char* s2)
{
    if (s1 == NULL && s2 == NULL)
        return 1;
    else if (s1 && s2 && strcmp(s1, s2) == 0)
        return 1;
    else 
        return 0;
}

static int compare_events(Event* e1, Event* e2)
{
   return compare_strings(e1->id, e2->id)
       && compare_strings(e1->title, e2->title)
       && compare_strings(e1->description, e2->description)
       && compare_strings(e1->location, e2->location)
       && compare_strings(e1->service_event_id, e2->service_event_id);
}

START_TEST(check_modules)
{
    int i;
    remove_all_events();
    for (i = 0; i < 1; i++) {
        Event* ev = get_random_event(0);
        kimi_store(ev, kimi, NULL);
        delay(ev->service_event_id);
        kimi_event_free(ev, NULL);
    }
    remove_all_events();
    
    time_t start = kimi_per_get_time_t("19700101");
    time_t end = kimi_per_get_time_t("20380101");
    
    kimi_get_by_period(start, end, kimi, NULL);

    GPtrArray* events = kimi_get_date_event_list_array(kimi);

    fail_unless(events->len == 0);
    g_ptr_array_free(events, TRUE);
}
END_TEST

START_TEST(check_modules_store_event) 
{
    int i, j;
    Event* events[modules->len];
    time_t start = kimi_per_get_time_t("19700101");
    time_t end = kimi_per_get_time_t("20380101");

    remove_all_events();
    for (i = 0; i < modules->len; i++) {
        Module* mod = modules->pdata[i];
        
        events[i] = get_random_event(mod);
        kimi_store(events[i], kimi, NULL);
    }

    kimi_get_by_period(start, end, kimi, NULL);
    GPtrArray* events2 = kimi_get_date_event_list_array(kimi);
    fail_unless(events2->len == modules->len);
    
    for (i = 0; i < modules->len; i++) {
        int ok = 0;
        Event* ev = events[i];
        for (j = 0; j < events2->len; i++) {
            Event* ev2 = events2->pdata[j];
            if (compare_events(ev, ev2)) {
                ok = 1;
                break;
            }
        }
        fail_unless(ok == 1);
    }
    
    g_ptr_array_free(events2, TRUE);

    for (i = 0; i < modules->len; i++) 
        kimi_event_free(events[i], NULL);
}
END_TEST

START_TEST(check_modules_update_event)
{
    int i;
    time_t start = kimi_per_get_time_t("19700101");
    time_t end = kimi_per_get_time_t("20380101");
    char buf[BUF_SIZE];

    remove_all_events();
    
    /* Create random events */
    for (i = 0; i < modules->len; i++) {
        Module* mod = modules->pdata[i];
        Event* ev = get_random_event(mod);

        kimi_store(ev, kimi, NULL);
        kimi_event_free(ev, NULL);
    }

    /* Get all events, and update "title" field */
    kimi_get_by_period(start, end, kimi, NULL);
    GPtrArray* events2 = kimi_get_date_event_list_array(kimi);
  
    for (i = 0; i < modules->len; i++) {
        Event* ev = events2->pdata[i];
        sprintf(buf, "test%d", i);
        KIMI_REPLACE_TEXT_FIELD(ev, title, buf);
        kimi_store(ev, kimi, NULL);
    }
    
    /* Get all events again */
    kimi_get_by_period(start, end, kimi, NULL);
    events2 = kimi_get_date_event_list_array(kimi);
    fail_unless(events2->len == modules->len);

    for (i = 0; i < modules->len; i++) {
        Event* ev = events2->pdata[i];
        sprintf(buf, "test%d", i);
        fail_unless(0 == strcmp(buf, ev->title));
    }
}
END_TEST

START_TEST(check_modules_delete_event)
{
    int i;
    
    time_t start = kimi_per_get_time_t("19700101");
    time_t end = kimi_per_get_time_t("20380101");

    remove_all_events();

    /* Create random events */
    for (i = 0; i < modules->len; i++) {
        Module* mod = modules->pdata[i];
        Event* ev = get_random_event(mod);

        kimi_store(ev, kimi, NULL);
        kimi_event_free(ev, NULL);
    }

    /* Delete events */
    kimi_get_by_period(start, end, kimi, NULL);
    GPtrArray* events2 = kimi_get_date_event_list_array(kimi);
    
    for (i = 0; i < modules->len; i++) {
        Event* ev = events2->pdata[i];
        kimi_event_remove(ev->id, ev->service_event_id, kimi, NULL);
    }

    kimi_get_by_period(start, end, kimi, NULL);
    events2 = kimi_get_date_event_list_array(kimi);

    fail_unless(events2->len == 0);

}
END_TEST

START_TEST(check_modules_get_by_period)
{
    int i;
    time_t start = kimi_per_get_time_t("19700101");
    time_t end = kimi_per_get_time_t("20380101");
    for (i = 0; i < modules->len; i++) {
        remove_all_events();
        Module* mod = modules->pdata[i];

        Event* ev = kimi_event_new(NULL);

        Period* per = kimi_per_create_period();
        Rule* rule = kimi_per_create_rule_from_ical("FREQ=DAILY;COUNT=6;INTERVAL=2", 0, 1);
        kimi_per_add_rule(per, rule);

        kimi_event_fill(ev,
                mod->service_string,
                NULL,
                "title",
                "description",
                "location",
                kimi_per_get_time_t("20110101T120000"),
                kimi_per_get_time_t("20110101T140000"),
                per,
                NULL);

        kimi_store(ev, kimi, NULL);
        kimi_get_by_period(start, end, kimi, NULL);
        
        fail_unless(kimi->date_event_list->len == 6);
        kimi_event_free(ev, NULL);
    } 
    remove_all_events();
}
END_TEST

START_TEST(check_modules_store_null_fields)
{
    int i;
    time_t start = kimi_per_get_time_t("19700101");
    time_t end = kimi_per_get_time_t("20380101");

    for (i = 0; i < modules->len; i++) {
        remove_all_events();
        Module* mod = modules->pdata[i];
        Event* ev = get_random_event(mod);
        ev->title = NULL;
        kimi_store(ev, kimi, NULL);
        
        kimi_get_by_period(start, end, kimi, NULL);
        GPtrArray* events2 = kimi_get_date_event_list_array(kimi);

        fail_unless(events2->len == 1);
        fail_unless(0 == strcmp(((Event*)events2->pdata[0])->title, ""));
        kimi_event_free(ev, NULL);
    }
}
END_TEST

Suite* test_suite(void)
{
    Suite* s = suite_create("Modules test");

    TCase* tc_modules = tcase_create("Modules");

    tcase_add_checked_fixture(tc_modules, setup, teardown);
    tcase_set_timeout(tc_modules, 10);
    tcase_add_test(tc_modules, check_modules);
    tcase_add_test(tc_modules, check_modules_store_event);
    tcase_add_test(tc_modules, check_modules_update_event);
    tcase_add_test(tc_modules, check_modules_delete_event);
    tcase_add_test(tc_modules, check_modules_get_by_period);
    tcase_add_test(tc_modules, check_modules_store_null_fields);

    suite_add_tcase(s, tc_modules);
    return s;
}

int main()
{
    g_type_init();
    srand(time(NULL));
    int number_failed;

    Suite* s = test_suite();

    SRunner *sr = srunner_create(s);

    srunner_run_all (sr, CK_NORMAL);
    number_failed = srunner_ntests_failed(sr);
    srunner_free(sr);
    return (number_failed == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}

