/*
 * Copyright (C) 2008-2009 Petrozavodsk State University
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */

/*
  Curl requests
  Author: Epifanov S.A.
  Last Modification Date: 26.08.2009
*/

#include <remote_storage/protocols/curl_request.h>

static void* protocols_myrealloc(void* ptr, size_t size)
{
/* There might be a realloc() out there that doesn't like reallocing
   NULL pointers, so we take care of it here */
    if (ptr)
	return realloc(ptr, size);
    else
	return malloc(size);
}


static size_t protocols_write_memory_callback(void* ptr, size_t size,
					      size_t nmemb, void* mydata)
{
    size_t realsize = size * nmemb;
    struct MemoryStruct* mem = (struct MemoryStruct *) mydata;

    mem->memory =
	protocols_myrealloc(mem->memory, mem->size + realsize + 1);
    if (mem->memory) {
	memcpy(&(mem->memory[mem->size]), ptr, realsize);
	mem->size += realsize;
	mem->memory[mem->size] = 0;
    }
    return realsize;
}


/**
 * Curl header set module
 * @param header pointer to header
 * @return headers pointer to list of headers
 * */

struct curl_slist* protocols_curl_set_header(struct curl_slist* headers, const char* header)
{
    headers = curl_slist_append(headers, header);
    return headers;
}


/**
 * Curl requests module
 * @param url URL for request
 * @param Poiter on Application data structure
 * @param http_header Header of request
 * @param https SSL flag(0 - Don't user SSL, 1 - Use SSL)
 * @param resp_code Response Code storage
 * @return Body of server answer
 * */

char* protocols_curl_get(const char* url, AppData* data,
			 struct curl_slist* headers,
			 int https, int* resp_code)
{
    CURL* curl;
    char* tmp;
    /* Memory for body and header */

    struct MemoryStruct body, header;

    body.memory = NULL;
    body.size = 0;

    header.memory = NULL;
    header.size = 0;


    /* Set Curl */
    curl = curl_easy_init();

    /* Set URL for HTTP-request */
    curl_easy_setopt(curl, CURLOPT_URL, url);
   
    /* Set Proxy with Conic */
#ifdef CONIC

    switch (con_ic_connection_get_proxy_mode(data->connection)) {

    case CON_IC_PROXY_MODE_NONE:
	g_debug("No proxies defined, it is direct connection");
	break;

    case CON_IC_PROXY_MODE_MANUAL:
	g_debug("CON_IC_PROXY_MANUAL: PreCurlSetOpt");
	
	tmp = strdup(con_ic_connection_get_proxy_host(data->connection, CON_IC_PROXY_PROTOCOL_HTTP));
	curl_easy_setopt(curl, CURLOPT_PROXY, tmp);
	curl_easy_setopt(curl, CURLOPT_PROXYPORT, 
			 con_ic_connection_get_proxy_port(data->connection, CON_IC_PROXY_PROTOCOL_HTTP));
        free(tmp);
	break;

    case CON_IC_PROXY_MODE_AUTO:
	g_debug("Proxy auto-config URL %s should be used", 
		con_ic_connection_get_proxy_autoconfig_url(data->connection));
	break;
    }

#else

    curl_easy_setopt(curl, CURLOPT_PROXY, "proxy.karelia.ru");
    curl_easy_setopt(curl, CURLOPT_PROXYPORT, 81);

#endif


    curl_easy_setopt(curl, CURLOPT_TIMEOUT, 30);

    /* Add Header */

    if (headers != NULL) {
	curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers);
    } else {
	curl_easy_setopt(curl, CURLOPT_USERAGENT, "Kimi-0.1");
    }

    /* Set SSL connection */
    if (https == 1) {
	curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0L);
	curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0L);
    }


    /* Set Data Callbacks  */

    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION,
		     protocols_write_memory_callback);
    curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION,
		     protocols_write_memory_callback);

    /* Set Data Callbacks parameters */
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *) &body);
    curl_easy_setopt(curl, CURLOPT_WRITEHEADER, (void *) &header);

    /* Set Auto-Redirect option */
    curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);

    /* Perform request */
    if (curl_easy_perform(curl) != NULL) {
	fprintf(stderr, "Can't perform the request\n");
	return NULL;
    }

    /* Get Response Code */
    curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, resp_code);

    /* Free Curl and Headers */
    curl_slist_free_all(headers);	/* free the header list */
    curl_easy_cleanup(curl);

    free(header.memory);
    return body.memory;
}


/**
 * Curl POST module
 * @param url URL for request
 * @param http_body Body of the posted message
 * @param data pointer on Application Data Structure
 * @param http_header Header of request
 * @param https SSL flag(0 - Don't user SSL, 1 - Use SSL)
 * @param resp_code Response Code storage
 * @return Body of server answer
 * */

char* protocols_curl_post(const char* url, char* send_data, AppData* data,
                          struct curl_slist* headers,
                          int https, int* resp_code)
{
    CURL* curl;
    char* tmp;
    char* redirect_url;
    char* redirect_url_end;
    /* Memory for body and header */

    struct MemoryStruct body, header;

    body.memory = NULL;
    body.size = 0;

    header.memory = NULL;
    header.size = 0;


    /* Set Curl */
    curl = curl_easy_init();

    /* Set URL for HTTP-request */
    curl_easy_setopt(curl, CURLOPT_URL, url);
   
    /* Set Proxy with Conic */
#ifdef CONIC

    switch (con_ic_connection_get_proxy_mode(data->connection)) {

    case CON_IC_PROXY_MODE_NONE:
	g_debug("No proxies defined, it is direct connection");
	break;

    case CON_IC_PROXY_MODE_MANUAL:
	
	tmp = strdup(con_ic_connection_get_proxy_host(data->connection, CON_IC_PROXY_PROTOCOL_HTTP));
	   
	curl_easy_setopt(curl, CURLOPT_PROXY, tmp);

	curl_easy_setopt(curl, CURLOPT_PROXYPORT, 
			 con_ic_connection_get_proxy_port(data->connection, CON_IC_PROXY_PROTOCOL_HTTP));

	break;

    case CON_IC_PROXY_MODE_AUTO:
	g_debug("Proxy auto-config URL %s should be used", 
		con_ic_connection_get_proxy_autoconfig_url(data->connection));
	break;
    }

#else

    curl_easy_setopt(curl, CURLOPT_PROXY, "proxy.karelia.ru");
    curl_easy_setopt(curl, CURLOPT_PROXYPORT, 81);

#endif


    curl_easy_setopt(curl, CURLOPT_TIMEOUT, 30);

    /* Add Header */

    if (headers != NULL) {
	curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers);
    } else {
	curl_easy_setopt(curl, CURLOPT_USERAGENT, "Kimi-0.1");
    }

    /* Add Body and POST option */

    curl_easy_setopt(curl, CURLOPT_POST, 1);
    curl_easy_setopt(curl, CURLOPT_POSTFIELDS, (void *) send_data);
    curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, -1);
    
    /* Set SSL connection */
    if (https == 1) {
	curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0L);
	curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0L);
    }

    /* Set Data Callbacks  */

    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION,
		     protocols_write_memory_callback);
    curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION,
		     protocols_write_memory_callback);

    /* Set Data Callbacks parameters */
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *) &body);
    curl_easy_setopt(curl, CURLOPT_WRITEHEADER, (void *) &header);

    /* Perform request */
    if (curl_easy_perform(curl) != NULL) {
	g_debug("Can't perform the request");
	return NULL;
    }

    g_debug("Get Response Code");

    /* Get Response Code */
    curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, resp_code);

    /* If 302 code returned */

    if(*resp_code == 302){
	
/*XXX: Get Redirect URL from header of the answer */

	redirect_url = strstr(header.memory, "Location:");
	redirect_url = strstr(redirect_url, "http://");
	redirect_url_end = strstr(header.memory, "Content-Type:");
	*(redirect_url_end - 1) = '\0';

	body.memory = NULL;
	body.size = 0;

	/* Send POST again to new URL */
	curl_easy_setopt(curl, CURLOPT_URL, redirect_url);

	if (curl_easy_perform(curl) != NULL) {
	    g_debug("Can't perform the request");
	}
	
    }
    curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, resp_code);
    /* Free Curl and Headers */
    curl_slist_free_all(headers);	/* free the header list */
    curl_easy_cleanup(curl);

#ifdef CONIC    
    free(tmp);
#endif
    free(header.memory);
    return body.memory;

}
