/**
 * kblayout_status_menu_item.c
 * 
 * @author Emil Biserov <biserov@gmail.com>
 * Copyright (c) 2010 Nokia Corporation. All rights reserved.
 * 
 * This code is based on maemo code example is licensed under a 
 * MIT-style license, that can be found in the file called "COPYING" in the same
 * directory as this file.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <libosso.h>
#include <gconf/gconf-client.h>

#include "kblayout_status_menu_item.h"

#define KBLAYOUT_STATUS_MENU_ITEM_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE (obj, KBLAYOUT_TYPE_STATUS_MENU_ITEM, KBLayoutStatusMenuItemPrivate))

#define STATUSMENU_KBLAYOUT_LIBOSSO_SERVICE_NAME "kblayout_status_menu_item"
#define STATUSMENU_KBLAYOUT_AREA_ICON_SIZE    16

#define HILDON_IM_GCONF_DIR "/apps/osso/inputmethod"

#define GC_INT_KBDLEVEL_SHIFTED HILDON_IM_GCONF_DIR "/int_kb_level_shifted"
#define GC_INT_KB_LAYOUT HILDON_IM_GCONF_DIR "/int_kb_layout"

struct _KBLayoutStatusMenuItemPrivate {
    GConfClient* client;
    guint notifyLevel;
    guint notifyLayout;

    gboolean dual;
    GdkPixbuf *layoutIcon;
    GdkPixbuf *layoutIcon2;

    /* Other stuff */
    osso_context_t *osso;
};

HD_DEFINE_PLUGIN_MODULE(KBLayoutStatusMenuItem, kblayout_status_menu_item, HD_TYPE_STATUS_MENU_ITEM)

static void _set_area_icon(KBLayoutStatusMenuItem * item)
{
    KBLayoutStatusMenuItemPrivate *priv = item->priv;
    GdkPixbuf *icon = priv->layoutIcon;

    if (priv->dual) {
        gboolean layout_shifted = gconf_client_get_bool(priv->client, GC_INT_KBDLEVEL_SHIFTED, NULL);
        if (!layout_shifted) {
            icon = priv->layoutIcon2;
        }
    }

    hd_status_plugin_item_set_status_area_icon(HD_STATUS_PLUGIN_ITEM(item), icon);
}

static void _reload_icons(KBLayoutStatusMenuItem *item)
{
    char *kb_layout, *icon_name;

    KBLayoutStatusMenuItemPrivate *priv = item->priv;

    GtkIconTheme *icon_theme = gtk_icon_theme_get_default();

    priv->dual = FALSE;

    kb_layout = gconf_client_get_string (priv->client, GC_INT_KB_LAYOUT, NULL);

    if (priv->layoutIcon != NULL) {
        g_object_unref(priv->layoutIcon);
        priv->layoutIcon2 = NULL;
    }

    icon_name = g_strconcat("status-area-kblayout-", kb_layout, NULL);
    priv->layoutIcon = gtk_icon_theme_load_icon(icon_theme, icon_name,
        STATUSMENU_KBLAYOUT_AREA_ICON_SIZE, GTK_ICON_LOOKUP_NO_SVG, NULL);
    g_free (icon_name);

    if (priv->layoutIcon2 != NULL) {
        g_object_unref(priv->layoutIcon2);
        priv->layoutIcon2 = NULL;
    }

    if (g_strcmp0 (kb_layout, "ru") == 0) {
        priv->dual = TRUE;

        priv->layoutIcon2 = gtk_icon_theme_load_icon(icon_theme, "status-area-kblayout-en",
            STATUSMENU_KBLAYOUT_AREA_ICON_SIZE, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }

    _set_area_icon(item);

    g_free (kb_layout);
}

static void _level_changed_callback(GConfClient* client, guint cnxn_id, GConfEntry *entry, gpointer user_data)
{
    _set_area_icon(user_data);
}

static void _layout_changed_callback(GConfClient* client, guint cnxn_id, GConfEntry *entry, gpointer user_data)
{
    _reload_icons(user_data);
}

static void kblayout_status_menu_item_finalize(GObject * object)
{
    KBLayoutStatusMenuItem *item = KBLAYOUT_STATUS_MENU_ITEM(object);
    KBLayoutStatusMenuItemPrivate *priv = item->priv;

    gconf_client_notify_remove (priv->client, priv->notifyLevel);
    gconf_client_notify_remove (priv->client, priv->notifyLayout);
    gconf_client_remove_dir(priv->client, HILDON_IM_GCONF_DIR, NULL);

    if (priv->layoutIcon != NULL) {
        g_object_unref(priv->layoutIcon);
    }

    if (priv->layoutIcon2 != NULL) {
        g_object_unref(priv->layoutIcon2);
    }

    if (priv->osso) {
        osso_deinitialize(priv->osso);
        priv->osso = NULL;
    }

    G_OBJECT_CLASS(kblayout_status_menu_item_parent_class)->finalize(object);
}

static void kblayout_status_menu_item_class_init(KBLayoutStatusMenuItemClass * class)
{
    GObjectClass *object_class = G_OBJECT_CLASS(class);

    object_class->finalize = kblayout_status_menu_item_finalize;

    g_type_class_add_private(class, sizeof(KBLayoutStatusMenuItemPrivate));
}

static void kblayout_status_menu_item_class_finalize(KBLayoutStatusMenuItemClass * class)
{
}

static void _init_priv(KBLayoutStatusMenuItem *item)
{
    KBLayoutStatusMenuItemPrivate *priv = item->priv;

    priv->client = gconf_client_get_default();

    gconf_client_add_dir(priv->client, HILDON_IM_GCONF_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL);

    priv->notifyLayout = gconf_client_notify_add(priv->client, GC_INT_KB_LAYOUT,
        _layout_changed_callback, item, NULL, NULL);

    priv->notifyLevel = gconf_client_notify_add(priv->client, GC_INT_KBDLEVEL_SHIFTED,
        _level_changed_callback, item, NULL, NULL);

    priv->dual = FALSE;
    priv->layoutIcon = NULL;
    priv->layoutIcon2 = NULL;

    _reload_icons(item);
}

static void kblayout_status_menu_item_init(KBLayoutStatusMenuItem * item)
{
    KBLayoutStatusMenuItemPrivate *priv;

    priv = item->priv = KBLAYOUT_STATUS_MENU_ITEM_GET_PRIVATE(item);

    /* Initialize */
    _init_priv(item);

    priv->osso = osso_initialize(STATUSMENU_KBLAYOUT_LIBOSSO_SERVICE_NAME, "0.2", TRUE, NULL);
}
