/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QString>
#include <QtCore/QDebug>
#include <QtCore/QDir>
#include "themepicker.h"
#include "themescheduler.h"
#include "themeloader.h"
#include "settings.h"

ThemePicker::ThemePicker(QString const& text, QWidget* parent):
FileSelector(text, parent)
{
}

void ThemePicker::setVisible(bool visible)
{
    if(visible)
    {
        loadThemes();
    }

    FileSelector::setVisible(visible);
}

void ThemePicker::loadThemes()
{
    QString current = Settings::instance().value("theme", ThemeScheduler::getDefaultTheme()).toString();
    clear();
    addItem(tr("Default"), ThemeScheduler::getDefaultTheme());

    QDir themeDir(ThemeLoader::getThemeDir());

    if(!themeDir.exists() || !themeDir.isReadable())
    {
        qDebug() << "Warning: theme dir (" + ThemeLoader::getThemeDir() + ") doesn't exist or is read protected";
        return;
    }

    QStringList filters;
    filters << "*" + ThemeLoader::getThemeSuffix();
    themeDir.setNameFilters(filters);
    themeDir.setFilter(QDir::Files);
    QStringList files = themeDir.entryList();

    for(int i = 0; i < files.size(); i++)
    {
        QString name;
        QString id;
        getThemeDetails(files.at(i), name, id);
        addItem(name, id);

        if(id == current)
        {
            setCurrentIndex(i + 1);
        }
    }
}

bool ThemePicker::selectTheme(QString const& filename)
{
    QString name;
    QString id;
    getThemeDetails(filename, name, id);
    return selectByValue(id);
}

void ThemePicker::getThemeDetails(QString filename,
                                  QString& name,
                                  QString& id)
{
    static QRegExp cleaner(QRegExp::escape(ThemeLoader::getThemeSuffix()) + "$");
    filename = filename.replace(cleaner, "");

    id = filename;
    name = filename.at(0).toUpper() + filename.mid(1);
}
