/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QString>
#include <QtCore/QDebug>
#include <QtGui/QGraphicsScene>
#include <QtGui/QGraphicsPixmapItem>
#include "imageelement.h"
#include "reader.h"
#include "graphicsscene.h"

namespace
{
    const GraphicsElement::AttributeDetails ATTRIBUTES[ImageElement::ATTRIBUTE_COUNT] =
    {
     {"xpos", true},
     {"ypos", true},
     {"zpos", true},
     {"visiblewhen", false},
     {"src", false},
     {"width", true},
     {"height", true}
    };
}

ImageElement::ImageElement(Reader* reader): GraphicsElement(reader),
width_(0), height_(0), imageSet_(false)
{
    element_ = new QGraphicsPixmapItem();
}

bool ImageElement::setAttribute(QString const& name,
                                QString const& value)
{
    int intVal = 0;
    int attrId = -1;

    if((attrId = getAttribute(name, value, ATTRIBUTES, ATTRIBUTE_COUNT, intVal)) != -1)
    {
        Attribute attr = static_cast<Attribute>(attrId);

        switch(attr)
        {
        case XPOS:
            element_->setX(intVal);
            break;
        case YPOS:
            element_->setY(intVal);
            break;
        case ZPOS:
            element_->setZValue(intVal);
            break;
        case VISIBLEWHEN:
            setVisibleWhen(strToVisibleWhen(value));
            break;
        case SRC:
            return loadImage(value);
            break;
        case WIDTH:
            width_ = intVal;
            break;
        case HEIGHT:
            height_ = intVal;
            break;
        default:
            qDebug() << "Unknown attribute: " << attr;
            return false;
        }

        return true;
    }
    else
    {
        return false;
    }
}

void ImageElement::addToScene(GraphicsScene* scene)
{
    if(!imageSet_)
    {
        return;
    }

    QPixmap pix;

    if(width_ > 0 && height_ > 0)
    {
        pix = pixmap_.scaled(width_, height_, Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
    }
    else if(width_ > 0)
    {
        pix = pixmap_.scaledToWidth(width_, Qt::SmoothTransformation);
    }
    else if(height_ > 0)
    {
        pix = pixmap_.scaledToHeight(height_, Qt::SmoothTransformation);
    }
    else
    {
        pix = pixmap_;
    }

    element_->setPixmap(pix);
    scene->addItem(element_);
}

void ImageElement::update()
{
}

bool ImageElement::loadImage(QString const& name)
{
    QByteArray data;

    if(!readFile(name, data))
    {
        return false;
    }

    if(!pixmap_.loadFromData(data))
    {
        setError("Invalid image file: " + name);
        return false;
    }

    imageSet_ = true;

    return true;
}

QGraphicsItem* ImageElement::getElement() const
{
    return element_;
}
