/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDir>
#include <QtCore/QString>
#include <QtCore/QDebug>
#include <QtCore/QTimer>
#include <QtGui/QStackedWidget>
#include <QtXml/QDomNode>
#include <QtXml/QDomDocument>
#include <QtXml/QDomAttr>
#include <QMaemo5InformationBox>
#include "themeloader.h"
#include "reader.h"
#include "zipreader.h"
#include "filereader.h"
#include "themescreen.h"
#include "detailscreen.h"
#include "settings.h"
#include "widgetscreen.h"
#include "themescheduler.h"

namespace
{
    QString const RESOURCE_DIR = ":/resources/themes/";
    QString const THEME_SUFFIX = ".jspeed";
}

ThemeLoader::ThemeLoader(WidgetScreen* mainScreen, WidgetScreen* detailScreen, QObject* parent): QObject(parent),
reader_(0), mainScreen_(mainScreen), detailScreen_(detailScreen)
{
}

ThemeLoader::~ThemeLoader()
{
}

bool ThemeLoader::load()
{
    if(theme_.isEmpty())
    {
        connect(&(ThemeScheduler::instance()), SIGNAL(themeChanged()), this, SLOT(load()));
    }

    QString theme = ThemeScheduler::instance().currentTheme();

    if(theme == theme_)
    {
        return true;
    }

    theme_ = theme;

    QString themeDir = getThemeDir();

    if(QFile::exists(themeDir + theme_ + THEME_SUFFIX))
    {
        reader_ = new ZipReader(themeDir + theme_ + THEME_SUFFIX);

        if(read())
        {
            return true;
        }
        else
        {
            QMaemo5InformationBox::information(mainScreen_,
                                               tr("Unable to load theme: %1.").arg(error_),
                                               QMaemo5InformationBox::NoTimeout);
        }
    }

    theme_ = ThemeScheduler::instance().getDefaultTheme();
    Settings::instance().setValue("theme", theme_);

    if(QFile::exists(RESOURCE_DIR + theme_))
    {
        if(reader_)
        {
            delete reader_;
        }

        reader_ = new FileReader(RESOURCE_DIR + theme_);
        return read();
    }

    error_ = "No themes found";

    return false;
}

bool ThemeLoader::read()
{
    Q_ASSERT(reader_ != 0);

    if(!reader_->open())
    {
        error_ = reader_->errorString();
        return false;
    }

    QByteArray xmlData;

    if(!reader_->readFile("theme.xml", xmlData))
    {
        error_ = "Unable to find <b>theme.xml</b> from theme file";
        return false;
    }

    QDomDocument doc;
    int line = 0;
    int column = 0;
    QString msg;

    if(!doc.setContent(xmlData, false, &msg, &line, &column))
    {
        error_ = "Invalid xml file, " + msg + " (line " + QString::number(line)+", column " + QString::number(column) + ")";
        return false;
    }

    detailScreen_->clear();
    detailScreen_->addScreen(new DetailScreen, WidgetScreen::LANDSCAPE);
    detailScreen_->addScreen(new DetailScreen, WidgetScreen::PORTRAIT);
    loadScreen("detailscreen", doc, detailScreen_);

    mainScreen_->clear();
    mainScreen_->addScreen(new ThemeScreen, WidgetScreen::LANDSCAPE);
    mainScreen_->addScreen(new ThemeScreen, WidgetScreen::PORTRAIT);
    bool ret = loadScreen("mainscreen", doc, mainScreen_);
    mainScreen_->removeUnloaded();

    detailScreen_->reArrange();
    mainScreen_->reArrange();

    reader_->close();

    return ret;
}

bool ThemeLoader::loadScreen(QString const& tag, QDomDocument const& doc, WidgetScreen* screen)
{
    QDomNodeList orientations = doc.elementsByTagName(tag);

    if(orientations.isEmpty())
    {
        error_ = "No <"+tag+"> tags found";
        return false;
    }

    bool ok = true;

    for(int i = 0; i < orientations.size(); i++)
    {
        QDomNode data = orientations.at(i);
        QString type = data.attributes().namedItem("orientation").toAttr().value();
        QDomNode color = data.attributes().namedItem("color");

        if(type == "landscape")
        {
            if(screen->orientationLoaded(WidgetScreen::LANDSCAPE))
            {
                error_ = "More than one <"+tag+" orientation=\"landscape\"> specified";
                ok = false;
            }

            ok = ok && screen->load(WidgetScreen::LANDSCAPE, data, reader_);

            if(ok && !color.isNull())
            {
                screen->setColor(WidgetScreen::LANDSCAPE, color.toAttr().value());
            }

        }
        else if(type == "portrait")
        {
            if(screen->orientationLoaded(WidgetScreen::PORTRAIT))
            {
                error_ = "More than one <"+tag+" orientation=\"portrait\"> specified";
                ok = false;
            }

            ok = ok && screen->load(WidgetScreen::PORTRAIT, data, reader_);

            if(ok && !color.isNull())
            {
                screen->setColor(WidgetScreen::PORTRAIT, color.toAttr().value());
            }
        }
        else if(type == "both")
        {
            if(screen->orientationLoaded(WidgetScreen::PORTRAIT) ||
               screen->orientationLoaded(WidgetScreen::LANDSCAPE))
            {
                error_ = "Multiple orientations specified for same screen";
                ok = false;
            }

            ok = ok && screen->load(data, reader_);

            if(ok && !color.isNull())
            {
                screen->setColor(color.toAttr().value());
            }
        }
        else
        {
            error_ = "Invalid orientation: " + type;
            ok = false;
        }
    }

    return ok;
}

QString ThemeLoader::getThemeDir()
{
    return Settings::getDir() + "themes" + QDir::separator();
}

QString const& ThemeLoader::getThemeSuffix()
{
    return THEME_SUFFIX;
}

QString const& ThemeLoader::error() const
{
    return error_;
}

