/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDebug>
#include "location.h"

namespace
{
    static const double KM_MULTIPLIER = 1.0;
    static const double MILE_MULTIPLIER = 0.621371192;
}

Location::Unit Location::unit_ = Location::KM;

Location::Location(QObject* parent): QObject(parent), started_(false),
control_(0), device_(0)
{
    g_type_init();
}

Location::~Location()
{
    end();
}

void Location::start()
{
    control_ = location_gpsd_control_get_default();
    device_ = (LocationGPSDevice*) g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);

    g_object_set(G_OBJECT(control_),
                 "preferred-method", LOCATION_METHOD_USER_SELECTED,
                 "preferred-interval", LOCATION_INTERVAL_DEFAULT,
                 NULL);

    g_signal_connect(control_, "error-verbose", G_CALLBACK(onError), this);
    g_signal_connect(device_, "changed", G_CALLBACK(onChanged), this);

    location_gpsd_control_start(control_);

    started_ = true;
}

void Location::end()
{
    if(!started_)
    {
        return;
    }

    location_gpsd_control_stop(control_);

    g_object_unref(device_);
    device_ = 0;
    g_object_unref(control_);
    control_ = 0;

    started_ = false;
}

bool Location::hasFix() const
{
    if(!started_)
    {
        return false;
    }

    return (device_->status == LOCATION_GPS_DEVICE_STATUS_FIX);
}

double Location::getSignalStrength() const
{
    if(!hasFix())
    {
        return 0.0;
    }

    if(device_->satellites_in_view == 0)
    {
        return 0.0;
    }

    double val = (device_->satellites_in_use / static_cast<double>(device_->satellites_in_view)) * 100.0;

    if(val > 100.0)
    {
        val = 100.0;
    }

    return val;
}

void Location::setUnit(Location::Unit unit)
{
    unit_ = unit;
}

Location::Unit Location::getUnit()
{
    return unit_;
}

double Location::getUnitMultiplier()
{
    if(unit_ == MILE)
    {
        return MILE_MULTIPLIER;
    }

    return KM_MULTIPLIER;
}

void Location::onChanged(LocationGPSDevice *device, gpointer data)
{
    if(device && device->fix &&
       !(device->fix->fields & LOCATION_GPS_DEVICE_NONE_SET))
    {

        Fix fix;
        fix.unit = unit_;
        fix.time = device->fix->time;
        fix.ept = device->fix->ept;

        if (device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET)
        {
            fix.latitude = device->fix->latitude;
            fix.longitude = device->fix->longitude;
            fix.eph = device->fix->eph;
        }

        if(device->fix->fields & LOCATION_GPS_DEVICE_ALTITUDE_SET)
        {
            fix.altitude = device->fix->altitude;
            fix.epv = device->fix->epv;
        }

        if(device->fix->fields & LOCATION_GPS_DEVICE_SPEED_SET)
        {
            fix.kmSpeed = device->fix->speed;
            fix.speed = device->fix->speed * getUnitMultiplier();
            fix.eps = device->fix->eps;
        }

        if(device->fix->fields & LOCATION_GPS_DEVICE_TRACK_SET)
        {
            fix.track = device->fix->track;
            fix.epd = device->fix->epd;
        }

        if(device->fix->fields & LOCATION_GPS_DEVICE_CLIMB_SET)
        {
            fix.climb = device->fix->climb;
            fix.epc = device->fix->epc;
        }

        Location* meh = (Location*) data;
        meh->emit locationChanged(fix);
    }
}

void Location::onError(LocationGPSDControl *control, LocationGPSDControlError error, gpointer data)
{
    Q_UNUSED(control);

    Error err = SYSTEM_ERROR;

    switch(error)
    {
    case LOCATION_ERROR_USER_REJECTED_DIALOG:
        err = USER_REJECTED_DIALOG;
        break;
    case LOCATION_ERROR_USER_REJECTED_SETTINGS:
        err = USER_REJECTED_SETTINGS;
        break;
    case LOCATION_ERROR_BT_GPS_NOT_AVAILABLE:
        err = GPS_NOT_AVAILABLE;
        break;
    case LOCATION_ERROR_METHOD_NOT_ALLOWED_IN_OFFLINE_MODE:
        err = NOT_ALLOWED_IN_OFFLINE_MODE;
        break;
    case LOCATION_ERROR_SYSTEM:
        err = SYSTEM_ERROR;
        break;
    }

    Location* meh = (Location*) data;
    meh->emit locationError(err);
}

Location::Fix::Fix()
{
    unit = KM;
    kmSpeed = 0.0;
    time = 0.0;
    ept = 0.0;
    latitude = 0.0;
    longitude = 0.0;
    eph = 0.0;
    altitude = 0.0;
    epv = 0.0;
    track = 0.0;
    epd = 0.0;
    speed = 0.0;
    eps = 0.0;
    climb = 0.0;
    epc = 0.0;
}
