/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtGui/QPushButton>
#include <QtGui/QVBoxLayout>
#include <QtGui/QHBoxLayout>
#include <QtGui/QMessageBox>
#include <QtGui/QLabel>
#include <QtGui/QPixmap>
#include "mainmenu.h"
#include "themeselector.h"
#include "unitselector.h"
#include "poisettings.h"
#include "speedalarmsettings.h"

MainMenu::MainMenu(QWidget* parent): QDialog(parent), itemCount_(0),
currentRow_(0), themeSelector_(0), unitSelector_(0), poiSettings_(0),
speedAlarmSettings_(0), aboutDialog_(0)
{
    layout_ = new QVBoxLayout;
    setLayout(layout_);

    setWindowTitle(tr("Menu"));
    addAction(tr("Reset trip"), this, SIGNAL(resetTrip()));
    addAction(tr("Reset all"), this, SLOT(confirmReset()));
    addAction(tr("Theme"), this, SLOT(selectTheme()));
    addAction(tr("Set unit"), this, SLOT(selectUnit()));
    addAction(tr("Speed camera alerts"), this, SLOT(openPoiAlerts()));
    addAction(tr("Speed alarm"), this, SLOT(openSpeedAlarm()));
    addAction(tr("Flip screen"), this, SIGNAL(flip()));
    addAction(tr("About"), this, SLOT(showAbout()));
}

void MainMenu::selectTheme()
{
    if(!themeSelector_)
    {
        themeSelector_ = new ThemeSelector(this);
        connect(themeSelector_, SIGNAL(themeChanged()), this, SIGNAL(themeChanged()));
    }

    themeSelector_->show();
}

void MainMenu::openPoiAlerts()
{
    if(!poiSettings_)
    {
        poiSettings_ = new PoiSettings(this);
    }

    poiSettings_->show();
}

void MainMenu::openSpeedAlarm()
{
    if(!speedAlarmSettings_)
    {
        speedAlarmSettings_ = new SpeedAlarmSettings(this);
    }

    speedAlarmSettings_->show();
}

void MainMenu::selectUnit()
{
    if(!unitSelector_)
    {
        unitSelector_ = new UnitSelector(this);
        connect(unitSelector_, SIGNAL(unitChanged()), this, SIGNAL(unitChanged()));
    }

    unitSelector_->show();
}

void MainMenu::showAbout()
{
    if(!aboutDialog_)
    {
        aboutDialog_ = new QDialog(this);
        aboutDialog_->setWindowTitle(tr("About"));

        QHBoxLayout* mainLayout = new QHBoxLayout;
        QPixmap pixmap(":/resources/appicon.png");
        QLabel* image = new QLabel;
        image->setPixmap(pixmap);
        mainLayout->addWidget(image, 0, Qt::AlignLeft);
        mainLayout->addSpacing(15);
        QLabel* text = new QLabel(QString::fromUtf8("<font size='4'><b>jSpeed</b></font><br>(c) 2010 Jesse Hakanen<br><a href='http://jspeed.garage.maemo.org' style='color:#ffffff'>http://jspeed.garage.maemo.org</a>"));
        text->setOpenExternalLinks(true);
        mainLayout->addWidget(text, 10, Qt::AlignLeft);

        aboutDialog_->setLayout(mainLayout);
    }

    aboutDialog_->show();
}

void MainMenu::confirmReset()
{
    hide();

    QMessageBox::StandardButton result =  QMessageBox::question(this, tr("Reset all"),
             tr("Are you sure? All recorded data will be lost."),
             QMessageBox::Yes | QMessageBox::No);

    if(result == QMessageBox::Yes)
    {
        emit resetAll();
    }
}

void MainMenu::addAction(const QString& text,
                         const QObject* receiver,
                         const char* member)
{
    if(itemCount_ % 2 == 0 || !currentRow_)
    {
        currentRow_ = new QHBoxLayout;
        layout_->addLayout(currentRow_);
    }

    QPushButton* button = new QPushButton(text);
    connect(button, SIGNAL(clicked(bool)), receiver, member);
    connect(button, SIGNAL(clicked(bool)), this, SLOT(hide()));
    currentRow_->addWidget(button);
    itemCount_++;
}
