/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>
#include <QtCore/QString>
#include <QtCore/QDir>
#include <QtCore/QStringList>
#include "fileselector.h"

FileSelector::FileSelector(QString const& text, QWidget* parent): ButtonSelector(text, parent)
{
}

bool FileSelector::loadFiles(QString const& dir, QStringList const& patterns)
{
    QDir directory(dir);

    if(directory.exists() && directory.isReadable())
    {
        directory.setNameFilters(patterns);
        directory.setFilter(QDir::Files);
        QStringList files = directory.entryList();

        for(int i = 0; i < files.size(); i++)
        {
            addItem(files.at(i));
        }

        return true;
    }

    return false;
}

bool FileSelector::loadFiles(QString const& dir, QString const& pattern)
{
    return loadFiles(dir, QStringList() << pattern);
}

bool FileSelector::importFile(QString const& dir,
                              QString const& name,
                              QString const& pattern,
                              bool add,
                              QString* addedFile)
{
    QString filename = QFileDialog::getOpenFileName(this, tr("Open file"),
                                                    QDir::home().path(),
                                                    name + "(" + pattern + ")");

    if(filename.isEmpty())
    {
        return true;
    }

    QString basename;

    int i = filename.lastIndexOf(QDir::separator());

    if(i == -1)
    {
        basename = filename;
    }
    else
    {
        basename = filename.mid(i + 1);
    }

    QString targetFile = dir + basename;

    QDir targetDir(dir);

    if(!targetDir.exists())
    {
        if(!targetDir.mkpath(dir))
        {
            return false;
        }
    }

    bool ok = true;

    if(QFile::exists(targetFile))
    {
        ok = confirmCopy(basename);

        if(ok)
        {
            QFile::remove(targetFile);
        }
    }

    if(ok)
    {
        if(!QFile::copy(filename, targetFile))
        {
            return false;
        }

        if(addedFile)
        {
            *addedFile = basename;
        }

        if(add)
        {
            addItem(basename);
            selectByValue(basename);
        }

        return true;
    }
    else
    {
        return false;
    }
}

bool FileSelector::confirmCopy(QString const& filename)
{
    QMessageBox::StandardButton result =  QMessageBox::question(this, tr("File exists"),
                                                                tr("File %1 already exists in directory, overwrite?").arg(filename),
                                                                QMessageBox::Yes | QMessageBox::No);

    return (result == QMessageBox::Yes);
}

