/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QString>
#include <QtCore/QByteArray>
#include <QtCore/QDebug>
#include "zipreader.h"

namespace
{
    int const BUFFER_SIZE = 16384;
}

ZipReader::ZipReader(QString const& filename): Reader(),
filename_(filename), zip_(0), error_("")
{
}

bool ZipReader::open()
{
    int error = 0;

    zip_ = zip_open(filename_.toLatin1().data(), 0, &error);

    if(error)
    {
        error_ = "Not a valid zip file";
        zip_ = 0;
        return false;
    }

    return true;
}

bool ZipReader::readFile(QString const& filename, QByteArray& data)
{
    zip_file* file = 0;

    file = zip_fopen(zip_, filename.toLatin1().data(), 0);

    if(!file)
    {
        error_ = QString(zip_strerror(zip_));
        return false;
    }

    char buffer[BUFFER_SIZE];

    int read = 0;

    data.clear();

    while((read = zip_fread(file, buffer, BUFFER_SIZE)) > 0)
    {
        data.append(buffer, read);
    }

    return true;
}

bool ZipReader::fileExists(QString const& filename) const
{
    return zip_name_locate(zip_, filename.toLatin1().data(), 0) != -1;
}

bool ZipReader::close()
{
    if(zip_close(zip_) == 0)
    {
        zip_ = 0;
        return true;
    }

    return false;
}

QString const& ZipReader::errorString() const
{
    return error_;
}
