/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDir>
#include <QtCore/QDebug>
#include <QtCore/QString>
#include <QtCore/QStringList>
#include <QtGui/QFileDialog>
#include <QtGui/QDialogButtonBox>
#include <QtGui/QPushButton>
#include <QtGui/QHBoxLayout>
#include <QtGui/QVBoxLayout>
#include <QtGui/QMessageBox>
#include <QMaemo5InformationBox>
#include "themeselector.h"
#include "themepicker.h"
#include "themeloader.h"
#include "settings.h"
#include "themeschedulersettings.h"

ThemeSelector::ThemeSelector(QWidget* parent): QDialog(parent), themeScheduler_(0)
{
    setWindowTitle(tr("Select theme"));

    QHBoxLayout* layout = new QHBoxLayout;
    QVBoxLayout* left = new QVBoxLayout;
    QHBoxLayout* first = new QHBoxLayout;

    QPushButton* saveButton = new QPushButton(tr("Save"));
    connect(saveButton, SIGNAL(clicked(bool)), this, SLOT(saveTheme()));
    QDialogButtonBox* buttons = new QDialogButtonBox;
    buttons->setCenterButtons(false);
    buttons->setOrientation(Qt::Vertical);
    buttons->addButton(saveButton, QDialogButtonBox::AcceptRole);

    selector_ = new ThemePicker(tr("Theme"), this);
    connect(selector_, SIGNAL(selected()), this, SLOT(disableScheduler()));

    QPushButton* loadButton = new QPushButton(tr("Import"));
    connect(loadButton, SIGNAL(clicked(bool)), this, SLOT(loadFromFile()));

    QPushButton* scheduler = new QPushButton(tr("Theme scheduler"));
    connect(scheduler, SIGNAL(clicked(bool)), this, SLOT(openScheduler()));

    first->addWidget(selector_, Qt::AlignLeft);
    first->addWidget(loadButton);

    left->addLayout(first);
    left->addWidget(scheduler);

    layout->addLayout(left, Qt::AlignLeft);
    layout->addWidget(buttons);

    setLayout(layout);

}

void ThemeSelector::saveTheme()
{
    QString theme = selector_->value().toString();
    Settings::instance().setValue("theme", theme);
    hide();
    emit themeChanged();
}

void ThemeSelector::loadFromFile()
{
    QString file;

    if(selector_->importFile(ThemeLoader::getThemeDir(),
                          "Theme files",
                          "*" + ThemeLoader::getThemeSuffix(),
                          false,
                          &file))
    {
        selector_->loadThemes();
        selector_->selectTheme(file);
    }
}

void ThemeSelector::openScheduler()
{
    if(!themeScheduler_)
    {
        themeScheduler_ = new ThemeSchedulerSettings(this);
        connect(themeScheduler_, SIGNAL(themeChanged()), this, SIGNAL(themeChanged()));
    }

    themeScheduler_->show();
}

void ThemeSelector::disableScheduler()
{
    if(ThemeScheduler::instance().isEnabled())
    {
        QMaemo5InformationBox::information(this, tr("Disabling theme scheduler..."), 1000);
        ThemeScheduler::instance().setEnabled(false);
    }
}
