/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDir>
#include <QtCore/QDebug>
#include <QtGui/QLineEdit>
#include <QtGui/QLabel>
#include <QtGui/QHBoxLayout>
#include <QtGui/QVBoxLayout>
#include <QtGui/QPushButton>
#include <QtGui/QDialogButtonBox>
#include <QtGui/QCheckBox>
#include <QtGui/QIntValidator>
#include <math.h>
#include "speedalarmsettings.h"
#include "speedalarm.h"
#include "odometer.h"
#include "settings.h"
#include "fileselector.h"
#include "poialerts.h"
#include "mediaplayer.h"
#include "soundselector.h"
#include "buttonbox.h"


SpeedAlarmSettings::SpeedAlarmSettings(QWidget* parent): QDialog(parent)
{
    setWindowTitle(tr("Speed alarm"));

    speedLabel_ = new QLabel;
    speed_ = new QLineEdit;
    speed_->setValidator(new QIntValidator(0, 999, this));

    QPushButton* current = new QPushButton(tr("Current"));
    connect(current, SIGNAL(clicked(bool)), this, SLOT(loadCurrentSpeed()));

    QHBoxLayout* speedLayout = new QHBoxLayout;
    speedLayout->addWidget(speedLabel_);
    speedLayout->addWidget(speed_);
    speedLayout->addWidget(current);

    soundSelector_ = new SoundSelector;

    enabled_ = new QCheckBox(tr("Enabled"));
    enabled_->setChecked(Settings::instance().value("alarm_enabled", false).toBool());

    ButtonBox* buttons = new ButtonBox;
    connect(buttons->addButton(tr("Save"), QDialogButtonBox::AcceptRole), SIGNAL(clicked(bool)), this, SLOT(saveSettings()));

    QHBoxLayout* layout = new QHBoxLayout;
    QVBoxLayout* left = new QVBoxLayout;

    left->addLayout(speedLayout);
    left->addLayout(soundSelector_);
    left->addWidget(enabled_);
    layout->addLayout(left, Qt::AlignLeft);
    layout->addWidget(buttons);

    setLayout(layout);
}


void SpeedAlarmSettings::loadData()
{
    speedLabel_->setText(tr("Speed threshold (%1)").arg(Odometer::getSpeedUnit()));
    int speedValue = round(Settings::instance().value("alarm_threshold", 100).toDouble() * Odometer::getUnitMultiplier());
    speed_->setText(QString::number(static_cast<int>(speedValue)));

    QString selected = Settings::instance().value("alarm_sound", "").toString();
    soundSelector_->load();
    soundSelector_->setValue(selected);
}

void SpeedAlarmSettings::loadCurrentSpeed()
{
    double speed = round(Odometer::instance().getLatestFix().speed);
    speed_->setText(QString::number(static_cast<int>(speed)));
}

void SpeedAlarmSettings::saveSettings()
{
    double kmSpeed = speed_->text().toInt() / Odometer::getUnitMultiplier();

    Settings::instance().setValue("alarm_threshold", kmSpeed);
    Settings::instance().setValue("alarm_enabled", enabled_->isChecked());
    Settings::instance().setValue("alarm_sound", soundSelector_->value());

    SpeedAlarm::instance().loadConfig();

    hide();
}

void SpeedAlarmSettings::setVisible(bool visible)
{
    if(visible)
    {
        loadData();
    }

    QDialog::setVisible(visible);
}
