/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtGui/QGraphicsRectItem>
#include <QtCore/QString>
#include <QtCore/QDebug>
#include "rectangle.h"
#include "reader.h"
#include "graphicsscene.h"

namespace
{
    const GraphicsElement::AttributeDetails ATTRIBUTES[Rectangle::ATTRIBUTE_COUNT] =
    {
     {"xpos", true},
     {"ypos", true},
     {"zpos", true},
     {"visiblewhen", false},
     {"width", true},
     {"height", true},
     {"color", false}
    };
}

Rectangle::Rectangle(Reader* reader): GraphicsElement(reader),
x_(0), y_(0), width_(0), height_(0)
{
    element_ = new QGraphicsRectItem();
}

bool Rectangle::setAttribute(QString const& name, QString const& value)
{
    int intVal = 0;
    int attrId = -1;

    if((attrId = getAttribute(name, value, ATTRIBUTES, ATTRIBUTE_COUNT, intVal)) != -1)
    {
        Attribute attr = static_cast<Attribute>(attrId);

        switch(attr)
        {
        case XPOS:
            x_ = intVal;
            break;
        case YPOS:
            y_ = intVal;
            break;
        case ZPOS:
            element_->setZValue(intVal);
            break;
        case VISIBLEWHEN:
            setVisibleWhen(strToVisibleWhen(value));
            break;
        case WIDTH:
            width_ = intVal;
            break;
        case HEIGHT:
            height_ = intVal;
            break;
        case COLOR:
            element_->setBrush(QBrush(QColor(value)));
            break;

        default:
            qDebug() << "Unknown attribute: " << attr;
            return false;
        }

        return true;
    }
    else
    {
        return false;
    }
}

void Rectangle::addToScene(GraphicsScene* scene)
{
    int width = width_;

    int maxSize = scene->width();

    if(scene->height() > maxSize)
    {
        maxSize = scene->height();
    }

    if(width <= 0)
    {
        width = maxSize;
    }

    int height = height_;

    if(height <= 0)
    {
        height = maxSize;
    }

    element_->setRect(x_, y_, width, height);
    scene->addItem(element_);
}

void Rectangle::update()
{
}

QGraphicsItem* Rectangle::getElement() const
{
    return element_;
}
