/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDebug>
#include <QtGui/QApplication>
#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusMessage>
#include <QtDBus/QDBusReply>
#include <QtDBus/QDBusArgument>
#include <QtDBus/QDBusMetaType>
#include "ovimaps.h"

namespace
{
    QString const MAPS_SERVICE = "com.nokia.Navigation.NokiaMapsProvider";
    QString const MAPS_PATH = "/Provider";
    QString const MAPS_INTERFACE = "com.nokia.Navigation.MapProvider";
    QString const MAPS_ADDRESS_TO_LOCATIONS = "AddressToLocations";
    QString const MAPS_ADDRESS_TO_LOCATIONS_REPLY = "AddressToLocationsReply";
    QString const MAPS_LOCATION_TO_ADDRESSES = "LocationToAddresses";
    QString const MAPS_LOCATION_TO_ADDESSES_REPLY = "LocationToAddressReply";
    QString const MAPS_START = "ShowPlaceGeo";

}

inline QDBusArgument& operator<<(QDBusArgument& argument,
                                 const OviMaps::Location& location)
{
    argument.beginStructure();
    argument << location.latitude << location.longitude;
    argument.endStructure();
    return argument;
}

inline const QDBusArgument& operator>>(const QDBusArgument& argument,
                                       OviMaps::Location& location)
{
    argument.beginStructure();
    argument >> location.latitude >> location.longitude;
    argument.endStructure();
    return argument;
}

Q_DECLARE_METATYPE(OviMaps::Location);
Q_DECLARE_METATYPE(QList<OviMaps::Location>);
Q_DECLARE_METATYPE(QList<QStringList>);

OviMaps::OviMaps(QObject* parent): QObject(parent),
locationsTarget_(0), addressesTarget_(0), ready_(false), timer_(0)
{
    qDBusRegisterMetaType<OviMaps::Location>();
    qDBusRegisterMetaType< QList<OviMaps::Location> >();
    qDBusRegisterMetaType< QList<QStringList> >();
}

bool OviMaps::addressToLocations(OviMaps::Address const& address,
                                 QList<OviMaps::Location>& locations)
{
    QDBusMessage msg = QDBusMessage::createMethodCall(MAPS_SERVICE,
                                                      MAPS_PATH,
                                                      MAPS_INTERFACE,
                                                      MAPS_ADDRESS_TO_LOCATIONS);

    QList<QVariant> arguments;

    QStringList data;

    data << address.number << "" << address.street << ""
         << address.city << "" << "" << address.zipCode << address.country
         << "" << "" << "" << "" << "" << "";

    arguments.append(QVariant(data));
    arguments.append(QVariant(true));

    msg.setArguments(arguments);

    QDBusMessage rep = QDBusConnection::sessionBus().call(msg);

    QDBusReply<QDBusObjectPath> reply(rep);

    if(reply.isValid())
    {
        QDBusObjectPath path = reply.value();
        locationsTarget_ = &locations;
        QDBusConnection::sessionBus().connect(MAPS_SERVICE,
                                              path.path(),
                                              MAPS_INTERFACE,
                                              MAPS_ADDRESS_TO_LOCATIONS_REPLY,
                                              this,
                                              SLOT(handleAddressToLocationsReply(QList<OviMaps::Location>)));

        bool ret = waitSignal();
        locationsTarget_ = 0;
        return ret;
    }

    return false;

}

bool OviMaps::locationToAddresses(Location const& location, QList<Address>& address)
{
    QDBusMessage msg = QDBusMessage::createMethodCall(MAPS_SERVICE,
                                                      MAPS_PATH,
                                                      MAPS_INTERFACE,
                                                      MAPS_LOCATION_TO_ADDRESSES);

    QList<QVariant> arguments;

    arguments.append(QVariant(location.latitude));
    arguments.append(QVariant(location.longitude));
    arguments.append(QVariant(false));

    msg.setArguments(arguments);

    QDBusMessage rep = QDBusConnection::sessionBus().call(msg);

    QDBusReply<QDBusObjectPath> reply(rep);

    if(reply.isValid())
    {
        QDBusObjectPath path = reply.value();
        addressesTarget_ = &address;
        QDBusConnection::sessionBus().connect(MAPS_SERVICE,
                                              path.path(),
                                              MAPS_INTERFACE,
                                              MAPS_LOCATION_TO_ADDESSES_REPLY,
                                              this,
                                              SLOT(handleLocationToAddressesReply(QList<QStringList>)));

        bool ret = waitSignal();
        addressesTarget_ = 0;
        return ret;
    }

    return false;
}

bool OviMaps::openMaps(Address const& address)
{
    QList<Location> locations;

    if(!addressToLocations(address, locations) || locations.size() == 0)
    {
        return false;
    }

    return openMaps(locations.at(0));
}

bool OviMaps::openMaps(Location const& location)
{
    QDBusMessage msg = QDBusMessage::createMethodCall(MAPS_SERVICE,
                                                      MAPS_PATH,
                                                      MAPS_INTERFACE,
                                                      MAPS_START);

    QList<QVariant> arguments;

    arguments.append(QVariant(location.latitude));
    arguments.append(QVariant(location.longitude));

    uint val = 0;

    arguments.append(QVariant(val));

    msg.setArguments(arguments);

    QDBusMessage rep = QDBusConnection::sessionBus().call(msg);

    if(rep.type() == QDBusMessage::ErrorMessage)
    {
        return false;
    }

    return true;
}

void OviMaps::timerEvent(QTimerEvent* event)
{
    Q_UNUSED(event);

    if(timer_)
    {
        killTimer(timer_);
        timer_ = 0;
    }
}

void OviMaps::handleAddressToLocationsReply(QList<OviMaps::Location> locations)
{
    if(locationsTarget_)
    {
        *locationsTarget_ = locations;
        ready_ = true;
    }
}

void OviMaps::handleLocationToAddressesReply(QList<QStringList> addresses)
{
    if(addressesTarget_)
    {
        for(int i = 0; i < addresses.size(); i++)
        {
            Address addr;
            addr.street = addresses.at(i).at(2);
            addr.number = addresses.at(i).at(0);
            addr.zipCode = addresses.at(i).at(7);
            addr.city = addresses.at(i).at(4);
            addr.country = addresses.at(i).at(8);

            addressesTarget_->push_back(addr);
        }

        ready_ = true;
    }
}

bool OviMaps::waitSignal()
{
    ready_ = false;
    timer_ = startTimer(TIMEOUT);

    while(!ready_ && timer_)
    {
        QApplication::processEvents(QEventLoop::WaitForMoreEvents);
    }

    if(!timer_)
    {
        return false;
    }
    else
    {
        killTimer(timer_);
        timer_ = 0;
    }

    return true;
}
