/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDebug>
#include "contactmanager.h"


ContactManager::ContactManager(): book_(0)
{
}

ContactManager::~ContactManager()
{
    if(book_)
    {
        g_object_unref(book_);
    }
}

bool ContactManager::numberExists(QString const& number)
{
    if(!load())
    {
        return false;
    }

    EBookQuery* query;
    GList *g_contacts;

    // Just check the last 7 digits
    QString clean = number.right(7);

    query = e_book_query_any_field_contains(clean.toLatin1());

    if (!e_book_get_contacts (book_, query, &g_contacts, NULL))
    {
        qDebug() << "Couldn't get query results.\n";
        return false;
    }

    e_book_query_unref(query);

    if (g_contacts == 0)
    {
        return false;
    }

    return true;

}

bool ContactManager::addContact(Contact const& contact)
{
    if(!load())
    {
        return false;
    }

    EContact* newContact = e_contact_new();
    GError* error = NULL;
    EContactAddress* addr = NULL;

    char* firstname = contact.name.firstname.toUtf8().data();
    char* surname = contact.name.surname.toUtf8().data();
    e_contact_set(newContact, E_CONTACT_GIVEN_NAME, (gpointer)firstname);
    e_contact_set(newContact, E_CONTACT_FAMILY_NAME, (gpointer)surname);

    if(!contact.city.isEmpty() || !contact.street.isEmpty())
    {
        addr = g_new0 (EContactAddress, 1);
        addr->address_format = g_strdup("");
        addr->po = g_strdup("");
        addr->ext = g_strdup("");
        addr->region = g_strdup("");
        addr->code = g_strdup(contact.zipCode.toUtf8().data());
        addr->country = g_strdup(contact.country.toUtf8().data());
        addr->street = g_strdup(contact.street.toUtf8().data());
        addr->locality = g_strdup(contact.city.toUtf8().data());
        e_contact_set(newContact, E_CONTACT_ADDRESS_OTHER, (gpointer)addr);
    }

    if(!contact.number.isEmpty())
    {
        char* number = contact.number.toUtf8().data();
        e_contact_set(newContact, E_CONTACT_PHONE_OTHER, (gpointer)number);
    }

    bool ret = true;

    if(!e_book_add_contact(book_, newContact, &error))
    {
        qDebug() << "Couldn't add contact: " <<  error->message;
        g_error_free(error);
        ret = false;
    }

    if(addr)
    {
        e_contact_address_free(addr);
    }

    return ret;
}

void ContactManager::stringToName(QString const& strname, ContactManager::Name& name)
{
    EContactName* ename = e_contact_name_from_string(strname.toUtf8().data());

    if(ename)
    {
        QString additional = QString(ename->additional);

        if(additional.isEmpty())
        {
            name.firstname = QString(ename->given);
            name.surname = QString(ename->family);
        }
        else
        {
            name.surname = strname;
        }

        e_contact_name_free(ename);
    }
    else
    {
        name.surname = strname;
    }
}


bool ContactManager::load()
{
    if(book_)
    {
        return true;
    }

    GError *error;
    error = 0;
    book_ = e_book_new_system_addressbook(&error);

    if (!book_)
    {
        qDebug() << "Couldn't open addressbook: %s" << error->message;
        g_error_free(error);
        return false;
    }

    /* Open connection to the address book */
    if (!e_book_open(book_, FALSE, &error))
    {
        qDebug() << "Couldn't open addressbook: %s" << error->message;
        g_error_free(error);
        return false;
    }

    return true;

}

