/****************************************************************************
**
** Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights.  These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you have questions regarding the use of this file, please contact
** Nokia at qt-info@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

#include <QtGui/QApplication>
#include <QtGui/QDesktopWidget>
#include <QtGui/QLabel>
#include <QtGui/QLayout>
#include <QtGui/QMaemo5Style>
#include <QtGui/QStyleOption>
#include <QtGui/QDesktopWidget>
#include <QtGui/QMouseEvent>
#include <QtGui/QPainter>

#include <QtGui/QX11Info>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

#include "informationbox.h"

class InformationBoxPrivate
{
    Q_DECLARE_PUBLIC(InformationBox)

public:
    InformationBoxPrivate()
        : m_timeout(InformationBox::DefaultTimeout),
          m_timer_id(0), m_layout(0), m_widget(0), q_ptr(0)
    { }

    void enforceInformationType();

    int m_timeout;
    int m_timer_id;
    QBoxLayout *m_layout;
    QWidget *m_widget;
    InformationBox *q_ptr;
};

/*!
    Constructs an information box with no text, a default timeout
    (\c DefaultTimeout) and the given \a parent widget.
*/
InformationBox::InformationBox(QWidget *parent)
    : QDialog(parent, Qt::CustomizeWindowHint), d_ptr(new InformationBoxPrivate())
{
    Q_D(InformationBox);
    d_ptr->q_ptr = this;

    setAttribute(Qt::WA_X11NetWmWindowTypeNotification);
    setAttribute(Qt::WA_ShowWithoutActivating);
    setFocusPolicy(Qt::NoFocus);
    d->m_layout = new QHBoxLayout(this);
    d->m_layout->setContentsMargins(8, 0, 8, 0); // 8 == HILDON_MARGIN_DEFAULT
    setFixedWidth(QApplication::desktop()->screenGeometry().width());
    //move(0, 56 /*HILDON_WINDOW_TITLEBAR_HEIGHT*/);
}


/*!
    Destroys the information box.
*/
InformationBox::~InformationBox()
{ }

/*! \internal
 */
void InformationBoxPrivate::enforceInformationType()
{
    Q_Q(InformationBox);

    const char *type = "_HILDON_NOTIFICATION_TYPE_BANNER";

    Atom atom = XInternAtom(QX11Info::display(), "_HILDON_NOTIFICATION_TYPE", False);

    XChangeProperty(QX11Info::display(), q->winId(), atom, XA_STRING, 8,
                    PropModeReplace, (unsigned char *) type, qstrlen(type));
}


/*!
  \property InformationBox::timeout
  \brief the timeout after which the informaton box should automatically be
  hidden.

  Setting this value while the box is already visible will reset its timer.

  The timeout value can be specified as an arbitrary millisecond value,
  although it is recommended to use the predefined values \c NoTimeout (for
  Hildon Notes) and \c DefaultTimeout (for style guide compliant Banners).

  The default value of this property is \c DefaultTimeout (3 seconds).
*/

int InformationBox::timeout() const
{
    Q_D(const InformationBox);
    return d->m_timeout;
}

void InformationBox::setTimeout(unsigned int ms)
{
    Q_D(InformationBox);
    d->m_timeout = ms;

    // restart the timer
    if (d->m_timeout != InformationBox::NoTimeout
            && d->m_timer_id) {
        killTimer(d->m_timer_id);
        d->m_timer_id = startTimer(d->m_timeout);
    }
}

/*!
    Returns the central widget of the information box.
*/
QWidget *InformationBox::widget() const
{
    Q_D(const InformationBox);
    return d->m_widget;
}

/*!
    Sets the central widget of the information box to \a widget.

    InformationBox will take ownership of \a widget by reparenting
    it.  Any previously set widget() will be destroyed.
*/
void InformationBox::setWidget(QWidget *widget)
{
    Q_D(InformationBox);
    delete d->m_widget;
    d->m_widget = widget;
    if (widget) {
        d->m_layout->addWidget(widget);
        widget->show();
    }
}

/*! \reimp
*/
void InformationBox::showEvent(QShowEvent *)
{
    Q_D(InformationBox);

    d->enforceInformationType();
    if (d->m_timeout != InformationBox::NoTimeout)
        d->m_timer_id = startTimer(d->m_timeout);
}

/*! \reimp
*/
void InformationBox::timerEvent(QTimerEvent *te)
{
    Q_D(InformationBox);

    if (te->timerId() == d->m_timer_id) {
        killTimer(d->m_timer_id);
        hide();
    }
}

/*! \reimp
*/
void InformationBox::mousePressEvent(QMouseEvent *me)
{
    if (me->button() == Qt::LeftButton) {
        hide();
        emit clicked();
    }
}

/*! \reimp
*/
void InformationBox::paintEvent(QPaintEvent *)
{
    if (qobject_cast<QMaemo5Style*>(style())) {
        QPainter painter(this);
        QStyleOption option;
        option.initFrom(this);
        style()->drawPrimitive(static_cast<QStyle::PrimitiveElement>(QMaemo5Style::PE_Maemo5InformationBox),
                               &option, &painter, this);
    }
}

