/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtGui/QLabel>
#include <QtGui/QPushButton>
#include <QtGui/QVBoxLayout>
#include <QtGui/QHBoxLayout>
#include <QtGui/QIntValidator>
#include <QtGui/QDialogButtonBox>
#include <QMaemo5ValueButton>
#include <QMaemo5InformationBox>
#include <QDebug>
#include "settingsdialog.h"
#include "settings.h"
#include "db.h"
#include "daemon.h"
#include "cache.h"

QMap <Eniro::Site, Eniro::SiteDetails> SettingsDialog::sites_;

SettingsDialog::SettingsDialog(QWidget* parent): QDialog(parent),
usernameInput_(0), passwordInput_(0), cacheInput_(0), siteSelector_(0),
autostartSelector_(0)
{
    setWindowTitle(tr("Settings"));

    DB::connect();

    QVBoxLayout* left = new QVBoxLayout;
    QHBoxLayout* mainLayout = new QHBoxLayout;
    QHBoxLayout* username = new QHBoxLayout;
    QHBoxLayout* password = new QHBoxLayout;
    QHBoxLayout* cache = new QHBoxLayout;

    QLabel* usernameLabel = new QLabel(tr("Eniro username"));
    usernameInput_ = new QLineEdit(Settings::instance()->get("eniro_username"));

    QLabel* passwordLabel = new QLabel(tr("Eniro password"));
    passwordInput_ = new QLineEdit(Settings::instance()->get("eniro_password"));
    passwordInput_->setEchoMode(QLineEdit::Password);

    QLabel* cacheLabel = new QLabel(tr("Cache size (numbers)"));
    cacheInput_ = new QLineEdit(Settings::instance()->get("cache_size"));
    cacheInput_->setValidator(new QIntValidator(0, 10000, this));
    QPushButton* cacheResetButton = new QPushButton(tr("Clear"), this);
    connect(cacheResetButton, SIGNAL(pressed()), this, SLOT(resetCache()));

    siteSelector_ = new ButtonSelector(tr("Eniro site"), this);
    QString site = Settings::instance()->get("eniro_site");
    int i = 0;

    if(sites_.empty())
    {
       sites_ = Eniro::getSites();
    }

    QMap <Eniro::Site, Eniro::SiteDetails>::const_iterator it;
    for(it = sites_.begin(); it != sites_.end(); it++)
    {
        QString name;

        switch(it.key())
        {
        case Eniro::FI:
            name = tr("Finnish");
            break;
        case Eniro::SE:
            name = tr("Swedish");
            break;
        case Eniro::DK:
            name = tr("Danish");
            break;
        default:
            qDebug() << "Unknown site";
            continue;

        }
        siteSelector_->addItem(name, it.value().id);

        if(it.value().id == site)
        {
            siteSelector_->setCurrentIndex(i);
        }

        i++;
    }

    autostartSelector_ = new ButtonSelector(tr("Autostart"), this);
    QString autostart = Settings::instance()->get("autostart");
    autostartSelector_->addItem(tr("Enabled"), "1");
    autostartSelector_->addItem(tr("Disabled"), "0");
    autostartSelector_->setCurrentIndex(autostart == "1" ? 0 : 1);

    QPushButton* submitButton = new QPushButton(tr("Save"), this);
    connect(submitButton, SIGNAL(pressed()), this, SLOT(saveSettings()));

    username->addWidget(usernameLabel);
    username->addWidget(usernameInput_);
    password->addWidget(passwordLabel);
    password->addWidget(passwordInput_);
    cache->addWidget(cacheLabel);
    cache->addWidget(cacheInput_);
    cache->addWidget(cacheResetButton);
    left->addLayout(username);
    left->addLayout(password);
    left->addLayout(cache);
    left->addWidget(siteSelector_);
    left->addWidget(autostartSelector_);

    QDialogButtonBox* buttons = new QDialogButtonBox;
    buttons->setCenterButtons(false);
    buttons->addButton(submitButton, QDialogButtonBox::AcceptRole);

    mainLayout->addLayout(left);
    mainLayout->addWidget(buttons);

    setLayout(mainLayout);

    DB::disconnect();
}

void SettingsDialog::saveSettings()
{
    DB::connect();

    Settings::instance()->set("eniro_username", usernameInput_->text());
    Settings::instance()->set("eniro_password", passwordInput_->text());
    Settings::instance()->set("cache_size", cacheInput_->text());
    QString site = siteSelector_->value().toString();
    Settings::instance()->set("site", site);
    QString autostart = autostartSelector_->value().toString();
    Settings::instance()->set("autostart", autostart);

    DB::disconnect();

    hide();

    if(site != currentSite_ && Daemon::isRunning())
    {
        QMaemo5InformationBox::information(this, tr("Restarting daemon..."));
        Daemon::restart();
        currentSite_ = site;
    }

}

void SettingsDialog::setVisible(bool visible)
{
    QDialog::setVisible(visible);

    if(visible)
    {
        currentSite_ = siteSelector_->value().toString();
    }

}

void SettingsDialog::resetCache()
{
    int ret = Cache::instance().clear();

    if(ret >= 0)
    {
        QMaemo5InformationBox::information(this, tr("%n number(s) were deleted from cache", "", ret));
    }
}
