#include <tangle.h>
#include <clutter/clutter.h>
#include <fcntl.h> //we check some files
#include <math.h> //sin
#include <string.h>

#include "../configure.h" //e.g. DATA_DIR
#include "../jammo-track-view.h"
#include "../jammo-cursor.h"
#include "../jammo-mentor.h"
#include "../jammo.h"

#include "chum.h" //chum_is_easy_game
#include "theme.h"
#include "composition_game.h"
#include "composition_game_file_helper.h"


static int x_place_for_icon = 0; //Used when making icons for save-menu


#define WIDTH_OF_SLOT 40.0
static gdouble phasing(gdouble start_value, int phase) {
	gdouble value = start_value;
	value += phase/100.0;
	if (value>=1) value -= 1;
	return value;
}
/*
Draw cupboard with two doors. It is placed far right.
Right-door will open little.
Clicked button will move inside cupboard. (over left door, but under right).
*/
static void animate_icon_to_cupboard(JammoMentor* mentor, const gchar* speech, gboolean interrupted, gpointer button){
	int x_for_cupboard = 510;
	//Cupboard itself
	ClutterActor* cupboard = clutter_texture_new_from_file("/opt/jammo/cupboard.png", NULL);
	clutter_actor_set_x(cupboard,x_for_cupboard);
	clutter_actor_set_y(cupboard,0);
	tangle_widget_add(TANGLE_WIDGET(clutter_actor_get_parent(CLUTTER_ACTOR(button))), cupboard, NULL);

	//Left door. this will be closed
	ClutterActor* cupboard_left = clutter_texture_new_from_file("/opt/jammo/cupboard_leftdoor.png", NULL);
	clutter_actor_set_x(cupboard_left,x_for_cupboard);
	clutter_actor_set_y(cupboard_left,60);
	tangle_widget_add(TANGLE_WIDGET(clutter_actor_get_parent(CLUTTER_ACTOR(button))), cupboard_left, NULL);

	//Raise clicked button (also over all other buttons)
	clutter_actor_raise_top(CLUTTER_ACTOR(button));

	//Right door. This will be animated. This will be topmost actor
	ClutterActor* cupboard_right = clutter_texture_new_from_file("/opt/jammo/cupboard_rightdoor.png", NULL);
	clutter_actor_set_anchor_point(cupboard_right, clutter_actor_get_width(cupboard_right), 0.0);  //anchor is on right side
	clutter_actor_set_x(cupboard_right,x_for_cupboard+clutter_actor_get_width(cupboard_left)+clutter_actor_get_width(cupboard_right));
	clutter_actor_set_y(cupboard_right,60);
	tangle_widget_add(TANGLE_WIDGET(clutter_actor_get_parent(CLUTTER_ACTOR(button))), cupboard_right, NULL);
	ClutterAnimation* animation_for_door;
	animation_for_door = clutter_actor_animate(cupboard_right, CLUTTER_LINEAR, 3000, "rotation-angle-y", 10.0,  NULL);

	//Animate (=move) clicked button inside cupboard
	ClutterAnimation* animation;
	ClutterTimeline* timeline;
	animation = clutter_actor_animate(button, CLUTTER_LINEAR, 4000, "x", 700.0, NULL);
	timeline = clutter_animation_get_timeline(animation);
	g_signal_connect(timeline, "completed", G_CALLBACK(composition_game_close), NULL);
}


static gboolean save_image_pressed(TangleButton *tanglebutton, gpointer data){
	jammo_theme* theme = (jammo_theme*)data;
	const gchar* filename = clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton));

	clutter_actor_hide(jammo_get_actor_by_id("saving_dialog_cancel")); //No canceling after image is selected.


	printf("Selected '%s'\n",filename);
	composition_game_file_helper_save_composition((gchar*)filename, theme->bottom_track_view, theme->upper_track_view, theme->prefix_for_saving);
	jammo_mentor_speak_with_callback(jammo_mentor_get_default(),"save_confirmed.spx",animate_icon_to_cupboard,tanglebutton);
	return FALSE;
}


/*
clutter_timeline_get_progress returns value between [0,1]
with phasing we can shift it always same amount
*/
static gdouble sine_wave (ClutterAlpha *alpha,gpointer user_data) {
return sin (phasing(clutter_timeline_get_progress (clutter_alpha_get_timeline (alpha)),GPOINTER_TO_INT(user_data)) * G_PI);
}


static int tempo_table[] = {0,140,110,90}; //variation1 is 140bpm, variation2 is 110bpm, variation3 is 90pbm

//On advanced game, variation is not used, they are all using same set of sample_buttons
ClutterActor* theme_give_sample_button_for_this_theme_and_variation_for_this_id(gchar* themename, guint variation, guint id){
	gchar *name;
	if (chum_is_easy_game())
		name = g_strdup_printf("composition_game_theme_%s_%d_id%d", themename,tempo_table[variation],id);
	else
		name = g_strdup_printf("composition_game_theme_%s_advanced_id%d", themename,id);

	JammoSampleButton* sb = JAMMO_SAMPLE_BUTTON(jammo_get_actor_by_id(name)); //clone this!
	g_free(name);
	//It is unique, we can't return it.  We make clone of it.

	ClutterActor* sample_button;
	sample_button = jammo_sample_button_new_from_files(jammo_sample_button_get_image_filename(sb),jammo_sample_button_get_sample_filename(sb));
	jammo_sample_button_set_loop_id(JAMMO_SAMPLE_BUTTON(sample_button),jammo_sample_button_get_loop_id(sb));
	return sample_button;
}

//soundfamily is button containing samplebuttons.
static GList* soundfamily=NULL;
static GList* buttons_for_soundfamily=NULL;

/*
This is used for_each in TangleWidget.
a) if only_playing_mode, disable moving and clicking
b) use these icons for savemenu
c) add looping animation for samplebutton
*/
static void tune_samplebutton(ClutterActor* sample_button, gpointer user_data){
	//Skip if actor is background-actor (it has name "background")
	const gchar* name = clutter_actor_get_name(sample_button);
	if (name && strncmp(clutter_actor_get_name(sample_button),"background",10)==0)
		return;
	jammo_theme* theme = (jammo_theme*)user_data;
	/* If we are in listening mode, we do not need:
	*pushable
	*draggable
	*animated
	   sample_buttons
	just make textures*/

	if (theme->only_playing_mode) {
		tangle_draggable_actor_set_dragging_enabled(TANGLE_DRAGGABLE_ACTOR(sample_button), FALSE);
		clutter_actor_set_reactive(sample_button,FALSE);
		return;
		}


	//This is good place to do menu for selecting icon for saved composition
	if (x_place_for_icon<500) { //if 500px is used do not add anymore icons
		ClutterActor* icon_button = tangle_button_new_with_background_actor(clutter_texture_new_from_file(jammo_sample_button_get_image_filename(JAMMO_SAMPLE_BUTTON(sample_button)), NULL));
		gchar* name_for_callback = g_strdup_printf("%s",jammo_sample_button_get_image_filename(JAMMO_SAMPLE_BUTTON(sample_button))); //FIXME: this is not free'ed!
		g_signal_connect (icon_button, "clicked", G_CALLBACK (save_image_pressed),theme);
		clutter_actor_set_name(icon_button, name_for_callback);
		clutter_actor_set_x(icon_button,x_place_for_icon);
		clutter_actor_set_y(icon_button,200);
		clutter_actor_set_width(icon_button,80); //This is the size/ratio when it is in cupboard
		tangle_widget_add(TANGLE_WIDGET(theme->save_menu_container), icon_button, NULL);
		x_place_for_icon+=clutter_actor_get_width(icon_button)+5;
	}

	//Add animation to this sample_button

	//Each animation has own phase
	int phase = (rand () % 100) + 1;
	//printf("phase %d\n",phase);

	//Each animation has own wobbling 'mode'-function
	gulong sine_function;
	sine_function = clutter_alpha_register_func (sine_wave, GINT_TO_POINTER(phase));

	ClutterTimeline *timeline1 = tangle_timeline_new(3000+phase*20);  //each has own rhythm (use same random number than phase)

	theme->animation_timelines = g_list_append(theme->animation_timelines,timeline1);
	ClutterAnimation* animation;
	animation = clutter_actor_animate_with_timeline (sample_button,
            sine_function,
            timeline1,
            "height", 50.0,  //add some vertical scaling, it doesn't disturb movement
            NULL);
	clutter_animation_set_loop (animation,TRUE);
	g_object_unref(timeline1);
}

/* This will assing soundfamilies and will call tune_samplebutton */
static void advanced_tune(ClutterActor* container, gpointer user_data){
	//printf("making soundfamily\n");
	if (TANGLE_IS_BUTTON(container)) {
		//printf("not container, quit\n");
		buttons_for_soundfamily = g_list_append(buttons_for_soundfamily,container);
		return;
	}
	//printf("container, tune each inside\n");
	soundfamily = g_list_append(soundfamily,container);
	clutter_container_foreach(CLUTTER_CONTAINER(container), CLUTTER_CALLBACK(tune_samplebutton), user_data);
}

/*
actor is button, not interested.
event is not interested.
user_data is container
*/
void theme_soundfamily_clicked (TangleButton *tanglebutton, gpointer user_data){
	//check if this menu is clicked already (is it visible)
	//printf("theme_soundfamily_clicked\n");

	//when anything is pressed, hide all.
	//printf("hiding\n");
	GList* l;
	for (l=soundfamily;l;l=l->next)
		{
		//printf("hide this\n");
		tangle_actor_hide_animated(TANGLE_ACTOR(l->data));
		}

	//non-select each other-button
	for (l=buttons_for_soundfamily;l;l=l->next)
		{
			if (l->data!=tanglebutton) {
				//printf("unselect this\n");
				tangle_button_set_selected(TANGLE_BUTTON(l->data),FALSE);
			}
		}

	//if it is pressed down, show it's soundfamily
	if (tangle_button_get_selected(tanglebutton)) {
		gchar* id_of_soundfamily = g_strdup_printf("composition_game_theme_%s", clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)));
		//printf("showing\n");
		tangle_actor_show(TANGLE_ACTOR(jammo_get_actor_by_id(id_of_soundfamily)));
		g_free(id_of_soundfamily);
	}

}








/*
Make colored rectangles and add them to given parent.
Returns number of used slots (length of color_coding).
white=# because it has special meaning. Samples can't be dropped on this slot. (TODO)
*/
static int create_backing_track_color_coding(const gchar* color_coding, gfloat slot_size, gfloat y_offset,TangleWidget* parent) {
	ClutterActor* actor;
	ClutterColor white  = { 255,255,255,   255 };
	ClutterColor yellow = { 255, 255, 0,   255 };
	ClutterColor green  = { 0, 255, 0,     255 };
	ClutterColor orange = { 255, 165, 0,   255 };
	ClutterColor blue   = { 25, 25, 112,   255 };
	ClutterColor red    = { 255, 0, 0,     255 };
	ClutterColor lblue  = { 100, 149, 237, 255 };
	ClutterColor violet = { 148,0,211,     255 };
	ClutterColor pink   = { 255,105,180,   255 };

	int max=strlen(color_coding);
	int i;
	for (i=0;i<max;i++)
		{
		actor = tangle_widget_new();
		ClutterColor* color;
		switch (color_coding[i])
			{
			case '#': color =  &white         ;break;
			case 'y': color =  &yellow        ;break;
			case 'g': color =  &green         ;break;
			case 'o': color =  &orange        ;break;
			case 'b': color =  &blue          ;break;
			case 'r': color =  &red           ;break;
			case 'l': color =  &lblue         ;break;
			case 'v': color =  &violet        ;break;
			case 'p': color =  &pink          ;break;
			default: color = &white;
			}

		tangle_widget_set_background_color(TANGLE_WIDGET(actor), color);
		g_object_set(actor, "width", slot_size, "height", 40.0, "x", i* slot_size, "y", 80.0+y_offset, NULL);
		clutter_container_add_actor(CLUTTER_CONTAINER(parent), actor);
		}
	return max;
}


static void on_cursor_notify_x(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	TangleVault* vault;
	TangleScrollingActor* scrolling_actor;
	JammoCursor* cursor;
	gfloat x, visible, offset, max;

	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 2, TANGLE_TYPE_SCROLLING_ACTOR, &scrolling_actor, JAMMO_TYPE_CURSOR, &cursor);
	g_object_get(object, "x", &x, NULL);
	g_object_get(scrolling_actor, "width", &visible, "scrolling-offset-x", &offset, "scrolling-width", &max, NULL);
	if (x < offset) {
		offset = x - visible + 80.0;
		if (offset < 0.0) {
			offset = 0.0;
		}
		clutter_actor_animate(CLUTTER_ACTOR(scrolling_actor), CLUTTER_EASE_IN_OUT_QUAD, 250, "scrolling-offset-x", offset, NULL);
	} else if (x > offset + visible - 80.0) {
		offset += 160.0;
		if (offset > max) {
			offset = max;
		}
		clutter_actor_animate(CLUTTER_ACTOR(scrolling_actor), CLUTTER_EASE_IN_OUT_QUAD, 250, "scrolling-offset-x", offset, NULL);
	}
}


static void on_sequencer_stopped(JammoSequencer* sequencer, gpointer data) {
	jammo_theme* theme = (jammo_theme*)data;
	if (theme->only_playing_mode)
		{
		composition_game_close();
		return;
		}

	printf("Sequencer stopped\n");
	tangle_actor_show(TANGLE_ACTOR(jammo_mentor_get_default()));

	theme_play_all_icon_animations(theme);

	jammo_track_view_set_editing_enabled(theme->bottom_track_view, TRUE);
	if (theme->upper_track_view && !(chum_get_pair_composition())) {
		jammo_track_view_set_editing_enabled(theme->upper_track_view, TRUE);
	}


	g_object_set(theme->play_button, "x", 0.0,  NULL);
	//If user press stop, this is not needed (it happens automatically)
	//But if track ends and sequencer stops, we must call this
	//g_object_set(theme->play_button, "selected", TRUE,  NULL);
	tangle_button_set_selected(TANGLE_BUTTON(theme->play_button),FALSE);
}


void theme_stop_all_icon_animations(jammo_theme* theme){
	//printf("theme_stop_all_icon_animations \n");
	GList *timeline;
	for (timeline = theme->animation_timelines;timeline;timeline = timeline->next)
		{
		clutter_timeline_pause(CLUTTER_TIMELINE(timeline->data));
		}
}
void theme_play_all_icon_animations(jammo_theme* theme) {
	GList *timeline;
	for (timeline = theme->animation_timelines;timeline;timeline = timeline->next)
		{
		clutter_timeline_start(CLUTTER_TIMELINE(timeline->data));
		}
}
static void remove_all_icon_animations(jammo_theme* theme) {
	GList *timeline;
	for (timeline = theme->animation_timelines;timeline;timeline = timeline->next)
		{
		clutter_timeline_set_loop (CLUTTER_TIMELINE(timeline->data),FALSE); //Animations will stop and send completed, and then they are free'ed
		}
}



jammo_theme* theme_create(const gchar* name,int variation, gboolean only_listening, ClutterActor* play_button) {
	x_place_for_icon = 0; //static (used only when initializing new theme)

	jammo_theme* theme;
	theme = g_new( jammo_theme, 1 );

	theme->only_playing_mode=only_listening;
	theme->animation_timelines=NULL;
	theme->play_button = play_button;


	JammoSequencer* sequencer = jammo_sequencer_new();
	g_signal_connect(sequencer, "stopped", G_CALLBACK(on_sequencer_stopped), theme);

	if (only_listening)
		clutter_actor_hide(play_button);
	else
		clutter_actor_show(play_button); //it could be hided if we have been in cupboard

	ClutterActor* save_menu_container = tangle_widget_new();

	theme->sequencer=sequencer;
	theme->save_menu_container=save_menu_container;

	//Advanced game sets this to TRUE
	gboolean use_two_tracks=FALSE;


	const gchar* backing_track_color_code;
	const gchar* backing_track_filepath;

	int disabled_slots_for_begin = 0;
	int disabled_slots_for_end = 0;
	int tempo_factory = 4;    //4 is default
	int beats_per_minute=90; //This value should be overwritten.


	//static GList
	soundfamily=NULL;
	buttons_for_soundfamily=NULL;



	gchar* name_of_meta_data="";
	gchar* name_of_container ="";

	if (chum_is_easy_game()) {
		name_of_meta_data = g_strdup_printf("composition_game_theme_%s_%d_data", name,tempo_table[variation]);
		name_of_container = g_strdup_printf("composition_game_theme_%s_%d_container", name,tempo_table[variation]);
	}
	else {
		name_of_meta_data = g_strdup_printf("composition_game_theme_%s_advanced_%d_data", name,variation);  //pass variation value
		name_of_container = g_strdup_printf("composition_game_theme_%s_advanced_container", name); //each variation uses same loop set in advanced!
	}


	printf("looking for meta-data: '%s'\n",name_of_meta_data);
	printf("looking for container: '%s'\n",name_of_container);

	TangleProperties* theme_properties = TANGLE_PROPERTIES(jammo_get_object_by_id(name_of_meta_data));
	backing_track_filepath   = tangle_properties_get_string(theme_properties, "backing_track_filepath");
	backing_track_color_code    = tangle_properties_get_string(theme_properties, "backing_track_color_code");
	disabled_slots_for_begin = tangle_properties_get_int(theme_properties, "disabled_slots_for_begin");
	disabled_slots_for_end   = tangle_properties_get_int(theme_properties, "disabled_slots_for_end");
	tempo_factory            = tangle_properties_get_int(theme_properties, "tempo_factory");
	beats_per_minute = tangle_properties_get_int(theme_properties, "beats_per_minute");



	ClutterActor* container = jammo_get_actor_by_id(name_of_container);

	if (container){
		//printf("container NOT null\n");
	}
	else  {//debug. we should do more. actor_show will seg fault.
		printf("theme-container was NULL\n");
	}

	clutter_actor_show(container);
	theme->container=container;

	//Differencies between easy and advanced:
	if (chum_is_easy_game()){
		clutter_container_foreach(CLUTTER_CONTAINER(container), CLUTTER_CALLBACK(tune_samplebutton), theme);
	}
	else{
		clutter_container_foreach(CLUTTER_CONTAINER(container), CLUTTER_CALLBACK(advanced_tune), theme);
		use_two_tracks=TRUE;  //Use two tracks
	}


	g_free(name_of_meta_data);
	g_free(name_of_container);



//printf("making track_views\n");
int y_offset =0;
if (use_two_tracks) y_offset=80;


	//Container for tracks (one or two) and backing_track
	ClutterActor *container_for_tracks = jammo_get_actor_by_id("composition_game_container_for_tracks");


	ClutterActor *container_for_backingtrack_blocks = tangle_widget_new();
	clutter_actor_set_position(container_for_backingtrack_blocks, 0.0, 0.0);

	int number_of_slots = create_backing_track_color_coding(backing_track_color_code,WIDTH_OF_SLOT,y_offset,TANGLE_WIDGET(container_for_backingtrack_blocks));
	tangle_widget_add(TANGLE_WIDGET(container_for_tracks), container_for_backingtrack_blocks, NULL);

	//Tune width of containers
	clutter_actor_set_size(container_for_backingtrack_blocks,number_of_slots*WIDTH_OF_SLOT, 38.0);
	clutter_actor_set_size(container_for_tracks,number_of_slots*WIDTH_OF_SLOT, 120.0+y_offset); //track is 80 and backingtrack is 40


	printf("tempo_factory (time_sig): %d, beats_per_minute: %d\n",tempo_factory,beats_per_minute);

	//We want operate whole time with guint64.
	guint64 temp=60*tempo_factory;
	guint64 one_second = 1000000000L;
	guint64 big = temp * one_second;
	guint64 slot_duration = big / beats_per_minute;

	//Backing track
	printf("backingtrack filename: '%s'\n",backing_track_filepath);
	int fd = open(backing_track_filepath,O_RDONLY);
	if (fd != -1) { //Found it.
		JammoEditingTrack* backing_track = jammo_editing_track_new_fixed_duration(number_of_slots*slot_duration);
		jammo_sequencer_add_track(sequencer, JAMMO_TRACK(backing_track));
		jammo_editing_track_add_sample(backing_track, jammo_sample_new_from_file(backing_track_filepath), 0);
	}



	//Bottom track. Below of this there will be place for scrollbar.
	JammoEditingTrack* editing_track_theme = jammo_editing_track_new_fixed_duration(number_of_slots*slot_duration);
	jammo_sequencer_add_track(sequencer, JAMMO_TRACK(editing_track_theme));
	ClutterActor*	track_view = jammo_track_view_new(editing_track_theme, number_of_slots,slot_duration, WIDTH_OF_SLOT, 80.0);
	g_object_set(track_view, "disabled-slots-begin", disabled_slots_for_begin, "disabled-slots-end", disabled_slots_for_end,NULL);
	clutter_actor_set_reactive(track_view, TRUE);
	g_object_set(track_view, "line-every-nth-slot", 2, NULL);
	clutter_actor_set_position(track_view, 0.0, 2.0+y_offset);
	tangle_widget_add(TANGLE_WIDGET(container_for_tracks), track_view, NULL);

	theme->bottom_track_view = JAMMO_TRACK_VIEW(track_view); //This is needed when loading/saving
	theme->upper_track_view = NULL; //This will be NULL, if use_two_tracks is not TRUE.

	if (use_two_tracks){
	//Upper track.
	JammoEditingTrack* editing_track_theme2 = jammo_editing_track_new_fixed_duration(number_of_slots*slot_duration);
	jammo_sequencer_add_track(sequencer, JAMMO_TRACK(editing_track_theme2));
	ClutterActor* track_view2 = jammo_track_view_new(editing_track_theme2, number_of_slots,slot_duration, WIDTH_OF_SLOT, 80.0);


	//pair-game:
	if (chum_get_pair_composition()) {
		printf("pairgame\n");
		jammo_track_view_set_editing_enabled(JAMMO_TRACK_VIEW(track_view2),FALSE);
		g_object_set(chum_get_pair_composition(),"peer-track-view", track_view2,NULL); //this will handle all other things
	}
	else {
		printf("single player game\n");
		g_object_set(track_view2, "disabled-slots-begin", disabled_slots_for_begin, "disabled-slots-end", disabled_slots_for_end,NULL);
		clutter_actor_set_reactive(track_view2, TRUE);
	}

	g_object_set(track_view2, "line-every-nth-slot", 2, NULL);
	clutter_actor_set_position(track_view2, 0.0, 2.0);
	tangle_widget_add(TANGLE_WIDGET(container_for_tracks), track_view2, NULL);

	theme->upper_track_view = JAMMO_TRACK_VIEW(track_view2); //This is needed when loading/saving
	}

	//printf("making scrolling area for tracks\n");
	//Scrolling area for track (or tracks)
	ClutterActor* scrolling_actor =  jammo_get_actor_by_id("composition_game_scroller_for_tracks");

	//tune scroller area
	clutter_actor_set_y(scrolling_actor, 360.0-y_offset);
	clutter_actor_set_height(scrolling_actor,  120.0+y_offset);


	//printf("adding cursor\n");


	ClutterColor red = { 255, 0, 0, 128 };
	ClutterActor* texture;
	ClutterActor* cursor;

	texture = clutter_rectangle_new_with_color(&red);
	clutter_actor_set_size(texture, 10.0, 80.0 +y_offset);
	cursor = jammo_cursor_new(sequencer, texture);
	tangle_actor_set_depth_position(TANGLE_ACTOR(cursor), 1);
	clutter_actor_set_position(cursor, 0.0, 0.0);
	clutter_actor_set_size(cursor, number_of_slots*WIDTH_OF_SLOT, 80.0);
	clutter_container_add_actor(CLUTTER_CONTAINER(container_for_tracks), cursor);

	TangleVault* vault;

	vault = tangle_vault_new(2, TANGLE_TYPE_SCROLLING_ACTOR, scrolling_actor, JAMMO_TYPE_CURSOR, cursor);
	tangle_signal_connect_vault(texture, "notify::x", G_CALLBACK(on_cursor_notify_x), vault);


	theme->prefix_for_saving = g_strdup_printf("+%s+%d+%s",name,variation,chum_is_easy_game()? "e":"a");
	//example: +fantasy+2+e
	//printf("prefix is '%s'\n",prefix_for_saving);


	return theme;
}

void delete_theme(jammo_theme* theme){
	printf("**DELETING THEME**\n");

	if (theme->animation_timelines != NULL) {
		remove_all_icon_animations(theme);
		g_list_free(theme->animation_timelines);
	}

	//TODO, free everything.
	if (theme->save_menu_container != NULL){
	clutter_actor_destroy(theme->save_menu_container);
	}

	//Free contents of tracks
	clutter_container_foreach(CLUTTER_CONTAINER(jammo_get_actor_by_id("composition_game_container_for_tracks")), CLUTTER_CALLBACK(clutter_actor_hide), NULL);
	clutter_container_foreach(CLUTTER_CONTAINER(jammo_get_actor_by_id("composition_game_container_for_tracks")), CLUTTER_CALLBACK(clutter_actor_destroy), NULL);

	//Hide this set of samplebuttons
	clutter_actor_hide(theme->container);

	//g_free(theme->upper_track_view);
	//g_free(theme->bottom_track_view);
	//g_free(theme->sequencer);
	g_free(theme);
}
