/*
 * gems_message_functions.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu, Lappeenranta University of Technology
 *
 * Authors: Jussi Laakkonen <jussi.laakkonen@lut.fi>
 */
 
#include <glib.h>
#include <arpa/inet.h>
#include <string.h>
#include <stdio.h>

#include "gems_message_functions.h"
#include "gems_definitions.h"
#include "groupmanager.h"
#include "gems.h"

#include "../meam/jammo-midi.h"
#include "../meam/jammo-slider-event.h"

gems_message* gems_create_error_message(gint16 errortype, gint16 serviceid)
{	
	// allocate struct
	gems_message* errormsg = (gems_message*)g_malloc(sizeof(gems_message));
	
	// error message = error + length + service id + error type + message
	errormsg->length = sizeof(gint16) + sizeof(gint32) + sizeof(gint16) + sizeof(gint16) + strlen(get_errormessage(errortype)) + 1;
	
	errormsg->message = (gchar*)g_malloc(sizeof(gchar*) * errormsg->length);
	
	memset(errormsg->message,'\0',errormsg->length);
	
	gint16 error = ERROR_MSG;
	guint position = 0;
	
	*(gint16*)&errormsg->message[position] = g_htons(error); // ERROR_MSG
	position = position + sizeof(gint16);
	
	*(guint32*)&errormsg->message[position] = g_htonl(errormsg->length); // LENGTH
	position = position + sizeof(guint32);
	
	*(gint16*)&errormsg->message[position] = g_htons(serviceid); // Service id
	position = position + sizeof(gint16);
	
	*(gint16*)&errormsg->message[position] = g_htons(errortype); // Error type
	position = position + sizeof(gint16);
	
	g_strlcat(&errormsg->message[position],get_errormessage(errortype),errormsg->length); // Error message
	
	return errormsg;
}

void gems_clear_message(gems_message* message)
{
	g_free(message->message);
	message->message = NULL;
	g_free(message);
	message = NULL;
}

gchar* get_errormessage(gint16 errortype)
{
	switch(errortype)
	{
		case ERROR_ALREADY_CONNECTED_TYPE:
			return ERROR_ALREADY_CONNECTED_MSG;
			
		case ERROR_VERSION_MISMATCH_TYPE:
			return ERROR_VERSION_MISMATCH_MSG;
			
		case ERROR_INVALID_STATE_TYPE:
			return ERROR_INVALID_STATE_MSG;
			
		case ERROR_INVALID_SERVICE_ID_TYPE:
			return ERROR_INVALID_SERVICE_ID_MSG;
			
		case ERROR_SERVICE_NOT_RUNNING_TYPE:
			return ERROR_SERVICE_NOT_RUNNING_MSG;
			
		case ERROR_INVALID_ACCESS_TYPE:
			return ERROR_INVALID_ACCESS_MSG;
			
		case ERROR_CONNECTION_REJECTED_TYPE:
			return ERROR_CONNECTION_REJECTED_MSG;
			
		case ERROR_NOT_IN_GROUP_TYPE:
			return ERROR_NOT_IN_GROUP_MSG;
		
		case ERROR_GROUP_NOT_ACTIVE_TYPE:
			return ERROR_GROUP_NOT_ACTIVE_MSG;
			
		case ERROR_INVALID_GROUP_ID_TYPE:
			return ERROR_INVALID_GROUP_ID_MSG;
			
		case ERROR_GROUP_FULL_TYPE:
			return ERROR_GROUP_FULL_MSG;
			
		case ERROR_ALREADY_IN_GROUP_TYPE:
			return ERROR_ALREADY_IN_GROUP_MSG;
			
		case ERROR_GROUP_LOCKED_TYPE:
			return ERROR_GROUP_LOCKED_MSG;
		
		case ERROR_GROUP_ADD_ERROR_TYPE:
			return ERROR_GROUP_ADD_ERROR_MSG;
			
		case ERROR_NOT_GROUP_OWNER_TYPE:
			return ERROR_NOT_GROUP_OWNER_MSG;
		
		default:
			return ERROR_NO_ERROR;
	}
}


gems_message* gems_create_message_service_jammo(gint16 type)
{
	gems_message* msg = (gems_message*)g_malloc(sizeof(gems_message));
	guint position = 0;
	
	msg->length = sizeof(gint16) + sizeof(gint32) + sizeof(gint16);
	
	// Adding version info
	if(type == VERSION_REPLY) msg->length = msg->length + strlen(VERSION) + 1;
	
	msg->message = (gchar*)g_malloc(sizeof(gchar*) * msg->length);
	
	memset(msg->message,'\0',msg->length);
	
	// Service id 16 bits
	*(gint16*)&msg->message[position] = g_htons(JAMMO_SERVICE_ID);
	position = position + sizeof(gint16);
	
	// Length 32 bits
	*(gint32*)&msg->message[position] = g_htonl(msg->length);
	position = position + sizeof(gint32);
	
	// Command 16 bits
	*(gint16*)&msg->message[position] = g_htons(type);
	position = position + sizeof(gint16);
	
	// Adding version info
	if(type == VERSION_REPLY) g_strlcat(&msg->message[position],VERSION,msg->length);
	
	return msg;
}

gems_message* gems_create_message_service_profile(gint16 type)
{
	gems_message* msg = (gems_message*)g_malloc(sizeof(gems_message));
	guint position = 0;
	
	const gchar* username = gems_profile_manager_get_username(NULL);
	
	msg->length = sizeof(gint16) + sizeof(gint32) + sizeof(gint16);
	
	if(type == PROFILE_PUB) msg->length = msg->length + sizeof(gint32) + strlen(username) + 1 + sizeof(gint32) + sizeof(gint16);
	// TODO full profile
	//else if(type == PROFILE_FULL)
		
	msg->message = (gchar*)g_malloc(sizeof(gchar*) * msg->length);
	
	memset(msg->message,'\0',msg->length);
	
	// Service id 16 bits
	*(gint16*)&msg->message[position] = g_htons(PROFILE_SERVICE_ID);
	position = position + sizeof(gint16);
	
	// Length 32 bits
	*(gint32*)&msg->message[position] = g_htonl(msg->length);
	position = position + sizeof(gint32);
	
	// Command 16 bits
	*(gint16*)&msg->message[position] = g_htons(type);
	position = position + sizeof(gint16);
	
	// Username
	g_strlcat(&msg->message[position],username,msg->length);
	position = position + strlen(username) + 1; //Add the '\0'
	
	// User id
	guint32 userid = gems_profile_manager_get_userid(NULL);
	*(guint32*)&msg->message[position] = g_htonl(userid);
	position = position + sizeof(guint32);
	
	// Age
	guint16 age = gems_profile_manager_get_age(NULL);
	*(guint16*)&msg->message[position] = g_htons(age);
	position = position + sizeof(guint16);
	
	// Avatar id
	guint16 avatarid = 111;
	*(guint16*)&msg->message[position] = g_htons(avatarid);
	//position = position + sizeof(gint16);
	
	if(type == PROFILE_PUB) return msg;
	// TODO add full profile info
	else if(type == PROFILE_FULL) return msg;
	else
	{
		gems_clear_message(msg);
		return NULL; // Invalid type
	}
}

gems_message* gems_create_message_profile_request(gint16 type)
{
	gems_message* msg = (gems_message*)g_malloc(sizeof(gems_message));
	guint position = 0;
	
	msg->length = sizeof(gint16) + sizeof(gint32) + sizeof(gint16) + sizeof(gint16);
	
	msg->message = (gchar*)g_malloc(sizeof(gchar*) * msg->length);
	
	memset(msg->message,'\0',msg->length);
	
	// Service id 16 bits
	*(gint16*)&msg->message[position] = g_htons(PROFILE_SERVICE_ID);
	position = position + sizeof(gint16);
	
	// Length 32 bits
	*(gint32*)&msg->message[position] = g_htonl(msg->length);
	position = position + sizeof(gint32);
	
	// Command 16 bits
	*(gint16*)&msg->message[position] = g_htons(GET_PROFILE);
	position = position + sizeof(gint16);
	
	// Type
	*(gint16*)&msg->message[position] = g_htons(type);
	
	return msg;
}

/* Use with notify messages which do not need the group information, for sending:
		- REQUEST_GROUP_INFO
		- REQUEST_MEMBERS
		- JOIN_GROUP
		- MEMBER_DROPPED
		- NEW_MEMBER
		- GROUP_UNLOCKED
		- GROUP_LOCKED
*/
gems_message* gems_create_message_group_management_notify(gint16 type, guint32 groupid, guint32 member)
{
	gems_message* msg = (gems_message*)g_malloc(sizeof(gems_message));	
	guint position = 0;
	
	msg->length = sizeof(gint16) + sizeof(gint32) + sizeof(gint16);
	if(type != REQUEST_GROUP_INFO || type != REQUEST_MEMBERS)
	{
		msg->length = msg->length + sizeof(guint32);
		if(type == MEMBER_DROPPED || type == NEW_MEMBER) msg->length = msg->length + sizeof(guint32);
	}
	
	msg->message = (gchar*)g_malloc(sizeof(gchar*) * msg->length);
	
	// Service id 16 bits
	*(gint16*)&msg->message[position] = g_htons(GROUP_SERVICE_ID);
	position = position + sizeof(gint16);
	
	// Length 32 bits
	*(gint32*)&msg->message[position] = g_htonl(msg->length);
	position = position + sizeof(gint32);
	
	// Command 16 bits
	*(gint16*)&msg->message[position] = g_htons(type);
	position = position + sizeof(gint16);
	
	// If not requesting group info or members, message is longer
	if(type != REQUEST_GROUP_INFO || type != REQUEST_MEMBERS)
	{
		// Group id
		*(guint32*)&msg->message[position] = g_htonl(groupid);
		position = position + sizeof(guint32);
		
		// If notifying about dropped or new member, add member
		if(type == MEMBER_DROPPED || type == NEW_MEMBER) *(guint32*)&msg->message[position] = g_htonl(member);
	}
					
	return msg;
}


/* Use with info messages which include the group information, for sending:
		- OFFER_GROUP
		- CURREPT_GROUP
		- MEMBER_LIST
*/
gems_message* gems_create_message_group_management_group_info(gint16 type)
{
	gems_group_info* info = gems_get_data()->service_group->group_info;
	gems_message* msg = (gems_message*)g_malloc(sizeof(gems_message));
	guint position = 0;
	
	// service id + packet length + command id + group id
	msg->length = sizeof(gint16) + sizeof(gint32) + sizeof(gint16) + sizeof(guint32);
	
	switch(type)
	{
		case OFFER_GROUP:
			// type + size + spaces + theme
			msg->length = msg->length + sizeof(gint16) + sizeof(gint16) + sizeof(gint16) + sizeof(gint16);
			break;
		case CURRENT_GROUP:
			// owner + type + size + spaces + theme
			msg->length = msg->length + sizeof(guint32) + sizeof(gint16) + sizeof(gint16) + sizeof(gint16) + sizeof(gint16);
			break;
		case MEMBER_LIST:
			// owner + member + member + member
			msg->length = msg->length + sizeof(guint32) + sizeof(guint32) + sizeof(guint32) + sizeof(guint32);
			break;
		default:
			break;
	}
	
	msg->message = (gchar*)g_malloc(sizeof(gchar*) * msg->length);
	
	// Service id 16 bits
	*(gint16*)&msg->message[position] = g_htons(GROUP_SERVICE_ID);
	position = position + sizeof(gint16);
	
	// Length 32 bits
	*(gint32*)&msg->message[position] = g_htonl(msg->length);
	position = position + sizeof(gint32);
	
	// Command 16 bits
	*(gint16*)&msg->message[position] = g_htons(type);
	position = position + sizeof(gint16);
	
	// Group id 32 bits
	*(guint32*)&msg->message[position] = g_htonl(info->id);
	position = position + sizeof(guint32);
	
	switch(type)
	{
		case OFFER_GROUP:
			// type
			*(gint16*)&msg->message[position] = g_htons(info->type);
			position = position + sizeof(gint16);
			
			// size
			*(gint16*)&msg->message[position] = g_htons(info->size);
			position = position + sizeof(gint16);
			
			// spaces
			*(gint16*)&msg->message[position] = g_htons(gems_group_get_spaces(info));
			position = position + sizeof(gint16);
			
			// theme
			*(gint16*)&msg->message[position] = g_htons(info->theme);
			
			// TODO add group lock type into packet
			
			break;
		case CURRENT_GROUP:
			// owner
			*(guint32*)&msg->message[position] = g_htonl(info->owner);
			position = position + sizeof(guint32);
			
			// type
			*(gint16*)&msg->message[position] = g_htons(info->type);
			position = position + sizeof(gint16);
			
			// size
			*(gint16*)&msg->message[position] = g_htons(info->size);
			position = position + sizeof(gint16);
			
			// spaces
			*(gint16*)&msg->message[position] = g_htons(gems_group_get_spaces(info));
			position = position + sizeof(gint16);
			
			// theme
			*(gint16*)&msg->message[position] = g_htons(info->theme);
			
			// TODO add group lock type into packet

			break;
		case MEMBER_LIST:
			// owner
			*(guint32*)&msg->message[position] = g_htonl(info->owner);
			position = position + sizeof(guint32);
			
			// member
			*(guint32*)&msg->message[position] = g_htonl(info->peers[0]);
			position = position + sizeof(guint32);
			
			// member
			*(guint32*)&msg->message[position] = g_htonl(info->peers[1]);
			position = position + sizeof(guint32);
			
			// member
			*(guint32*)&msg->message[position] = g_htonl(info->peers[2]);

			break;
		default:
			break;
	}
	
	return msg;
}

gems_message* gems_create_message_collaboration_action_confirmed()
{
	gems_message* msg = (gems_message*)g_malloc(sizeof(gems_message));	
	guint position = 0;
	
	msg->length = sizeof(gint16) + sizeof(gint32) + sizeof(gint16);

	msg->message = (gchar*)g_malloc(sizeof(gchar*) * msg->length);
	
	memset(msg->message,'\0',msg->length);
	
	// Service id 16 bits
	*(gint16*)&msg->message[position] = g_htons(COLLABORATION_SERVICE_ID);
	position = position + sizeof(gint16);
	
	// Length 32 bits
	*(gint32*)&msg->message[position] = g_htonl(msg->length);
	position = position + sizeof(gint32);
	
	// Command 16 bits
	*(gint16*)&msg->message[position] = g_htons(ACTION_CONFIRMED);
	position = position + sizeof(gint16);
	
	
	return msg;

}

//Message for adding or removing a loop - Note - currently using paths of image and wav files!
gems_message* gems_create_message_collaboration_action_vi_single(gint16 type, const gchar png_name[], const gchar wav_name[], guint16 slot)
{
	gems_message* msg = (gems_message*)g_malloc(sizeof(gems_message));	
	guint position = 0;
	
	msg->length = sizeof(gint16) + sizeof(gint32) + sizeof(gint16) + sizeof(gint16);

	msg->length = msg->length + strlen(png_name) + 1 + strlen(wav_name) + 1 + sizeof(guint16);
	
	msg->message = (gchar*)g_malloc(sizeof(gchar*) * msg->length);

	
	memset(msg->message,'\0',msg->length);

	// Service id 16 bits
	*(gint16*)&msg->message[position] = g_htons(COLLABORATION_SERVICE_ID);
	position = position + sizeof(gint16);
	
	// Length 32 bits
	*(gint32*)&msg->message[position] = g_htonl(msg->length);
	position = position + sizeof(gint32);
	
	// Command 16 bits
	*(gint16*)&msg->message[position] = g_htons(ACTION_VI_SINGLE);
	position = position + sizeof(gint16);

	// Type 16 bits
	*(gint16*)&msg->message[position] = g_htons(type);
	position = position + sizeof(gint16);
	
	// png_name TODO should this be id?
	g_strlcat(&msg->message[position],png_name,msg->length);
	position = position + strlen(png_name)+1 ;

	// wav_name TODO should this be id?
	g_strlcat(&msg->message[position],wav_name,msg->length);
	position = position + strlen(wav_name)+1 ;

	// slot
	*(guint16*)&msg->message[position] = g_htons(slot);
	position = position + sizeof(guint16);
	
	return msg;
}

//Message for transferring the song file - used when starting a game (or synchronizing after disconnection?) 
gems_message* gems_create_message_collaboration_action_song_info(gint16 type, const gchar song_info[])
{
	gems_message* msg = (gems_message*)g_malloc(sizeof(gems_message));	
	guint position = 0;
	
	msg->length = sizeof(gint16) + sizeof(gint32) + sizeof(gint16);

	msg->length = msg->length + strlen(song_info) + 1;
	
	msg->message = (gchar*)g_malloc(sizeof(gchar*) * msg->length);

	
	memset(msg->message,'\0',msg->length);

	// Service id 16 bits
	*(gint16*)&msg->message[position] = g_htons(COLLABORATION_SERVICE_ID);
	position = position + sizeof(gint16);
	
	// Length 32 bits
	*(gint32*)&msg->message[position] = g_htonl(msg->length);
	position = position + sizeof(gint32);
	
	// Command 16 bits
	*(gint16*)&msg->message[position] = g_htons(SONG_INFO);
	position = position + sizeof(gint16);
	
	// song_info
	g_strlcat(&msg->message[position],song_info,msg->length);
	position = position + strlen(song_info)+1 ;

	
	return msg;
}

//Message for adding or removing a loop - Note - currently using paths of image and wav files!
gems_message* gems_create_message_collaboration_action_loop(gint16 type, guint loop_id, guint slot)
{
	gems_message* msg = (gems_message*)g_malloc(sizeof(gems_message));	
	guint position = 0;

	msg->length = sizeof(gint16) + sizeof(gint32) + sizeof(gint16) + sizeof(guint) + sizeof(guint) + sizeof(gint16) + sizeof(guint);
	
	msg->message = (gchar*)g_malloc(sizeof(gchar*) * msg->length);

	
	memset(msg->message,'\0',msg->length);

	// Service id 16 bits
	*(gint16*)&msg->message[position] = g_htons(COLLABORATION_SERVICE_ID);
	position = position + sizeof(gint16);
	
	// Length 32 bits
	*(gint32*)&msg->message[position] = g_htonl(msg->length);
	position = position + sizeof(gint32);
	
	// Command 16 bits
	*(gint16*)&msg->message[position] = g_htons(ACTION_LOOP);
	position = position + sizeof(gint16);

	//TODO change USER ID to real value
	*(guint*)&msg->message[position] = g_htonl(0);
	position = position + sizeof(guint);

	// TODO LOOP ID
	// slot 32 bits
	*(guint*)&msg->message[position] = g_htonl(loop_id);
	position = position + sizeof(guint);

	// Type 16 bits
	*(gint16*)&msg->message[position] = g_htons(type);
	position = position + sizeof(gint16);
	
	// slot 32 bits
	*(guint*)&msg->message[position] = g_htonl(slot);
	position = position + sizeof(guint);
	
	return msg;
}

gems_message* gems_create_message_collaboration_action_midi_series(gint16 type, gint16 instrument_id, GList * list)
{
	gems_message* msg = (gems_message*)g_malloc(sizeof(gems_message));	
	guint position = 0;

	msg->length = sizeof(gint16) + sizeof(gint32) + sizeof(gint16) + sizeof(guint) + sizeof(gint16) + sizeof(gint16) + g_list_length(list) * (sizeof(unsigned char)+sizeof(GstClockTime)+sizeof(JammmoMidiEventType));
	
	msg->message = (gchar*)g_malloc(sizeof(gchar*) * msg->length);

	
	memset(msg->message,'\0',msg->length);

	// Service id 16 bits
	*(gint16*)&msg->message[position] = g_htons(COLLABORATION_SERVICE_ID);
	position = position + sizeof(gint16);
	
	// Length 32 bits
	*(gint32*)&msg->message[position] = g_htonl(msg->length);
	position = position + sizeof(gint32);
	
	// Command 16 bits
	*(gint16*)&msg->message[position] = g_htons(ACTION_MIDI);
	position = position + sizeof(gint16);

	//TODO change USER ID to real value
	// 32 bits
	*(guint*)&msg->message[position] = g_htonl(0);
	position = position + sizeof(guint);

	// TODO INSTRUMENT ID
	// 16 bits
	*(guint*)&msg->message[position] = g_htons(instrument_id);
	position = position + sizeof(gint16);

	// Type 16 bits
	*(gint16*)&msg->message[position] = g_htons(type);
	position = position + sizeof(gint16);
	
	// list, one event is 12 bytes
	GList * temp;
	JammoMidiEvent * event;
	for (temp=g_list_first(list);temp;temp=temp->next) {
		event = temp->data;
		// note
		msg->message[position] = event->note;
		position = position + sizeof(unsigned char);
		// timestamp
		*(GstClockTime *)&msg->message[position] = GUINT64_TO_BE(event->timestamp);
		position = position + sizeof(GstClockTime);
		// type
		*(JammmoMidiEventType *)&msg->message[position] = g_htonl(event->type);
		position = position + sizeof(JammmoMidiEventType);		
	}
	
	return msg;
}
gems_message* gems_create_message_collaboration_action_slider_series(gint16 type, gint16 instrument_id, GList * list)
{
	gems_message* msg = (gems_message*)g_malloc(sizeof(gems_message));	
	guint position = 0;

	msg->length = sizeof(gint16) + sizeof(gint32) + sizeof(gint16) + sizeof(guint) + sizeof(gint16) + sizeof(gint16) + g_list_length(list) * (sizeof(guint32)+sizeof(GstClockTime)+sizeof(JammoSliderEventType));
	
	msg->message = (gchar*)g_malloc(sizeof(gchar*) * msg->length);

	
	memset(msg->message,'\0',msg->length);

	// Service id 16 bits
	*(gint16*)&msg->message[position] = g_htons(COLLABORATION_SERVICE_ID);
	position = position + sizeof(gint16);
	
	// Length 32 bits
	*(gint32*)&msg->message[position] = g_htonl(msg->length);
	position = position + sizeof(gint32);
	
	// Command 16 bits
	*(gint16*)&msg->message[position] = g_htons(ACTION_SLIDER);
	position = position + sizeof(gint16);

	//TODO change USER ID to real value
	// 32 bits
	*(guint*)&msg->message[position] = g_htonl(0);
	position = position + sizeof(guint);

	// TODO INSTRUMENT ID
	// 16 bits
	*(gint16*)&msg->message[position] = g_htons(instrument_id);
	position = position + sizeof(gint16);

	// Type 16 bits
	*(gint16*)&msg->message[position] = g_htons(type);
	position = position + sizeof(gint16);
	
	// list, one event is X bytes
	GList * temp;
	JammoSliderEvent * event;
	for (temp=g_list_first(list);temp;temp=temp->next) {
		event = temp->data;
		// freq, float converted to guint32 and packed
		*(guint32*)&msg->message[position] = htonf(event->freq);
		position = position + sizeof(guint32);
		// timestamp
		*(GstClockTime *)&msg->message[position] = GUINT64_TO_BE(event->timestamp);
		position = position + sizeof(GstClockTime);
		// type
		*(JammoSliderEventType *)&msg->message[position] = g_htonl(event->type);
		position = position + sizeof(JammoSliderEventType);		
	}
	
	return msg;
}
