/**sequencer_loop.c is part of JamMo.
License: GPLv2, read more from COPYING

This file is for clutter based gui.
This is part of the sequencer.
 */
#include <glib-object.h>
#include <clutter/clutter.h>
#include <string.h>
#include <stdlib.h>

#include "sequencer_loop.h"
#include "sequencer.h"
#include "config.h"
#include "jammo-texture.h"
#include "../chum/chum.h" //chum_go_next()

enum {
	LOOP_READY = 0,
	LOOP_MOVING
};

int loop_view_state = LOOP_READY;

//dragging
gboolean is_dragging;
gint x_start_of_dragging, y_start_of_dragging, y_start_of_rotation, x_drag_offset, y_drag_offset;
gint height, pos_0_y, pos_1_y, pos_2_y;
gfloat x_sample_dragging, y_sample_dragging;

JammoSequencer *sequencer;
/*
 * Packs ClutterActors in array which size is known.
 * Eases usage of actors in loop rolls.
 */
typedef struct {
	ClutterActor *roll_group;
	ClutterActor **roll_background;
	ClutterActor **sample_labels;
	int size_of_array;
	gboolean is_locked;
} ActorArray;

//Data structures for loop rolls
ActorArray *rhytmical_sample_looper;
ActorArray *melodical_sample_looper;
ActorArray *harmonical_sample_looper;
ActorArray *effect_sample_looper;

void rotate_loop(ActorArray *actor_array, gint y_drag_offset);

ActorArray *get_actor_array_new() {
	ActorArray *actor_array = malloc(sizeof(ActorArray));
	actor_array->size_of_array = 0;
	actor_array->roll_group = clutter_group_new();
	actor_array->roll_background = malloc(5 * sizeof(ClutterActor*));
	actor_array->sample_labels = NULL;
	actor_array->is_locked = FALSE;

	return actor_array;
}

/*
 * Adds actor to given ActorArray
 */
void add_actor_to_actor_array(ActorArray *array, ClutterActor *actor) {
	if(array->sample_labels != NULL) {
		array->size_of_array++;
		array->sample_labels = realloc(array->sample_labels, array->size_of_array * sizeof(ClutterActor*));
		array->sample_labels[array->size_of_array - 1] = actor;
	} else {
		array->size_of_array = 1;
		array->sample_labels = malloc(sizeof(ClutterActor*));
		array->sample_labels[0] = actor;
	}
	//	clutter_container_add_actor(CLUTTER_CONTAINER(array->roll_group), actor);
	gfloat pos_y = 0;

	if(array->size_of_array > 1) {
		int i = 0, old_max = array->size_of_array - 1;
		pos_y = clutter_actor_get_y(array->sample_labels[old_max - 1]);
		clutter_actor_set_position(actor, 0.0, pos_y + 95.0);

		for(i = 0; i < old_max; i++) {
			if(clutter_actor_get_y(array->sample_labels[i]) >= pos_y + 95.0) {
				clutter_actor_set_position(array->sample_labels[i], 0.0, clutter_actor_get_y(array->sample_labels[i]) + 95.0);
			}
		}

	} else {
		clutter_actor_set_position(array->sample_labels[array->size_of_array - 1], 0.0, 0.0);
	}
	pos_y = clutter_actor_get_y(array->sample_labels[array->size_of_array - 1]);
	if(pos_y > 2 && pos_y < 390) {
		clutter_actor_show(array->sample_labels[array->size_of_array - 1]);
	} else {
		clutter_actor_hide(array->sample_labels[array->size_of_array - 1]);
	}
}
static gboolean play_sample(ClutterActor *actor, ClutterEvent *event, gpointer data);

ClutterActor *get_centermost_sample_from_loop_roll(ActorArray *actor_array) {
	if(actor_array != NULL) {
		int i = 0;
		for(i = 0; i < actor_array->size_of_array; i++) {
			int y = (int) clutter_actor_get_y(actor_array->sample_labels[i]);
			if(y == 190) {
				return actor_array->sample_labels[i];
			}
		}
	}

	return NULL;
}

/*
 * locks loop wheel (given in data)
 */
static gboolean lock_loop_wheel(ClutterActor *actor, ClutterEvent *event, gpointer data) {
	ClutterActor *data_actor = data;
	clutter_actor_show(data_actor);
	clutter_actor_hide(actor);
	char *name = (char*) clutter_actor_get_name(actor);
	if(strcmp(name, "rhytmical_loop_lock") == 0) {
		rhytmical_sample_looper->is_locked = TRUE;
	} else if(strcmp(name, "melodical_loop_lock") == 0) {
		melodical_sample_looper->is_locked = TRUE;
	} else if(strcmp(name, "harmonical_loop_lock") == 0) {
		harmonical_sample_looper->is_locked = TRUE;
	} else if(strcmp(name, "effect_loop_lock") == 0) {
		effect_sample_looper->is_locked = TRUE;
	} else {
		printf("This shouldn't happen. lock_loop_wheel misuse\n");
	}

	return TRUE;
}

/*
 * unlock loop wheel
 */
static gboolean unlock_loop_wheel(ClutterActor *actor, ClutterEvent *event, gpointer data) {

	ClutterActor *data_actor = data;
	clutter_actor_show(data_actor);
	clutter_actor_hide(actor);
	char *name = (char*) clutter_actor_get_name(actor);
	if(strcmp(name, "rhytmical_loop_unlock") == 0) {
		rhytmical_sample_looper->is_locked = FALSE;
	} else if(strcmp(name, "melodical_loop_unlock") == 0) {
		melodical_sample_looper->is_locked = FALSE;
	} else if(strcmp(name, "harmonical_loop_unlock") == 0) {
		harmonical_sample_looper->is_locked = FALSE;
	} else if(strcmp(name, "effect_loop_unlock") == 0) {
		effect_sample_looper->is_locked = FALSE;
	} else {
		printf("This shouldn't happen. unlock_loop_wheel misuse\n");
	}

	return TRUE;
}

/*
 * Moves sample
 */
static gboolean drag_sample(ClutterActor *actor, ClutterEvent *event, gpointer data) {
	gfloat x = 0.0, y = 0.0;
	clutter_event_get_coords(event, &x, &y);
	clutter_actor_set_position(actor, x - 47.5, y - 47.5);
	if(y > 450.0) {
		change_view_to_sequencer(actor);
	}
	return TRUE;
}

/*
 * Stops moving sample
 */
static gboolean undrag_sample(ClutterActor *actor, ClutterEvent *event, gpointer data) {
//	clutter_actor_hide(actor);
	clutter_ungrab_pointer();
	clutter_actor_destroy(actor);
	return TRUE;
}

ClutterActor *actor_array_get_actor_at_y(ActorArray *actor_array, gfloat y) {
	if(actor_array != NULL) {
		int i = 0;
		gfloat tmp_x, tmp_y, tmp_widht, tmp_height;
		for(i = 0; i < actor_array->size_of_array; i++) {
			clutter_actor_get_position(actor_array->sample_labels[i], &tmp_x, &tmp_y);
			tmp_widht = clutter_actor_get_width(actor_array->sample_labels[i]);
			tmp_height = clutter_actor_get_height(actor_array->sample_labels[i]);
			if(y > tmp_y && y < (tmp_y + tmp_height)) {
				return actor_array->sample_labels[i];
			}
		}
	}

	return NULL;
}
/*
 * When triggered, enables moving of loop roll, given in data parameter.
 * Each roll has transparent ClutterRectangle, which is event (button-press-event) triggering actor
 */
static gboolean move_rolls(ClutterActor *actor, ClutterEvent *event, gpointer data) {
	ActorArray *actor_array = data;

	if (!actor_array->is_locked) {
		loop_view_state = LOOP_MOVING;
		gfloat tmp_x = 0, tmp_y = 0;
		clutter_event_get_coords(event, &tmp_x, &tmp_y);
		y_start_of_dragging = (gint) tmp_y;
		x_start_of_dragging = (gint) tmp_x;
		y_start_of_rotation = y_start_of_dragging;
		clutter_grab_pointer(actor);
	} else {
		gfloat x = 0.0, y = 0.0;
		clutter_event_get_coords(event, &x, &y);
		if(y > 189.9 && y < 285.1) {
			ClutterActor *tmp_actor = get_centermost_sample_from_loop_roll(actor_array);
			if(tmp_actor != NULL) {
				ClutterActor *draggable_actor = clutter_texture_new();
				CoglHandle handle = clutter_texture_get_cogl_texture(CLUTTER_TEXTURE(tmp_actor));
				clutter_texture_set_cogl_texture(CLUTTER_TEXTURE(draggable_actor), handle);

				clutter_actor_reparent(draggable_actor, jammo_get_actor_by_id("main-views-widget"));
				clutter_actor_set_position(draggable_actor, x - 47.5, y - 47.5);
				clutter_actor_set_reactive(draggable_actor, TRUE);
				g_signal_connect(CLUTTER_ACTOR(draggable_actor), "motion-event", G_CALLBACK(drag_sample), NULL);
				g_signal_connect(CLUTTER_ACTOR(draggable_actor), "button-release-event", G_CALLBACK(undrag_sample), NULL);
				clutter_grab_pointer(draggable_actor);
				clutter_actor_set_opacity(draggable_actor, 190);
				clutter_actor_show(draggable_actor);

			}
		}
	}
	return TRUE;
}

/*
 * When mouse button is released or function is called from roll_loop,
 * loop roll snaps into right position.
 */
static gboolean stop_rolls(ClutterActor *actor, ClutterEvent *event, gpointer data) {
	gfloat x = 0, y = 0;
	clutter_event_get_coords(event, &x, &y);
	loop_view_state = LOOP_READY;
	clutter_ungrab_pointer();
	ActorArray *tmp = data;
	int i = 0;
	if((x_start_of_dragging > (x - 10) && x_start_of_dragging < (x + 10))
			&& (y_start_of_dragging > (y - 10) && y_start_of_dragging < (y + 10))) {
		play_sample(actor_array_get_actor_at_y(tmp, y), NULL, NULL);
	}
	if(tmp != NULL && tmp->size_of_array > 0) {
		gint label_height = (gint) clutter_actor_get_height(tmp->sample_labels[0]);
		gint min = 1000, offset = 0;

		for(i = 0; i < tmp->size_of_array; i++) {
			gint y = (gint) clutter_actor_get_y(tmp->sample_labels[i]);
			if(y < min)
				min = y;
		}

		if(min != 0) {
			if(min > 48) {
				offset = label_height - min;
			} else {
				offset = -1 * min;
			}
			rotate_loop(tmp, offset);
		}
	}
	return TRUE;
}

/*
 * Organizes loop roll elements while roll is being rolled.
 */
void rotate_loop(ActorArray *actor_array, gint y_drag_offset) {
	int i = 0;
	/*TODO decide, if moving loop backgrounds is needed, or discard this code block and change to single background image.
	 Update this and edit stop_rolls-function, if this (useless?) feature is to be used
			  for (i = 0; i < 5; i++) {
				  new_pos_y = clutter_actor_get_y(tmp->roll_background[i]);
				  clutter_actor_set_position(tmp->roll_background[i], 0, new_pos_y + y_drag_offset);
				  int now_y = (int) clutter_actor_get_y(tmp->roll_background[i]);

				  if(now_y < 2) {
					  clutter_actor_set_position(tmp->roll_background[i], 0, now_y + 475);
				  } else if(now_y > 390) {
					  clutter_actor_set_position(tmp->roll_background[i], 0, now_y - 475);
				  }
			  }
	 */

	if(actor_array->size_of_array > 0) {
		gint label_height = (gint) clutter_actor_get_height(actor_array->sample_labels[0]);
		gint full_height = actor_array->size_of_array * label_height;
		gfloat new_pos_y = 0;
		for (i = 0; i < actor_array->size_of_array; i++) {
			new_pos_y = clutter_actor_get_y(actor_array->sample_labels[i]);
			clutter_actor_set_position(actor_array->sample_labels[i], 0, new_pos_y + y_drag_offset);
			int now_y = (int) clutter_actor_get_y(actor_array->sample_labels[i]);
			if(y_drag_offset < 0) {
				if(now_y < 0) {
					if(full_height > 475) {
						clutter_actor_set_position(actor_array->sample_labels[i], 0, full_height + now_y);
					} else {
						clutter_actor_set_position(actor_array->sample_labels[i], 0, now_y + 475);
					}
				}
			} else if (y_drag_offset > 0) {

				if(now_y > full_height && full_height > 475) {
					clutter_actor_set_position(actor_array->sample_labels[i], 0, now_y - full_height);

				} else if (now_y > 475 && full_height <= 475){
					clutter_actor_set_position(actor_array->sample_labels[i], 0, now_y - 475);
				}
				new_pos_y = clutter_actor_get_y(actor_array->sample_labels[i]);


			}
			if(new_pos_y >= 2 && new_pos_y <= 390) {
				clutter_actor_show(actor_array->sample_labels[i]);
			} else {
				clutter_actor_hide(actor_array->sample_labels[i]);
			}
		}
	}

}

/*
 * Event for rotating loop
 */
static gboolean roll_loop(ClutterActor *actor, ClutterEvent *event, gpointer data) {

	if(loop_view_state == LOOP_MOVING) {
		ActorArray *tmp = data;
		gfloat tmp_pos_y = 0, tmp_pos_x = 0;
		clutter_event_get_coords(event, &tmp_pos_x, &tmp_pos_y);
		y_drag_offset = (gint) tmp_pos_y - y_start_of_rotation;
		y_start_of_rotation = (gint) tmp_pos_y;

		if(tmp_pos_y > 102 && tmp_pos_y < 382) {
			rotate_loop(tmp, y_drag_offset);
		} else {
			stop_rolls(actor, event, data);
		}

	}

	return TRUE;
}

/*
 * plays single sample (JammoTexture) from loop roll
 */
static gboolean play_sample(ClutterActor *actor, ClutterEvent *event, gpointer data) {
	if(actor != NULL) {
		jammo_sample_stop_all();
		jammo_sample_play(jammo_texture_get_sample(JAMMO_TEXTURE(actor)));
	}

	return TRUE;
}

/*
 * plays all samples from centerline of loop rolls
 */
static gboolean play_all_samples(ClutterActor *actor, ClutterEvent *event, gpointer data) {

	JammoSequencer *seq = jammo_sequencer_new();
	char* tmp_path = (char*) jammo_texture_get_sound_file_name(JAMMO_TEXTURE(get_centermost_sample_from_loop_roll(effect_sample_looper)));
	if(tmp_path != NULL) {
		JammoEditingTrack *track1 = jammo_editing_track_new();
		JammoSample *sample1 = jammo_sample_new_from_file(tmp_path);
		jammo_editing_track_add_sample(track1, sample1, 0);
		jammo_sequencer_add_track(seq, JAMMO_TRACK(track1));
	}
	tmp_path = (char*) jammo_texture_get_sound_file_name(JAMMO_TEXTURE(get_centermost_sample_from_loop_roll(rhytmical_sample_looper)));

	if(tmp_path != NULL) {
		JammoEditingTrack *track2 = jammo_editing_track_new();
		JammoSample *sample2 = jammo_sample_new_from_file(tmp_path);
		jammo_editing_track_add_sample(track2, sample2, 0);
		jammo_sequencer_add_track(seq, JAMMO_TRACK(track2));
	}

	tmp_path = (char*) jammo_texture_get_sound_file_name(JAMMO_TEXTURE(get_centermost_sample_from_loop_roll(melodical_sample_looper)));
	if(tmp_path != NULL) {
		JammoEditingTrack *track3 = jammo_editing_track_new();
		JammoSample *sample3 = jammo_sample_new_from_file(tmp_path);
		jammo_editing_track_add_sample(track3, sample3, 0);
		jammo_sequencer_add_track(seq, JAMMO_TRACK(track3));
	}

	tmp_path = (char*) jammo_texture_get_sound_file_name(JAMMO_TEXTURE(get_centermost_sample_from_loop_roll(harmonical_sample_looper)));
	if(tmp_path != NULL) {
		JammoEditingTrack *track4 = jammo_editing_track_new();
		JammoSample *sample4 = jammo_sample_new_from_file(tmp_path);
		jammo_editing_track_add_sample(track4, sample4, 0);
		jammo_sequencer_add_track(seq, JAMMO_TRACK(track4));
	}

	jammo_sequencer_play(seq);

	return TRUE;
}

/*FIXME not final way of working?
 * send chosen samples into sequencer
 */
static gboolean take_samples_to_sequencer(ClutterActor *actor, ClutterEvent *event, gpointer data) {
	//TODO

	return TRUE;
}

/*
 * Loop rolls don't work properly, if they are turned large amount at once.
 * This function counters that problem.
 * For simplicity make sure, that 'amount' is multiple of sample label's height
 */
void rotate_loop_for_amount(ActorArray *actor_array, gint amount) {
	if(actor_array != NULL && !actor_array->is_locked) {
		int i = 0;
		if(amount > 0) {
			for(i = 0; i < amount; i++)
				rotate_loop(actor_array, 1);
		} else {
			for(i = 0; i > amount; i--)
				rotate_loop(actor_array, -1);
		}
	}
}

/*
 * Function for crazy button
 */
static gboolean randomize_loop_positions(ClutterActor *actor, ClutterEvent *event, gpointer data) {

	GRand *random_gen = g_rand_new();

	gint rot_rhyt = 95 * g_rand_int_range(random_gen, -30, 30);
	gint rot_melo = 95 * g_rand_int_range(random_gen, -30, 30);
	gint rot_harm = 95 * g_rand_int_range(random_gen, -30, 30);
	gint rot_effe = 95 * g_rand_int_range(random_gen, -30, 30);

	rotate_loop_for_amount(rhytmical_sample_looper, rot_rhyt);
	rotate_loop_for_amount(melodical_sample_looper, rot_melo);
	rotate_loop_for_amount(harmonical_sample_looper, rot_harm);
	rotate_loop_for_amount(effect_sample_looper, rot_effe);

	g_rand_free(random_gen);

	return TRUE;
}

/*
 * Function for cannot fail button
 */
static gboolean cannot_fail_loop_positions(ClutterActor *actor, ClutterEvent *event, gpointer data) {
	//TODO stump
	return TRUE;
}

void create_loop_roll(ActorArray *actor_array, ClutterColor *color) {

	GList *actors = clutter_container_get_children(CLUTTER_CONTAINER(actor_array->roll_group));
	GList *l;
	for (l = actors; l; l = l->next) {
		ClutterActor *actor = CLUTTER_ACTOR(l->data);
		add_actor_to_actor_array(actor_array, actor);
	}

	int i = 0;
	for(i = 0; i < 5; i++) {
		actor_array->roll_background[i] = clutter_rectangle_new_with_color(color);
		clutter_actor_set_size(actor_array->roll_background[i], 95.0, 95.0);
		clutter_container_add_actor(CLUTTER_CONTAINER(actor_array->roll_group), CLUTTER_ACTOR(actor_array->roll_background[i]));
		clutter_actor_set_position(actor_array->roll_background[i], 0.0, (i * 95.0));
		clutter_actor_lower_bottom(actor_array->roll_background[i]);
	}

	ClutterActor *reactive = clutter_rectangle_new_with_color(clutter_color_new(0,0,0,0));
	g_signal_connect(CLUTTER_ACTOR(reactive), "button-press-event", G_CALLBACK(move_rolls), actor_array);
	g_signal_connect(CLUTTER_ACTOR(reactive), "motion-event", G_CALLBACK(roll_loop), actor_array);
	g_signal_connect(CLUTTER_ACTOR(reactive), "button-release-event", G_CALLBACK(stop_rolls), actor_array);
	clutter_actor_set_reactive(reactive, TRUE);
	clutter_actor_set_size(reactive, 95.0, 285.0);
	clutter_container_add_actor(CLUTTER_CONTAINER(actor_array->roll_group), reactive);
	clutter_actor_set_position(reactive, 0.0, 95.0);
}

ClutterActor *create_loop_view_rolls() {
	rhytmical_sample_looper = get_actor_array_new();
	melodical_sample_looper = get_actor_array_new();
	harmonical_sample_looper = get_actor_array_new();
	effect_sample_looper = get_actor_array_new();

	ClutterActor *looper_container = clutter_group_new();

	rhytmical_sample_looper->roll_group = config_get_configured_actor("sequencer", "rhytmical_samples");
	melodical_sample_looper->roll_group = config_get_configured_actor("sequencer", "melodical_samples");
	harmonical_sample_looper->roll_group = config_get_configured_actor("sequencer", "harmonical_samples");
	effect_sample_looper->roll_group = config_get_configured_actor("sequencer", "effect_samples");

	ClutterActor *loop_mask = config_get_configured_actor("sequencer", "loop_container_mask");

	ClutterColor *orange = clutter_color_new(255, 150, 0, 255);
	ClutterColor *purple = clutter_color_new(255, 0, 255, 255);
	ClutterColor *green = clutter_color_new(0, 200, 0, 255);
	ClutterColor *red = clutter_color_new(255, 0, 0, 255);

	create_loop_roll(rhytmical_sample_looper, orange);
	create_loop_roll(melodical_sample_looper, purple);
	create_loop_roll(harmonical_sample_looper, red);
	create_loop_roll(effect_sample_looper, green);

	clutter_color_free(orange);
	clutter_color_free(purple);
	clutter_color_free(green);
	clutter_color_free(red);

	clutter_actor_set_reactive(loop_mask, FALSE);

	clutter_container_add_actor(CLUTTER_CONTAINER(looper_container), rhytmical_sample_looper->roll_group);
	clutter_container_add_actor(CLUTTER_CONTAINER(looper_container), melodical_sample_looper->roll_group);
	clutter_container_add_actor(CLUTTER_CONTAINER(looper_container), harmonical_sample_looper->roll_group);
	clutter_container_add_actor(CLUTTER_CONTAINER(looper_container), effect_sample_looper->roll_group);
	clutter_container_add_actor(CLUTTER_CONTAINER(looper_container), loop_mask);

	clutter_actor_set_position(rhytmical_sample_looper->roll_group, 120.0, 2.0);
	clutter_actor_set_position(melodical_sample_looper->roll_group, 240.0, 2.0);
	clutter_actor_set_position(harmonical_sample_looper->roll_group, 360.0, 2.0);
	clutter_actor_set_position(effect_sample_looper->roll_group, 480.0, 2.0);
	return looper_container;
}

ClutterActor *create_loop_view() {
	sequencer = jammo_sequencer_new();
	ClutterActor *loop_view = clutter_group_new();
	ClutterActor *view = config_get_configured_actor("sequencer", "loop_view");

	clutter_container_add_actor(CLUTTER_CONTAINER(loop_view), create_loop_view_rolls());
	clutter_container_add_actor(CLUTTER_CONTAINER(loop_view), view);

	ClutterActor *rhytmical_loop_lock = config_get_configured_actor("sequencer","rhytmical_loop_lock");
	ClutterActor *melodical_loop_lock = config_get_configured_actor("sequencer","melodical_loop_lock");
	ClutterActor *harmonical_loop_lock = config_get_configured_actor("sequencer","harmonical_loop_lock");
	ClutterActor *effect_loop_lock = config_get_configured_actor("sequencer","effect_loop_lock");

	ClutterActor *rhytmical_loop_unlock = config_get_configured_actor("sequencer","rhytmical_loop_unlock");
	ClutterActor *melodical_loop_unlock = config_get_configured_actor("sequencer","melodical_loop_unlock");
	ClutterActor *harmonical_loop_unlock = config_get_configured_actor("sequencer","harmonical_loop_unlock");
	ClutterActor *effect_loop_unlock = config_get_configured_actor("sequencer","effect_loop_unlock");

	clutter_actor_hide(rhytmical_loop_unlock);
	clutter_actor_hide(melodical_loop_unlock);
	clutter_actor_hide(harmonical_loop_unlock);
	clutter_actor_hide(effect_loop_unlock);

	clutter_actor_set_reactive(rhytmical_loop_lock, TRUE);
	clutter_actor_set_reactive(melodical_loop_lock, TRUE);
	clutter_actor_set_reactive(harmonical_loop_lock, TRUE);
	clutter_actor_set_reactive(effect_loop_lock, TRUE);
	clutter_actor_set_reactive(rhytmical_loop_unlock, TRUE);
	clutter_actor_set_reactive(melodical_loop_unlock, TRUE);
	clutter_actor_set_reactive(harmonical_loop_unlock, TRUE);
	clutter_actor_set_reactive(effect_loop_unlock, TRUE);

	ClutterActor *cannot_fail_mode_button = clutter_container_find_child_by_name(CLUTTER_CONTAINER(loop_view), "cannot_fail_mode_button");
	clutter_actor_set_reactive(cannot_fail_mode_button, TRUE);
	ClutterActor *crazy_mode_button = clutter_container_find_child_by_name(CLUTTER_CONTAINER(loop_view), "crazy_mode_button");
	clutter_actor_set_reactive(crazy_mode_button, TRUE);
	ClutterActor *handle_button = clutter_container_find_child_by_name(CLUTTER_CONTAINER(loop_view), "cannot_fail_mode_button");
	clutter_actor_set_reactive(handle_button, TRUE);
	ClutterActor *button_arrow = clutter_container_find_child_by_name(CLUTTER_CONTAINER(loop_view), "button_arrow");
	clutter_actor_set_reactive(button_arrow, TRUE);

	g_signal_connect (rhytmical_loop_lock, "button-press-event", G_CALLBACK (lock_loop_wheel), rhytmical_loop_unlock);
	g_signal_connect (melodical_loop_lock, "button-press-event", G_CALLBACK (lock_loop_wheel), melodical_loop_unlock);
	g_signal_connect (harmonical_loop_lock, "button-press-event", G_CALLBACK (lock_loop_wheel), harmonical_loop_unlock);
	g_signal_connect (effect_loop_lock, "button-press-event", G_CALLBACK (lock_loop_wheel), effect_loop_unlock);

	g_signal_connect (rhytmical_loop_unlock, "button-press-event", G_CALLBACK (unlock_loop_wheel), rhytmical_loop_lock);
	g_signal_connect (melodical_loop_unlock, "button-press-event", G_CALLBACK (unlock_loop_wheel), melodical_loop_lock);
	g_signal_connect (harmonical_loop_unlock, "button-press-event", G_CALLBACK (unlock_loop_wheel), harmonical_loop_lock);
	g_signal_connect (effect_loop_unlock, "button-press-event", G_CALLBACK (unlock_loop_wheel), effect_loop_lock);

	g_signal_connect (cannot_fail_mode_button, "button-press-event", G_CALLBACK (cannot_fail_loop_positions), NULL);
	g_signal_connect (crazy_mode_button, "button-press-event", G_CALLBACK (randomize_loop_positions), NULL);
	g_signal_connect (button_arrow, "button-press-event", G_CALLBACK (play_all_samples), NULL);
	g_signal_connect (handle_button, "button-press-event", G_CALLBACK (take_samples_to_sequencer), NULL);

	return loop_view;
}
