/*
 * jammo-mentor.c
 *
 * This file is part of JamMo.
 *
 * (c) 2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-mentor.h"
#include "../meam/jammo-sample.h"
#include <time.h>

G_DEFINE_TYPE(JammoMentor, jammo_mentor, TANGLE_TYPE_BUTTON);

enum {
	PROP_0,
	PROP_STANDBY_SCALE,
	PROP_ACTIVE_SCALE,
	PROP_IDLE_SPEECH,
	PROP_IDLE_TIMEOUT
};

struct _JammoMentorPrivate {
	gfloat standby_scale;
	gfloat active_scale;
	gchar* idle_speech;
	gulong idle_timeout;
	
	GList* spoken_speeches;
	ClutterScript* script;
	JammoSample* active_sample;

	ClutterActor* stage;
	gulong stage_captured_event_handler_id;
	guint timeout_event_source_id;
	time_t last_activity_time;
	
	guint idle_speech_spoken : 1;
};

static void speak(JammoMentor* mentor, const gchar* speech, JammoMentorSpokenCallback callback, gpointer user_data);
static void set_idle_timeout(JammoMentor* mentor, gboolean activate);

static GList* mentors;

ClutterActor* jammo_mentor_new(ClutterActor* actor, gfloat standby_scale, gfloat active_scale) {

	return CLUTTER_ACTOR(g_object_new(JAMMO_TYPE_MENTOR, "normal-background-actor", actor, "standby-scale", standby_scale, "active-scale", active_scale, "prefer-background-size", TRUE, NULL));
}

gfloat jammo_mentor_get_standby_scale(JammoMentor* mentor) {

	return mentor->priv->standby_scale;
}

void jammo_mentor_set_standby_scale(JammoMentor* mentor, gfloat scale) {
	if (mentor->priv->standby_scale != scale) {
		mentor->priv->standby_scale = scale;
		if (!mentor->priv->active_sample) {
			tangle_actor_animate(TANGLE_ACTOR(mentor), CLUTTER_EASE_IN_QUAD, 1500, "scale-x", mentor->priv->standby_scale, "scale-y", mentor->priv->standby_scale, NULL);
		}
		g_object_notify(G_OBJECT(mentor), "standby-scale");
	}
}

gfloat jammo_mentor_get_active_scale(JammoMentor* mentor) {

	return mentor->priv->active_scale;
}

void jammo_mentor_set_active_scale(JammoMentor* mentor, gfloat scale) {
	if (mentor->priv->active_scale != scale) {
		mentor->priv->active_scale = scale;
		if (mentor->priv->active_sample) {
			tangle_actor_animate(TANGLE_ACTOR(mentor), CLUTTER_EASE_IN_QUAD, 1500, "scale-x", mentor->priv->active_scale, "scale-y", mentor->priv->active_scale, NULL);
		}
		g_object_notify(G_OBJECT(mentor), "active-scale");
	}
}

void jammo_mentor_speak(JammoMentor* mentor, const gchar* speech) {
	speak(mentor, speech, NULL, NULL);
}

void jammo_mentor_speak_with_callback(JammoMentor* mentor, const gchar* speech, JammoMentorSpokenCallback callback, gpointer user_data) {
	speak(mentor, speech, callback, user_data);
}

void jammo_mentor_speak_once(JammoMentor* mentor, const gchar* speech) {
	if (!g_list_find_custom(mentor->priv->spoken_speeches, speech, (GCompareFunc)g_strcmp0)) {
		mentor->priv->spoken_speeches = g_list_prepend(mentor->priv->spoken_speeches, g_strdup(speech));
		speak(mentor, speech, NULL, NULL);
	}
}

void jammo_mentor_speak_once_with_callback(JammoMentor* mentor, const gchar* speech, JammoMentorSpokenCallback callback, gpointer user_data) {
	if (!g_list_find_custom(mentor->priv->spoken_speeches, speech, (GCompareFunc)g_strcmp0)) {
		mentor->priv->spoken_speeches = g_list_prepend(mentor->priv->spoken_speeches, g_strdup(speech));
		speak(mentor, speech, callback, user_data);
	}
}

void jammo_mentor_shut_up(JammoMentor* mentor) {
	if (mentor->priv->active_sample) {
		jammo_sample_stop(mentor->priv->active_sample);
	}
}

const gchar* jammo_mentor_get_idle_speech(JammoMentor* mentor) {

	return mentor->priv->idle_speech;
}

void jammo_mentor_set_idle_speech(JammoMentor* mentor, const gchar* speech) {
	if (g_strcmp0(mentor->priv->idle_speech, speech)) {
		g_free(mentor->priv->idle_speech);
		mentor->priv->idle_speech = g_strdup(speech);
		mentor->priv->idle_speech_spoken = FALSE;
		set_idle_timeout(mentor, TRUE);
		g_object_notify(G_OBJECT(mentor), "idle-speech");
	}
}

gulong jammo_mentor_get_idle_timeout(JammoMentor* mentor) {

	return mentor->priv->idle_timeout;
}

void jammo_mentor_set_idle_timeout(JammoMentor* mentor, gulong milliseconds) {
	if (mentor->priv->idle_timeout != milliseconds) {
		mentor->priv->idle_timeout = milliseconds;
		set_idle_timeout(mentor, TRUE);
		g_object_notify(G_OBJECT(mentor), "idle-timeout");
	}
}

JammoMentor* jammo_mentor_get_default(void) {

	return (mentors ? JAMMO_MENTOR(mentors->data) : NULL);
}

static gboolean jammo_mentor_clicked(TangleButton* button) {
	JammoMentor* mentor;
	
	mentor = JAMMO_MENTOR(button);
	
	if (mentor->priv->active_sample) {
		jammo_sample_stop(mentor->priv->active_sample);
	} else if (mentor->priv->idle_speech) {
		speak(mentor, mentor->priv->idle_speech, NULL, NULL);
	}

	return FALSE;
}

static void jammo_mentor_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoMentor* mentor;
	
	mentor = JAMMO_MENTOR(object);

	switch (prop_id) {
		case PROP_STANDBY_SCALE:
			jammo_mentor_set_standby_scale(mentor, g_value_get_float(value));
			break;
		case PROP_ACTIVE_SCALE:
			jammo_mentor_set_active_scale(mentor, g_value_get_float(value));
			break;
		case PROP_IDLE_SPEECH:
			jammo_mentor_set_idle_speech(mentor, g_value_get_string(value));
			break;
		case PROP_IDLE_TIMEOUT:
			jammo_mentor_set_idle_timeout(mentor, g_value_get_ulong(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_mentor_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoMentor* mentor;

	mentor = JAMMO_MENTOR(object);

        switch (prop_id) {
		case PROP_STANDBY_SCALE:
			g_value_set_float(value, mentor->priv->standby_scale);
			break;
		case PROP_ACTIVE_SCALE:
			g_value_set_float(value, mentor->priv->active_scale);
			break;
		case PROP_IDLE_SPEECH:
			g_value_set_string(value, mentor->priv->idle_speech);
			break;
		case PROP_IDLE_TIMEOUT:
			g_value_set_ulong(value, mentor->priv->idle_timeout);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_mentor_show(ClutterActor* actor) {
	set_idle_timeout(JAMMO_MENTOR(actor), TRUE);
	
	CLUTTER_ACTOR_CLASS(jammo_mentor_parent_class)->show(actor);
}

static void jammo_mentor_hide(ClutterActor* actor) {
	set_idle_timeout(JAMMO_MENTOR(actor), FALSE);
	
	CLUTTER_ACTOR_CLASS(jammo_mentor_parent_class)->hide(actor);
}

static void jammo_mentor_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_mentor_parent_class)->finalize(object);
}

static void jammo_mentor_dispose(GObject* object) {
	set_idle_timeout(JAMMO_MENTOR(object), FALSE);

	G_OBJECT_CLASS(jammo_mentor_parent_class)->dispose(object);
}

static void jammo_mentor_class_init(JammoMentorClass* mentor_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(mentor_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(mentor_class);
	TangleButtonClass* button_class = TANGLE_BUTTON_CLASS(mentor_class);

	gobject_class->finalize = jammo_mentor_finalize;
	gobject_class->dispose = jammo_mentor_dispose;
	gobject_class->set_property = jammo_mentor_set_property;
	gobject_class->get_property = jammo_mentor_get_property;

	clutter_actor_class->show = jammo_mentor_show;
	clutter_actor_class->hide = jammo_mentor_hide;

	button_class->clicked = jammo_mentor_clicked;

	/**
	 * JammoMentor:standby-scale:
	 *
	 * The scaling that is efective when the mentor is in standby.
	 */
	g_object_class_install_property(gobject_class, PROP_STANDBY_SCALE,
	                                g_param_spec_float("standby-scale",
	                                                   "Standby scale",
	                                                   "The scaling that is efective when the mentor is in standby",
	                                                   0.0, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_CONSTRUCT));
	/**
	 * JammoMentor:active-scale:
	 *
	 * The scaling that is efective when the mentor is active.
	 */
	g_object_class_install_property(gobject_class, PROP_ACTIVE_SCALE,
	                                g_param_spec_float("active-scale",
	                                                   "Active scale",
	                                                   "The scaling that is efective when the mentor is active",
	                                                   0.0, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_CONSTRUCT));
	/**
	 * JammoMentor:idle-speech:
	 *
	 * The speech that the mentor will speak after a specified time (see :idle-time).
	 */
	g_object_class_install_property(gobject_class, PROP_IDLE_SPEECH,
	                                g_param_spec_string("idle-speech",
	                                                    "Idle speech",
	                                                    "The speech that the mentor will speak after a specified time",
	                                                    NULL,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_CONSTRUCT));
	/**
	 * JammoMentor:idle-time:
	 *
	 * The time in seconds after the mentor will speak (see :idle-speech) if an user has been idle.
	 * Value 0 means disabled.
	 */
	g_object_class_install_property(gobject_class, PROP_IDLE_TIMEOUT,
	                                g_param_spec_ulong("idle-timeout",
	                                                   "Idle timeout",
	                                                   "The time in seconds after the mentor will speak if an user has been idle.",
	                                                   0, G_MAXULONG, 0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_CONSTRUCT));

	g_type_class_add_private (gobject_class, sizeof (JammoMentorPrivate));
}

static void jammo_mentor_init(JammoMentor* mentor) {
	mentor->priv = G_TYPE_INSTANCE_GET_PRIVATE(mentor, JAMMO_TYPE_MENTOR, JammoMentorPrivate);

	mentors = g_list_append(mentors, mentor);
}

static void on_sample_stopped(JammoSample* sample, gpointer user_data) {
	TangleVault* vault;
	JammoMentor* mentor;
	JammoMentorSpokenCallback callback;
	const gchar* speech;
	gpointer data;
	
	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 4, JAMMO_TYPE_MENTOR, &mentor, G_TYPE_POINTER, &callback, G_TYPE_STRING, &speech, G_TYPE_POINTER, &data);
	
	g_object_unref(mentor->priv->active_sample);
	mentor->priv->active_sample = NULL;

	tangle_actor_animate(TANGLE_ACTOR(mentor), CLUTTER_EASE_IN_QUAD, 1000, "scale-x", mentor->priv->standby_scale, "scale-y", mentor->priv->standby_scale, NULL);

	set_idle_timeout(mentor, TRUE);
	
	if (callback) {
		callback(mentor, speech, data);
	}
}

static void speak(JammoMentor* mentor, const gchar* speech, JammoMentorSpokenCallback callback, gpointer user_data) {
	gchar* filename;
	TangleVault* vault;
	
	set_idle_timeout(mentor, FALSE);
	tangle_actor_show(TANGLE_ACTOR(mentor));
	
	if (mentor->priv->active_sample) {
		jammo_sample_stop(mentor->priv->active_sample);
	}
	
	if (mentor->priv->script && (filename = clutter_script_lookup_filename(mentor->priv->script, speech))) {
		mentor->priv->active_sample = jammo_sample_new_from_file(filename);
		g_free(filename);
	} else {
		mentor->priv->active_sample = jammo_sample_new_from_file(speech);
	}
	
	vault = tangle_vault_new(4, JAMMO_TYPE_MENTOR, mentor, G_TYPE_POINTER, callback, G_TYPE_STRING, speech, G_TYPE_POINTER, user_data);
	tangle_signal_connect_vault(mentor->priv->active_sample, "stopped", G_CALLBACK(on_sample_stopped), vault);
	
	tangle_actor_animate(TANGLE_ACTOR(mentor), CLUTTER_EASE_IN_QUAD, 1000, "scale-x", mentor->priv->active_scale, "scale-y", mentor->priv->active_scale, NULL);
	
	jammo_sample_play(mentor->priv->active_sample);
}

static gboolean on_stage_captured_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	JammoMentor* mentor;
	
	mentor = JAMMO_MENTOR(user_data);
	
	mentor->priv->last_activity_time = time(NULL);
	
	return FALSE;
}

static gboolean on_timeout(gpointer user_data) {
	JammoMentor* mentor;
	time_t elapsed_time;
	
	mentor = JAMMO_MENTOR(user_data);
	
	elapsed_time = time(NULL) - mentor->priv->last_activity_time;
	if (elapsed_time >= mentor->priv->idle_timeout) {
		mentor->priv->timeout_event_source_id = 0;
		mentor->priv->idle_speech_spoken = TRUE;
		speak(mentor, mentor->priv->idle_speech, NULL, NULL);
	} else {
		mentor->priv->timeout_event_source_id = g_timeout_add((mentor->priv->idle_timeout - elapsed_time) * 1000, on_timeout, mentor);
	}
	
	return FALSE;
}

static void set_idle_timeout(JammoMentor* mentor, gboolean activate) {
	if (mentor->priv->timeout_event_source_id) {
		g_source_remove(mentor->priv->timeout_event_source_id);
		mentor->priv->timeout_event_source_id = 0;
	}
	
	if (activate && mentor->priv->idle_timeout && mentor->priv->idle_speech && !mentor->priv->idle_speech_spoken && !mentor->priv->active_sample && CLUTTER_ACTOR_IS_VISIBLE(mentor)) {
		if (!mentor->priv->stage && (mentor->priv->stage = clutter_actor_get_stage(CLUTTER_ACTOR(mentor)))) {
			mentor->priv->stage_captured_event_handler_id = g_signal_connect(mentor->priv->stage, "captured-event", G_CALLBACK(on_stage_captured_event), mentor);
		}

		mentor->priv->last_activity_time = time(NULL);

		mentor->priv->timeout_event_source_id = g_timeout_add(mentor->priv->idle_timeout * 1000, on_timeout, mentor);
	} else if (!activate && mentor->priv->stage) {
		g_signal_handler_disconnect(mentor->priv->stage, mentor->priv->stage_captured_event_handler_id);
		mentor->priv->stage = NULL;
	}
}
