/*
 * jammo-mentor-action.h
 *
 * This file is part of JamMo.
 *
 * (c) 2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-mentor-action.h"

G_DEFINE_TYPE(JammoMentorAction, jammo_mentor_action, TANGLE_TYPE_ACTION);

enum {
	PROP_0,
	PROP_MENTOR,
	PROP_SPEECH,
	PROP_SPEAK_ONCE,
	PROP_IDLE_SPEECH,
	PROP_SPOKEN_ACTION
};

enum {
	SPOKEN,
	LAST_SIGNAL
};

struct _JammoMentorActionPrivate {
	JammoMentor* mentor;
	gchar* speech;
	gchar* idle_speech;
	TangleAction* spoken_action;
	guint speak_once : 1;
};

static void on_mentor_spoken(JammoMentor* mentor, const gchar* speech, gpointer user_data);

static guint signals[LAST_SIGNAL] = { 0 };

TangleAction* jammo_mentor_action_new(void) {

	return TANGLE_ACTION(g_object_new(JAMMO_TYPE_MENTOR_ACTION, NULL));
}

JammoMentor* jammo_mentor_action_get_mentor(JammoMentorAction* action) {

	return action->priv->mentor;
}

void jammo_mentor_action_set_mentor(JammoMentorAction* action, JammoMentor* mentor) {
	if (action->priv->mentor != mentor) {
		if (action->priv->mentor) {
			g_object_unref(action->priv->mentor);
		}
		action->priv->mentor = mentor;
		g_object_ref(action->priv->mentor);
		g_object_notify(G_OBJECT(action), "mentor");
	}
}

const gchar* jammo_mentor_action_get_speech(JammoMentorAction* action) {

	return action->priv->speech;
}

void jammo_mentor_action_set_speech(JammoMentorAction* action, const gchar* speech) {
	if (g_strcmp0(action->priv->speech, speech)) {
		g_free(action->priv->speech);
		action->priv->speech = g_strdup(speech);
		g_object_notify(G_OBJECT(action), "speech");	
	}
}

gboolean jammo_mentor_action_get_speak_once(JammoMentorAction* action) {

	return action->priv->speak_once;
}

void jammo_mentor_action_set_speak_once(JammoMentorAction* action, gboolean once) {
	if (action->priv->speak_once != once) {
		action->priv->speak_once = once;
		g_object_notify(G_OBJECT(action), "speak-once");	
	}
}

const gchar* jammo_mentor_action_get_idle_speech(JammoMentorAction* action) {

	return action->priv->idle_speech;
}

void jammo_mentor_action_set_idle_speech(JammoMentorAction* action, const gchar* speech) {
	if (g_strcmp0(action->priv->idle_speech, speech)) {
		g_free(action->priv->idle_speech);
		action->priv->idle_speech = g_strdup(speech);
		g_object_notify(G_OBJECT(action), "idle-speech");	
	}
}

static void jammo_mentor_action_execute(TangleAction* tangle_action, GObject* source, const gchar* trigger, TangleProperties* properties) {
	JammoMentorAction* action;
	JammoMentor* mentor;
	
	action = JAMMO_MENTOR_ACTION(tangle_action);
	if (!(mentor = action->priv->mentor)) {
		mentor = jammo_mentor_get_default();
	}
	
	g_return_if_fail(mentor != NULL);
	g_return_if_fail(action->priv->speech != NULL || action->priv->idle_speech != NULL);
	
	jammo_mentor_shut_up(mentor);
	if (action->priv->speech) {
		if (action->priv->speak_once) {
			jammo_mentor_speak_once_with_callback(mentor, action->priv->speech, on_mentor_spoken, action);
		} else {
			jammo_mentor_speak_with_callback(mentor, action->priv->speech, on_mentor_spoken, action);
		}
	}
	if (action->priv->idle_speech) {
		jammo_mentor_set_idle_speech(mentor, action->priv->idle_speech);
	}
}

static void jammo_mentor_action_spoken(JammoMentorAction* action, const gchar* speech) {
	TangleProperties* properties;
	
	if (action->priv->spoken_action) {
		properties = tangle_properties_new();
		tangle_properties_set(properties, "speech", G_TYPE_STRING, speech, NULL);
		tangle_action_execute_full(action->priv->spoken_action, G_OBJECT(action), "spoken", properties);
		g_object_unref(G_OBJECT(properties));
	}
}

static void jammo_mentor_action_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoMentorAction* action;
	
	action = JAMMO_MENTOR_ACTION(object);

	switch (prop_id) {
		case PROP_MENTOR:
			jammo_mentor_action_set_mentor(action, JAMMO_MENTOR(g_value_get_object(value)));
			break;
		case PROP_SPEECH:
			jammo_mentor_action_set_speech(action, g_value_get_string(value));
			break;
		case PROP_SPEAK_ONCE:
			jammo_mentor_action_set_speak_once(action, g_value_get_boolean(value));
			break;
		case PROP_IDLE_SPEECH:
			jammo_mentor_action_set_idle_speech(action, g_value_get_string(value));
			break;
		case PROP_SPOKEN_ACTION:
			if (action->priv->spoken_action) {
				g_object_unref(action->priv->spoken_action);
			}
			action->priv->spoken_action = TANGLE_ACTION(g_object_ref(g_value_get_object(value)));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_mentor_action_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoMentorAction* action;

	action = JAMMO_MENTOR_ACTION(object);

        switch (prop_id) {
		case PROP_MENTOR:
			g_value_set_object(value, action->priv->mentor);
			break;
		case PROP_SPEECH:
			g_value_set_string(value, action->priv->speech);
			break;
		case PROP_SPEAK_ONCE:
			g_value_set_boolean(value, action->priv->speak_once);
			break;
		case PROP_IDLE_SPEECH:
			g_value_set_string(value, action->priv->idle_speech);
			break;
		case PROP_SPOKEN_ACTION:
			g_value_set_object(value, action->priv->spoken_action);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_mentor_action_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_mentor_action_parent_class)->finalize(object);
}

static void jammo_mentor_action_dispose(GObject* object) {
	G_OBJECT_CLASS(jammo_mentor_action_parent_class)->dispose(object);
}

static void jammo_mentor_action_class_init(JammoMentorActionClass* mentor_action_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(mentor_action_class);
	TangleActionClass* action_class = TANGLE_ACTION_CLASS(mentor_action_class);

	gobject_class->finalize = jammo_mentor_action_finalize;
	gobject_class->dispose = jammo_mentor_action_dispose;
	gobject_class->set_property = jammo_mentor_action_set_property;
	gobject_class->get_property = jammo_mentor_action_get_property;

	action_class->execute = jammo_mentor_action_execute;
	
	mentor_action_class->spoken = jammo_mentor_action_spoken;

	/**
	 * JammoMentorAction:mentor:
	 *
	 * If :mentor is NULL, the default mentor (see jammo_mentor_get_default()) is used.
	 */
	g_object_class_install_property(gobject_class, PROP_MENTOR,
	                                g_param_spec_object("mentor",
	                                                    "Mentor",
	                                                    "The mentor",
	                                                    JAMMO_TYPE_MENTOR,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * JammoMentorAction:speech:
	 *
	 * The speech that the mentor should speak when the action is executed.
	 * Can be NULL, if the purpose of the action is only to set idle speech.
	 */
	g_object_class_install_property(gobject_class, PROP_SPEECH,
	                                g_param_spec_string("speech",
	                                                    "Speech",
	                                                    "The speech that the mentor should speak when the action is executed",
	                                                    NULL,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * JammoMentorAction:speak-once:
	 *
	 * Whether the speech should be spoken only once in a lifetime of a mentor
	 */
	g_object_class_install_property(gobject_class, PROP_SPEAK_ONCE,
	                                g_param_spec_boolean("speak-once",
	                                                     "Speak once",
	                                                     "Whether the speech should be spoken only once in a lifetime of a mentor",
	                                                     FALSE,
	                                                     G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * JammoMentorAction:idle-speech:
	 *
	 * The speech that the mentor should speak when an user is idling.
	 * Leave this NULL, if you do not want to change idle speech when the action is executed.
	 */
	g_object_class_install_property(gobject_class, PROP_IDLE_SPEECH,
	                                g_param_spec_string("idle-speech",
	                                                    "Idle speech",
	                                                    "The speech that the mentor should speak when an user is idling",
	                                                    NULL,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * JammoMentorAction:spoken-action:
	 *
	 * An action that is executed when the mentor has spoken the speech.
	 */
	g_object_class_install_property(gobject_class, PROP_SPOKEN_ACTION,
	                                g_param_spec_string("spoken-action",
	                                                    "Spoken action",
	                                                    "An action that is executed when the mentor has spoken the speech",
	                                                    NULL,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	/**
	 * JammoMentorAction::spoken:
	 * @action: the object which received the signal
	 *
	 * The ::spoken signal is emitted when the action with :speech property has
	 * been completed (the mentor has spoken).
	 */
	signals[SPOKEN] = g_signal_new("spoken", G_TYPE_FROM_CLASS(gobject_class),
	                               G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(JammoMentorActionClass, spoken),
				       NULL, NULL,
				       g_cclosure_marshal_VOID__STRING,
				       G_TYPE_NONE, 1,
				       G_TYPE_STRING);
						   
	g_type_class_add_private (gobject_class, sizeof (JammoMentorActionPrivate));
}

static void jammo_mentor_action_init(JammoMentorAction* mentor_action) {
	mentor_action->priv = G_TYPE_INSTANCE_GET_PRIVATE(mentor_action, JAMMO_TYPE_MENTOR_ACTION, JammoMentorActionPrivate);
}

static void on_mentor_spoken(JammoMentor* mentor, const gchar* speech, gpointer user_data) {
	JammoMentorAction* action;
	
	action = JAMMO_MENTOR_ACTION(user_data);
	
	g_signal_emit(action, signals[SPOKEN], 0, speech);
}
