/**sequencer.c is part of JamMo.
License: GPLv2, read more from COPYING

This is one module of MEAM. This handles sequencer.
*/

#include <string.h>
#include "sequencer.h"

/*JamMo08a - sequencer module - please check readme.txt for more details! */

/**
 *  sequencer_init - Initializes Gstreamer functionalities 
 *  @return 0
 */
gboolean sequencer_init(void) 
{
	return pipeline_init();
}

/**
 *  sequencer_deinit - Frees all dynamically allocated memory with the help of our own freeing functions
 */
void sequencer_deinit(void)
{
	free_pipeline();
}


gboolean (*show_timeout_function)(void) ;
guint timeout_interval;
/*
Everytime track end, timeline shower stops too, and we must start timeout_callback again.
So we must know that callback function. 
time in milliseconds
*/
void set_timeout_callback (gboolean (*func_ptr)(void), guint interval){
show_timeout_function=func_ptr;
timeout_interval=interval;
printf("set: time_out callback\n");
}

/**
*  seq_loop_is_running - Checks if the Gstreamer is running a the main loop. Usable for debugging.
*  @return gboolean TRUE if the loop is running or FALSE if the loop is not running.
*/
gboolean seq_loop_is_running (void) 
{
	Pipeline * pipe = get_mainPipeline();
	if (g_main_loop_is_running(pipe->loop))
		return TRUE;
	else
		return FALSE;
}

/**
 *  seq_get_element_state - Returns the state of the wanted element. This function is mainly used to get states from the pipeline -element.
 *  This function is very useful in GUI, you can hold/wait for states to be certain. Note that in Gstreamer the current states are not always the most important to know, you might want to know the state transitions/state changes. 
 * Usage: If you want to know if your pipelines current state is PLAYING, you can check it for example "g_print("%s",seq_get_element_state()) or if you want to do something when your pipeline is in some specific state : if(strcmp(seq_get_element_state(),"PLAYING"))
 * For tracking state changes please study yourself with GST_MESSAGE_STATE_CHANGED, an example of tracking changes in bus:
	case GST_MESSAGE_STATE_CHANGED: {
     	GstState state , pending;
      	gst_message_parse_state_changed (msg, NULL, &state, &pending);
      	g_print("%s is %s, pending %s\n", GST_ELEMENT_NAME(GST_MESSAGE_SRC(msg)), 
	gst_element_state_get_name(state), gst_element_state_get_name(pending));
	break;
	}
 *  @return String that describes the state (of the pipeline): VOID_PENDING , NULL , READY , PAUSED or PLAYING.
*/
const gchar * seq_get_element_state (void) 
{
	gint state; // 0 or 1 or 2 or 3 or 4
	const gchar *state_string;
	/*
	 GST_STATE_VOID_PENDING        = 0,
	  GST_STATE_NULL                = 1,
	  GST_STATE_READY               = 2,
	  GST_STATE_PAUSED              = 3,
	  GST_STATE_PLAYING             = 4
	*/
	Pipeline * pipe = get_mainPipeline();
	state = GST_STATE(pipe->pipeline);
	state_string = gst_element_state_get_name (GST_STATE(pipe->pipeline));

	return state_string;
}

static gboolean end_sequencer_when_end(){
printf("All elements in timeline played!\n");
seq_stop();
return FALSE;
}

/**
 *  seq_play - Starts playing the timeline by setting the pipeline 
 *  state to PLAYING.
 *  Usage: In GUI put this to the "PLAY" -button. Note that there must be something to be played before using this function.
 */
void seq_play (void) {
    //g_timeout_add (timeout_interval, (GSourceFunc) show_timeout_function, NULL); 
    gst_element_set_state(get_mainPipeline()->pipeline, GST_STATE_PLAYING);
    guint64 end_time=get_mainPipeline()->pipelines_end_time;
    guint interval = end_time / GST_NSECOND /1000000;
    g_print(" Will stop on time: %" GST_TIME_FORMAT " = %d ns\n",GST_TIME_ARGS(end_time),interval);

    #ifdef N810
      /*On N810 it takes approx 2 seconds to start (and other slow computers) */ 
      interval+=2000;   /*FIX: Maybe this seq_stop should be handled very different way.*/
      g_print("Added 2 seconds\n");
    #else
      g_print("Not N810\n");
    #endif

    g_timeout_add(interval, (GSourceFunc) end_sequencer_when_end, NULL); 

}


/**
 *  seq_stop - Stops current timeline/playing process (if pipeline is in PLAYING or PAUSED -mode) and timeline is initialized back 
 *  to start (sets pipeline to NULL -state). Note that gst_stop does nothing if timeline is not on the
 *  RUNNING state.
 *  Usage: In GUI put this to the "STOP" -button.
 */
void seq_stop(void) 
{
	if( (g_main_loop_is_running(get_mainPipeline()->loop) && strcmp(seq_get_element_state(),"PLAYING")) || strcmp(seq_get_element_state(),"PAUSED"))
	{
		gst_element_set_state (get_mainPipeline()->pipeline, GST_STATE_NULL); // the NULL state or initial state of the element
	}
}

/**
 *  seq_pause - This function pauses timeline if song is played. This 
 *  function makes it possible to start playing from the pause-state 
 *  also (differs from gst_stop by that). Note that gst_pause does nothing 
 *  if timeline is not on the RUNNING -state.
 *  Usage: In GUI put this to the "PAUSE" -button.
 */
void seq_pause(void) {
	if(g_main_loop_is_running(get_mainPipeline()->loop))
		gst_element_set_state(get_mainPipeline()->pipeline, GST_STATE_PAUSED); // the element is PAUSED
}

#ifdef FIX_THIS
/* Do not use this until rethinked.*/
/**
 *  seq_quit - Closes gstreamer threads when closing the main 
 *  application. seq_quit quits the main loop (the gstreamer 
 *  functionalities by that). 
 *  Usage: In GUI call this function at latest on the window destroy functionalities, it frees resources and memory.
 */
void seq_quit(void) {
	/* If song is played (timeline initialized by GST_STATE) 
	and gstreamer is running in a loop */
	if(g_main_loop_is_running(get_mainPipeline()->loop))
	{
		g_main_loop_quit(get_mainPipeline()->loop);
		gst_element_set_state (get_mainPipeline()->pipeline, GST_STATE_NULL);
		gst_object_unref (GST_OBJECT (get_mainPipeline()->pipeline));
		sequencer_deinit();
	}
	else // If song is played then we should unref the pipeline, if not -> do nothing.
	{
		gst_element_set_state (get_mainPipeline()->pipeline, GST_STATE_NULL);
		gst_object_unref (GST_OBJECT (get_mainPipeline()->pipeline));
		sequencer_deinit();
	}
	gtk_main_quit();
}
#endif


void seq_get_lenght_of_pipeline(guint64 *arg){
    guint64 end_time=get_mainPipeline()->pipelines_end_time;
    guint interval = end_time / GST_NSECOND /1000000;
    g_print(" Will2 stop on time: %" GST_TIME_FORMAT " = %d ns\n",GST_TIME_ARGS(end_time),interval);
*arg=end_time;
return;

}

#ifdef ONLY_FOR_DEBUG
NOT WORKING
/**
 *  Prints / gets current position of the loop/song/timeline 
 *  and displays the total length of a timeline. Function currently prints position and timelines length to console.
 *  <br>
 *  Usage: in GUI the current position of the timeline can be get by using global variable 'pos'
 *  Timelines/pipelines length can be checked by 'get_mainPipeline()->pipelines_end_time'
 */
gboolean seq_print_position(void)
{
  	GstFormat fmt = GST_FORMAT_TIME;
	fmt = GST_FORMAT_TIME;
	 
    	if(GST_IS_ELEMENT(get_mainPipeline()->pipeline))
    	{
    		if (gst_element_query_position (get_mainPipeline()->pipeline, &fmt, &pos)
	    		&& gst_element_query_duration (get_mainPipeline()->pipeline, &fmt, &len) 
			&& g_main_loop_is_running(get_mainPipeline()->loop)) {

   			g_print ("Time: %" GST_TIME_FORMAT " / %" GST_TIME_FORMAT "\n",
	     		GST_TIME_ARGS (pos), GST_TIME_ARGS (get_mainPipeline()->pipelines_end_time)); // prints current time (pos) and current pipelines end time (get_mainPipeline()->pipelines_end_time) to console

    		}
    	}

  	if(!g_main_loop_is_running(get_mainPipeline()->loop))
  		return FALSE;

  	/* call me again if the loop is running, else return FALSE and this function distinguishes */
  	if(g_main_loop_is_running(get_mainPipeline()->loop))
  	{
  		return TRUE;
  	}
  	else
  	{
		g_print("\nSong ended/stopped\n");
	  	return FALSE;
  	}
}
#endif 

/**
 *  seq_seek_to_time -function, takes time in nanoseconds (for seconds please use the seq_seek_to_time_seconds).
 *  Moves to wanted position in the timeline (and continues to play onwards on the new position if the pipeline 
 *  state was PLAYING.) Note that seek_to_time functions can be done if the pipeline state is PLAYING or PAUSED, 
 *  READY or NULL -states won't work!
 * <br>
 * Usage: Usually seq_seek_to_time called from the GUI. If you want to move/jump to 10sec away from the start the usage is as follows: 	* seq_seek_to_time(10000000000LLU); More usage/help can be found in main.c.
*/
void
seq_seek_to_time (guint64 time_nanoseconds)
{
	if(g_main_loop_is_running(get_mainPipeline()->loop))
    	{
      		if (!gst_element_seek (get_mainPipeline()->pipeline, 1.0, GST_FORMAT_TIME, GST_SEEK_FLAG_FLUSH,
                         						GST_SEEK_TYPE_SET, time_nanoseconds,
                         						GST_SEEK_TYPE_NONE, GST_CLOCK_TIME_NONE)) 
		{
      		g_print ("Seek 'failed'! timeline stopped or some loops are too short (don't panic!)\n");
      		}
    	}
}

/**
*seq_seek_to_time_seconds - Same function as seq_seek_to_time, but time is given in full seconds (1 or 10 etc).
* Note: If you want to start playing from the beginning (from the zero/0) you can try to seq_seek_to_time_seconds(0) OR you can call seq_stop() and seq_play() after that (and if your pipeline is in PAUSED state, add seq_pause to that). Thats basically the same idea.
*/
void seq_seek_to_time_seconds (guint64 time_seconds)
{

  	if(g_main_loop_is_running(get_mainPipeline()->loop))
    	{
      		if (!gst_element_seek (get_mainPipeline()->pipeline, 1.0, GST_FORMAT_TIME, GST_SEEK_FLAG_FLUSH,
                         GST_SEEK_TYPE_SET, time_seconds * 1000000000 ,
                         GST_SEEK_TYPE_NONE, GST_CLOCK_TIME_NONE))
		{
      		g_print ("Seek 'failed'! timeline stopped or some loops are too short (don't panic, the seek is likely to have completed!)\n");
      		}
    	}
}

/*
* seek to percents - do we need this? Percents are more useful on streams... TO do if we have time..
*
void seek_to_time_percents (gint64 percents)
{
  if(counter == 1 && g_main_loop_is_running(loop))
    {
      if (!gst_element_seek (pipeline, 1.0, GST_FORMAT_PERCENT, GST_SEEK_FLAG_FLUSH,
                         GST_SEEK_TYPE_SET, GST_FORMAT_PERCENT_SCALEpercents ,
                         GST_SEEK_TYPE_NONE, GST_CLOCK_TIME_NONE)) {

      g_print ("Seek failed! (timeline stopped or loop too short)\n");
      }
    }

} */


/*
Just to avoid that user of this library isn't forced to use g_list-things.
*/
Track *get_nth_track_from_pipeline(int track_number) {
return g_list_nth_data(get_mainPipeline()->tracks, track_number);
}
