/**terminal_jammo.c is part of JamMo.
License: GPLv2, read more from COPYING

This is 'Graphical user interface' only because it implements all functions in
gui.h. Actually it is textual user interface not meant to use by anybody, only
for debugging.
*/
//This is because of getline
#define _GNU_SOURCE

#include "../chum/chum.h"
#include "../chum/gui.h"

#include "terminal_jammo.h"

#include <stdio.h>
#include <termios.h>
#include <unistd.h>
#include <pthread.h>
#include <string.h>

#include "instrument_mode.h"
#include "instrument_mode_joystick.h"

void gui_visualize_contour(float freq) {
printf("frequence is %f\n",freq);
}


/*
For reading one char, without echoing it
*/
int mygetch( ) {
  struct termios oldt,
                 newt;
  int            ch;
  tcgetattr( STDIN_FILENO, &oldt );
  newt = oldt;
  newt.c_lflag &= ~( ICANON | ECHO );
  tcsetattr( STDIN_FILENO, TCSANOW, &newt );
  ch = getchar();
  tcsetattr( STDIN_FILENO, TCSANOW, &oldt );
  return ch;
}

void gui_init() {
    printf("Initializing terminal JamMo (empty).\n");
}

void gui_visualize_playing(void) {
    printf("Playing starts!\n");
}

static void show_welcome() {
    printf("Welcome to JamMo, press any key to continue (open door).\n");
    mygetch(); 
    chum_play_one_sample("../share/jammo/door_opening.wav");
    printf("Door opening...\n");
    printf("(you can skip by pressing any key)\n");
    printf("(you must press any key to continue)\n");
    mygetch();
    chum_stop_one_sample();
    chum_go_next();
}

static void show_game_selection() {
    printf("Game selection menu: (only composing available)\n");
    printf("1 = singing \n2 = composing\n3 = song bank\n");
    mygetch();
    chum_go_next();
}

static void show_theme_selection() {
    int number_of_themes =chum_get_number_of_themes();
      if (number_of_themes==0)  {
	  printf("ERROR, JamMo doesn't find any theme.\n");
	  exit(1);
      }
	
    printf("Theme selection menu: press theme number 1-%d. Or 'x' to quit.\n",number_of_themes);
    int i;

    for (i=1;i<number_of_themes+1;i++){
	printf("%d =  %s\n",i,chum_get_theme_folder(i));	
    }

    int ch=mygetch();
    if (ch=='x') exit(0);
    int selected=ch-48;
    if (selected<1 || selected>number_of_themes) {
	printf("not valid selection!\n");
	show_theme_selection();
	return;
    }
    chum_change_to_theme(ch-48);
}


/*We will read THEMES_DIR/theme_folder/objects.json
and copy all image and sample filenames to two GList (they are gchars)
*/
  GList *images = NULL;
  GList *samples = NULL;

static void load_theme(const gchar *theme_folder) {
  printf("Loading theme:%s\n",theme_folder);


//These are needed for file handling
  int bytes_read;
  size_t nbytes = 100;
  char *my_string;
  my_string = (char *) malloc (nbytes + 1);
  int cont=1; /*this used for while-loops*/

 //Read objects.json
  FILE *ifp;
  gchar *objects_filename = g_strdup_printf("%s/%s/objects.json", THEMES_DIR, theme_folder);

  ifp = fopen(objects_filename, "r");
  if (ifp == NULL) {
      fprintf(stderr, "Can't open input file '%s'\n",objects_filename);
      exit(1);
  }

  //This is the loop which read whole file until end is reached
  while (cont){
    bytes_read = getline (&my_string, &nbytes, ifp);
    if (bytes_read <= 0) {
       cont=0;
    } else {
	//printf("readed: %s",my_string);
	char *pch=NULL;
	pch = strstr (my_string,"\"filename\"");  //if readed line contains text '"filename"'
	if (pch!=NULL){ 
	    char *pch2=NULL;
	    pch2 = strstr (my_string,":"); //there can be spaces before or after colon (':')
	    char *pch3=NULL;
	    pch3 = strstr (pch2,"\"");	 //this is founded starting quotation mark of looked name

	    //We need true memory block for this (freeing is our job too)
	    gchar *name;
	    name = g_strndup (pch3+1,strlen(pch3)-4);  //pch3 starts with quotation, so we add 1 to start
						       //and there are 3 useless chars in the end ",\0
	    images=g_list_append(images,name);
	}
	//Exactly same way we are looking and saving sound-file-names
	pch = strstr (my_string,"\"sound-file\"");
	if (pch!=NULL){ 
	    char *pch2=NULL;
	    pch2 = strstr (my_string,":");
	    char *pch3=NULL;
	    pch3 = strstr (pch2,"\"");

	    gchar *name;
	    name = g_strndup (pch3+1,strlen(pch3)-4); 
	    samples = g_list_append(samples,name);
	}
      }
  }
}

static void print_objects() {

//g_list_nth

  int i=0;
  for (i=0;i<g_list_length(images);i++) {
  //while (images_tmp!=NULL){
    printf("%d = %s / %s\n",i+1,(char*)g_list_nth(images,i)->data,(char*)g_list_nth(samples,i)->data);
  }
}

static void print_menu() {
printf("\npress number to select object and listen it\n");
printf("press space to see object list again\n");
printf("press enter to put selected on timeline (end of it)\n");
printf("press r to remove sample from timeline\n");
printf("press v to show timeline\n");
printf("press p to listen song\n");
printf("press t to go back theme selection\n");
printf("press h for help\n");
printf("press x to quit\n");
printf("\npress y to add flute track\n");
printf("press u to add drum track\n");
printf("press i to add ud track\n");

printf("press n to go virtual instrument mode\n");
}

/*g_main_loop must be started. So this is own thread.*/
static void *start_theme() {
  const char *theme_folder =chum_get_theme_folder(chum_get_current_theme());
  //empty these
  images = NULL;
  samples = NULL;
  load_theme(theme_folder);

  printf("Starting theme:%s\n",theme_folder);
  print_objects();
  print_menu();

  char *selected_sound=NULL;

  int ch=mygetch();
  while (1){
      //printf("key:%d\n",ch);
      switch (ch)
	{
	case 'x':
	  printf("\n\nQUITTED\n\n");
	  exit(1);
	break;

	case 'h':
	  print_menu();
	break;

	case 't':
	  chum_go_screen(SCREEN_THEME_SELECTION);
	break;

	case 'p':
	  chum_play_sequencer();
	break;

	case ' ':
	  print_objects();
	break;

	case 'v':
	  chum_get_all_slot_info(0); //for track number 0.
	break;

	case 10: //enter
	  if (selected_sound==NULL){
	    printf("No selected sound\n");
	    break;
	  }
	  printf("Where slot you want put it?\n");
	  int ch2=mygetch();
	  int selected2=ch2-48;
	  if (selected2<0 || selected2>9) {
	    printf("not valid slot! canceling adding\n");
	  break;
	  }

	  gchar* buffer0;
	  buffer0 = g_strdup_printf("../share/jammo/themes/%s/%s",chum_get_theme_folder(chum_get_current_theme()), selected_sound);

   	  chum_add_new_sample_to_track_nth_slot(0,buffer0,selected2);
	  break;

	case 'r':
	  printf("Where slot you want remove sample?\n");
	  int ch3=mygetch();
	  int selected3=ch3-48;
	  if (selected3<0 || selected3>9) {
	    printf("not valid slot! canceling removing\n");
	  break;
	  }
	  chum_remove_sample_from_slot(0,selected3); //first is track_number
	  break;
	case 'm':
	  virtual_instrument_mode_joystick();
	  printf("back to normal mode\n");
	  break;

	case 'n':
	  printf("\nVirtual Instrument mode:\n");
	  printf("Use keyboard to play notes [c1d2ef3g4a5b].\nArrow up/down to change octave.\nBackspace to leave virtual instrument mode.\nSpace to show status i.e note-table (usefull if note doesn't stop)\n");
	  
	  virtual_instrument_mode();

	  //there are stuff in keyboard buffer, let empty it (last char is backspace=127)
	  while(mygetch()!=127)
		;
	  break;

	case 'y':
	  chum_add_instrument_track(0);
	break;

	case 'u':
	  chum_add_instrument_track(1);
	break;

	case 'i':
	  chum_add_instrument_track(2);
	break;

	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	  ; //This is important row: see http://gcc.gnu.org/ml/gcc-bugs/2008-08/msg01894.html
	  //-48 because changing char to int, -1 because user might want start counting samples from 1 not from 0.
	  int selected_object= ch-48 -1;

	  GList *selected=g_list_nth(samples,selected_object);
	  if (selected==NULL){
	    printf("not object for this number\n");
	    break;
	  }
	  selected_sound = (char*) selected->data;

	  buffer0 = g_strdup_printf("../share/jammo/themes/%s/%s",chum_get_theme_folder(chum_get_current_theme()), selected_sound);

	  chum_play_one_sample(buffer0);
	  break;

	default:
	  printf("Not an option\n");
	}
	ch=mygetch();
    }

}

void gui_go_screen(int screen_number) {
    if (screen_number==SCREEN_WELCOME){
      show_welcome();
    }
    else if (screen_number==SCREEN_GAME_SELECTION){
    show_game_selection();
    }

    else if (screen_number==SCREEN_THEME_SELECTION){
    show_theme_selection();
    }

    else if (screen_number==SCREEN_COMPOSING){
    //start keyboard reading in own thread
    pthread_t loop_thread;
    pthread_create(&loop_thread, NULL, start_theme, NULL);
    }
}
