/*
 * gems_security.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu, Lappeenranta University of Technology
 *
 * Authors: Jussi Laakkonen <jussi.laakkonen@lut.fi>
 */

#include "gems.h"
#include "gems_security.h"
#include "../cem/cem.h"

gems_message* gems_security_create_envelope(gint16 type,gems_connection* element, gems_message* original)
{
	gems_message* envelope = (gems_message*)g_malloc(sizeof(gems_message));
	guint position = 0;
	
	envelope->length = sizeof(gint16) + sizeof(gint32) + JAMMO_MESSAGE_DIGEST_SIZE + 1 + original->length;
	
	envelope->message = (gchar*)g_malloc(sizeof(gchar*) * envelope->length);
	
	memset(envelope->message,'\0', envelope->length);
	
	// Packet type 16 bits
	*(gint16*)&envelope->message[position] = htons(type);
	position = position + sizeof(gint16);
	
	// Length 32 bits
	*(gint32*)&envelope->message[position] = htonl(envelope->length);
	position = position + sizeof(gint32);
	
	// Hash /*Casting by Aapo, because of compile time warnings!*/
	unsigned char* hash = SHA1((const unsigned char*)(original->message),(guint32)(original->length),NULL);
	g_strlcat(&envelope->message[position],(char*)hash,envelope->length);
	position = position + JAMMO_MESSAGE_DIGEST_SIZE + 1;
	
	// Copy original message
	memcpy(&envelope->message[position],original->message,original->length);
	
	return envelope;
}

gint gems_security_extract_envelope(gems_connection* element)
{
	// [id:2|lenght:4|hash|srvid:2|length:4|cmd:2|...]
	gint32 length = ntohl(gems_connection_get_32(element,sizeof(gint16)));
	
	//gchar* hash =
	gems_connection_get_data(element,sizeof(gint16)+sizeof(gint32),JAMMO_MESSAGE_DIGEST_SIZE);
	
	memmove(&element->nwbuffer[0],
		&element->nwbuffer[sizeof(gint16) + sizeof(gint32) + JAMMO_MESSAGE_DIGEST_SIZE + 1],
		(length - sizeof(gint16) - sizeof(gint32) - JAMMO_MESSAGE_DIGEST_SIZE - 1));
		
	// TODO compare hashes
	
	return SECURITY_OK;
}

gboolean gems_security_init_security(guchar* password_data, guint password_length)
{
	gems_security_context* security = gems_get_security_contexts();
	
	// Not set - allocate struct
	if(security == NULL) security = (gems_security_context*)g_malloc(sizeof(gems_security_context));
	
	security->encryption = (EVP_CIPHER_CTX*)g_malloc(sizeof(EVP_CIPHER_CTX));
	security->decryption = (EVP_CIPHER_CTX*)g_malloc(sizeof(EVP_CIPHER_CTX));
	security->digest = (EVP_MD_CTX*)g_malloc(sizeof(EVP_MD_CTX));
	
	// If profile loaded as encrypted - use salt from that
	// otherwise create new?
	if(gems_security_init_aes(security->encryption, security->decryption, password_data, password_length,NULL)) return FALSE;
	
	// No need to keep the password data - it is set to security contexts
	g_free(password_data);
	
	if(gems_security_init_sha(security->digest)) return FALSE;
	
	return TRUE;
}

gboolean gems_security_change_password(guchar* password_data, guint password_length)
{
	gems_security_clear_security();
	if(gems_security_init_security(password_data, password_length))
	{
		cem_add_to_log("gems_security_change_password() : success", J_LOG_DEBUG);
		return TRUE;
	}
	// TODO copy previous contexts in case of errors!
	else
	{
		cem_add_to_log("gems_security_change_password() : failure - re-initialize security!", J_LOG_DEBUG);
		return FALSE;
	}
}

void gems_security_clear_security()
{
	gchar* logmsg = NULL;
	gems_security_context* security = gems_get_security_contexts();
	
	if(EVP_CIPHER_CTX_cleanup(security->encryption) != 1)
	{
		logmsg = g_strdup_printf("gems_security_clear_security() : Cannot cleanup encryption context");
		cem_add_to_log(logmsg,J_LOG_ERROR);
		g_free(logmsg);
	}
	g_free(security->encryption);
	
	if(EVP_CIPHER_CTX_cleanup(security->decryption) != 1)
	{
		logmsg = g_strdup_printf("gems_security_clear_security() : Cannot cleanup decryption context");
		cem_add_to_log(logmsg,J_LOG_ERROR);
		g_free(logmsg);
	}
	g_free(security->decryption);
	
	if(EVP_MD_CTX_cleanup(security->digest) != 1)
	{
		logmsg = g_strdup_printf("gems_security_clear_security() : Cannot cleanup message digest");
		cem_add_to_log(logmsg,J_LOG_ERROR);
		g_free(logmsg);
	}
	g_free(security->digest);
	
	g_free(security);
	security = NULL;
}

guchar* gems_security_create_password_salt()
{
	guchar* salt = (guchar*)g_malloc(sizeof(guchar*) * AES_SALT_LENGTH);
	
	memset(salt,0,AES_SALT_LENGTH);

	GRand* grand = g_rand_new_with_seed(time(NULL));

	*(guint32*)&salt[0] = g_rand_int(grand);
	*(guint32*)&salt[sizeof(guint32)] = g_rand_int(grand);

	g_free(grand);
	
	return salt;
}

// encryption context, decryption context, key/passwd data, key data length, salt
gboolean gems_security_init_aes(EVP_CIPHER_CTX* enc, EVP_CIPHER_CTX* dec, guchar* keydata, guint keylength, guchar* salt)
{
	guchar key[32], iv[32];
	gchar* logmsg = NULL;
	gboolean returnvalue = TRUE;
	
	memset(&key,0,32);
	memset(&iv,0,32);
	
	if(EVP_BytesToKey(EVP_aes_256_cbc(),EVP_sha256(), NULL, keydata, keylength, AES_ROUNDS, key, iv) != 32)
	{
		logmsg = g_strdup_printf("gems_security_init_aes() : Cannot set bytes to key.");
		cem_add_to_log(logmsg,J_LOG_ERROR);
		g_free(logmsg);
		returnvalue =  FALSE;
	}
		
	// Encryption context
	EVP_CIPHER_CTX_init(enc);
	if(EVP_EncryptInit_ex(enc, EVP_aes_256_cbc(), NULL, key, iv) == 0)
	{
		logmsg = g_strdup_printf("gems_security_init_aes() : Cannot initialize aes_256_cbc encryption context.");
		cem_add_to_log(logmsg,J_LOG_ERROR);
		g_free(logmsg);
		returnvalue =  FALSE;
	}
	
	// Decryption contexxt
	EVP_CIPHER_CTX_init(dec);
	if(EVP_DecryptInit_ex(dec, EVP_aes_256_cbc(), NULL, key, iv) == 0)
	{
		logmsg = g_strdup_printf("gems_security_init_aes() : Cannot initialize aes_256_cbc decryption context.");
		cem_add_to_log(logmsg,J_LOG_ERROR);
		g_free(logmsg);
		returnvalue =  FALSE;
	}
	
	return returnvalue;
}

gboolean gems_security_init_sha(EVP_MD_CTX* md)
{
	EVP_MD_CTX_init(md);
	if(EVP_DigestInit_ex(md, EVP_sha256(), NULL) == 0)
	{
		gchar* logmsg = g_strdup_printf("gems_security_init_sha() : Cannot init sha digest.");
		cem_add_to_log(logmsg,J_LOG_ERROR);
		g_free(logmsg);
		return FALSE;
	}
	return TRUE;
}

guchar* gems_security_calculate_hash(guchar* data, guint* data_length)
{
	EVP_MD_CTX* md = NULL;
	guint hash_length = 0;
	guchar* hash = NULL;
	gchar* logmsg = NULL;
	
	if(gems_get_security_contexts() == NULL)
	{
		logmsg = g_strdup_printf("gems_security_calculate_hash() : Cannot calculate hash, security contexts not initialized.");
		cem_add_to_log(logmsg,J_LOG_ERROR);
		g_free(logmsg);
		return NULL;
	}
	
	if((md = gems_get_security_contexts()->digest) == NULL)
	{
		logmsg = g_strdup_printf("gems_security_calculate_hash() : Cannot calculate hash, hash context not initialized.");
		cem_add_to_log(logmsg,J_LOG_ERROR);
		g_free(logmsg);
		return NULL;
	}
	
	hash = (guchar*)g_malloc(sizeof(guchar*)*EVP_MAX_MD_SIZE);
	memset(hash,0,EVP_MAX_MD_SIZE);
	
	// Failure, cannot initialize
	if(EVP_DigestInit_ex(md, EVP_sha256(), NULL) == 0) 
	{
		logmsg = g_strdup_printf("gems_security_calculate_hash() : Cannot calculate hash, cannot initialize digest.");
		cem_add_to_log(logmsg,J_LOG_ERROR);
		g_free(logmsg);
		return NULL;
	}
	
	EVP_DigestUpdate(md,data,strlen((gchar*)data));
	EVP_DigestFinal_ex(md,hash,&hash_length);
	*data_length = hash_length;
	
	return hash;
}

gboolean gems_security_verify_hash(guchar* hash1, guchar* hash2)
{
	return (g_strcmp0((gchar*)hash1,(gchar*)hash2) == 0) ? TRUE : FALSE;
}

// encryption context, data to encrypt, data length
guchar* gems_security_encrypt_data(guchar* profile_data, guint* data_length)
{
	EVP_CIPHER_CTX* enc = NULL;
	gint enc_length = 0;
	gint final_length = 0;
	guchar* encrypted_profile = NULL;
	gchar* logmsg = NULL;
	
	if(gems_get_security_contexts() == NULL)
	{
		logmsg = g_strdup_printf("gems_security_encrypt_data() : Cannot encrypt, security contexts not initialized.");
		cem_add_to_log(logmsg,J_LOG_ERROR);
		g_free(logmsg);
		return NULL;
	}
	if((enc = gems_get_security_contexts()->encryption) == NULL)
	{
		logmsg = g_strdup_printf("gems_security_encrypt_data() : Cannot encrypt, decryption contexts not initialized.");
		cem_add_to_log(logmsg,J_LOG_ERROR);
		g_free(logmsg);
		return NULL;
	}
	
	enc_length = *data_length + AES_BLOCK_SIZE;
	encrypted_profile = (guchar*)g_malloc(sizeof(guchar*)*enc_length);
	memset(encrypted_profile,0,enc_length);
	
	//EVP_EncryptInit_ex(enc,NULL,NULL,NULL,NULL);
	EVP_EncryptUpdate(enc, encrypted_profile, &enc_length, profile_data, *data_length);
	EVP_EncryptFinal_ex(enc, encrypted_profile + enc_length, &final_length);
	
	*data_length = enc_length + final_length;
	
	return encrypted_profile;
}

// decryption context, data to decrypt, data length
guchar* gems_security_decrypt_data(guchar* profile_data, guint* data_length)
{
	EVP_CIPHER_CTX* dec = NULL;
	gint dec_length = 0;
	gint final_length = 0;
	guchar* decrypted_profile = NULL;
	gchar* logmsg = NULL;
	
	if(gems_get_security_contexts() == NULL)
	{
		logmsg = g_strdup_printf("gems_security_decrypt_data() : Cannot decrypt, security contexts not initialized");
		cem_add_to_log(logmsg,J_LOG_ERROR);
		g_free(logmsg);
		return NULL;
	}
	if((dec = gems_get_security_contexts()->decryption) == NULL)
	{
		logmsg = g_strdup_printf("gems_security_decrypt_data() : Cannot decrypt, decryption context not initialized");
		cem_add_to_log(logmsg,J_LOG_ERROR);
		g_free(logmsg);
		return NULL;
	}
	
	dec_length = *data_length + AES_BLOCK_SIZE;
	decrypted_profile = (guchar*)g_malloc(sizeof(guchar*)*dec_length);
	memset(decrypted_profile,0,dec_length);
	
	//EVP_DecryptInit_ex(dec,NULL,NULL,NULL,NULL);
	EVP_DecryptUpdate(dec, decrypted_profile, &dec_length, profile_data, *data_length);
	EVP_DecryptFinal_ex(dec, decrypted_profile + dec_length, &final_length);
	
	*data_length = dec_length + final_length;
	
	return decrypted_profile;
}

