/*
 * ProfileManager.cc
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2011 Lappeenranta University of Technology
 *
 * Authors: Janne Parkkila
 * 					Jussi Laakkonen <jussi.laakkonen@lut.fi>
 */
 
#include <iostream>
#include <unistd.h>
#include <sys/types.h>
#include <signal.h>
#include <sstream>
#include <cstdlib>
#include "ProfileManager.h"
#include "gems_definitions.h"

extern "C" {
	#include "../cem/cem.h"
}

using namespace std;

ProfileManager* ProfileManager::iInstance = NULL;

ProfileManager* ProfileManager::GetInstance() 
{
	// No instance->create
	if (iInstance == NULL) iInstance = new ProfileManager;
	
	return iInstance;
}

// Constructor
ProfileManager::ProfileManager()
{
	iStorageAgent = new StorageAgent();
	// process agent not yet used.
	//iProcessAgent = NULL;
}

// Destructor
ProfileManager::~ProfileManager()
{
	if(iStorageAgent != NULL)
	{
		delete iStorageAgent;
		iStorageAgent = NULL;
	}
	/*if(iProcessAgent != NULL)
	{
		delete iProcessAgent;
		iProcessAgent = NULL;
	}*/
}

// Login, calls StorageAgent to retrieve the user
// profile data
gboolean ProfileManager::Login(guint32 userID)
{
	// Store the id
	iUser = userID;
	
	string msg = "User Profile ";
	// C++ type int to string conversion
	stringstream user;
	user << userID;
	msg.append(user.str());
	
	// If profile can be retrieved correctly,
	// create a server for displaying the profile
	// to other users
	if (iStorageAgent->Retrieve(iUser) == true)
	{
		msg.append(" succesfully retrieved");
		log_wrap((char*)msg.c_str(),J_LOG_INFO);
		return true;
	}
	else
	{
		//cout << "Error while retrieving user profile data";
		msg.append(" cannot be found.");
		log_wrap((char*)msg.c_str(),J_LOG_INFO);
		return false;
	}
	
}

// Logout
gboolean ProfileManager::Logout()
{

	// Store the user data
	
	if (iStorageAgent->Store(iUser) == true)
	{
	
		// Clear the user from the memory
		iUser = 0;
		iStorageAgent->clearDetails();
	
		// If everything worked correctly, return true
		// Again... not implemented correctly
		return true;
	}
	
}

string ProfileManager::ViewDetails()
{
	// View my details here
	return iStorageAgent->getDetails();
}

string ProfileManager::ViewDetail(string& detail)
{
	return iStorageAgent->getDetail(detail);
}

// Edit user details. Takes the name of the detail and the new Value
// as a string
gboolean ProfileManager::EditDetails(string& detail, string& newValue)
{
	// If the detail changing operation is failure, function return false
	return iStorageAgent->setDetails(detail, newValue);
}

// This function is used to Force another user in to the
// profile manager. Requires admin level in the profile
gboolean ProfileManager::ForceUser(guint32 userID)
{
	string auth;
	// Get detail authLevel from the profile
	// if it is admin, the user is allowed to force
	// another user to the machine
	string param = string("authLevel");
	auth = iStorageAgent->getDetail(param);
	if (auth == "admin")
	{
		// Logout the admin and login wit the
		// wanted userID
		if (Logout() == true)
		{
			Login(userID);
			return TRUE;
		}
		else return FALSE;
	}
	else return FALSE;
}

guint32 ProfileManager::GetId()
{
	return iUser;
}

StorageAgent* ProfileManager::GetStorageAgent()
{
	return iStorageAgent;
}

void log_wrap(char* msg, int info)
{
	cem_add_to_log(msg,info);
	//printf("message: %s (%d)\n",msg,info);
}

/*//////////////////////////////////////////////////////
||||||||||| DO NOT USE THESE DIRECTLY !!!! |||||||||||||
||||||| USE FUNCTIONS IN gems_profile_manager.c  |||||||
////////////////////////////////////////////////////////*/

// New manager object
ProfileManager* profilemanager_new_manager()
{
	return ProfileManager::GetInstance();
}

// Delete manager object
void profilemanager_delete_manager(ProfileManager* _aManager)
{
	delete _aManager;
}

// Login to manager (no auth checking yet)
gboolean profilemanager_login(ProfileManager* _aManager, guint32 _userId)
{
	return _aManager->Login(_userId);
}

// Logout from manager
gboolean profilemanager_logout(ProfileManager* _aManager)
{
	return _aManager->Logout();
}

// View one information
const gchar* profilemanager_view_info(ProfileManager* _aManager, gchar* _parameter)
{
	string param = string(_parameter);
	return (_aManager->ViewDetail(param)).c_str();
}

// Edit own info
gboolean profilemanager_edit_info(ProfileManager* _aManager, gchar* _parameter, gchar* _value)
{
	string param = string(_parameter);
	string value = string(_value);
	return _aManager->EditDetails(param,value);
}

guint32 profilemanager_get_userid(ProfileManager* _aManager)
{
	return _aManager->GetId();
}

// Get own username
const gchar* profilemanager_get_username(ProfileManager* _aManager)
{
	string param = string(PROF_PARAM_USERNAME);
	return (_aManager->ViewDetail(param)).c_str();
}

// Get first name
const gchar* profilemanager_get_firstname(ProfileManager* _aManager)
{
	string param = string(PROF_PARAM_FIRSTNAME);
	return (_aManager->ViewDetail(param)).c_str();
}

// Get last name
const gchar* profilemanager_get_lastname(ProfileManager* _aManager)
{
	string param = string(PROF_PARAM_LASTNAME);
	return (_aManager->ViewDetail(param)).c_str();
}

// Get age
guint16 profilemanager_get_age(ProfileManager* _aManager)
{
	string param = string(PROF_PARAM_AGE);
	return atoi((_aManager->ViewDetail(param)).c_str());
}

guint32 profilemanager_get_points(ProfileManager* _aManager)
{
	string param = string(PROF_PARAM_POINTS);
	return (guint32)atoi(_aManager->ViewDetail(param).c_str());
}

gboolean profilemanager_set_username(ProfileManager* _aManager, gchar* _value)
{
	string param = string(PROF_PARAM_USERNAME);
	string value = string(_value);
	return _aManager->EditDetails(param,value);
}

gboolean profilemanager_set_firstname(ProfileManager* _aManager, gchar* _value)
{
	string param = string(PROF_PARAM_FIRSTNAME);
	string value = string(_value);
	return _aManager->EditDetails(param,value);
}

gboolean profilemanager_set_lastname(ProfileManager* _aManager, gchar* _value)
{
	string param = string(PROF_PARAM_LASTNAME);
	string value = string(_value);
	return _aManager->EditDetails(param,value);
}

gboolean profilemanager_set_age(ProfileManager* _aManager, guint16 _value)
{
	stringstream strvalue;
	string param = string(PROF_PARAM_AGE);
	
	if(_value > 120) return FALSE;
	
	strvalue << _value;
	string value = strvalue.str();
	
	return _aManager->EditDetails(param,value);
}

gboolean profilemanager_add_points(ProfileManager* _aManager, guint32 _value)
{
	string param = string(PROF_PARAM_POINTS);
	guint32 points = atoi((_aManager->ViewDetail(param)).c_str());
	
	points += _value;
	
	stringstream strvalue;
	strvalue << points;
	
	string value = strvalue.str();
	
	return _aManager->EditDetails(param,value);
}

gboolean profilemanager_remove_points(ProfileManager* _aManager, guint32 _value)
{
	string param = string(PROF_PARAM_POINTS);
	guint32 points = atoi((_aManager->ViewDetail(param)).c_str());
	
	if((points - _value) < 0) points = 0;
	else points = points - _value;
	
	stringstream strvalue;
	strvalue << points;
	
	string value = strvalue.str();
	
	return _aManager->EditDetails(param,value);	
}

void profilemanager_reset_points(ProfileManager* _aManager)
{
	string param = string(PROF_PARAM_POINTS);
	string value = "0";
	_aManager->EditDetails(param,value);
}

guchar* profilemanager_storage_get_password_salt(ProfileManager* _aManager)
{
	if(_aManager == NULL) return NULL;
	return _aManager->GetStorageAgent()->getSalt();
}

guchar* profilemanager_storage_get_password_hash(ProfileManager* _aManager)
{
	if(_aManager == NULL) return NULL;
	return _aManager->GetStorageAgent()->getPasswordHash();
}

guchar* profilemanager_storage_get_profile_hash(ProfileManager* _aManager)
{
	if(_aManager == NULL) return NULL;
	return _aManager->GetStorageAgent()->getProfileHash();
}

guchar* profilemanager_storage_get_encrypted_profile(ProfileManager* _aManager)
{
	if(_aManager == NULL) return NULL;
	return _aManager->GetStorageAgent()->getEncryptedProfile();
}

guint profilemanager_storage_get_encrypted_profile_size(ProfileManager* _aManager)
{
	if(_aManager == NULL) return NULL;
	return _aManager->GetStorageAgent()->getEncryptedProfileSize();
}

void profilemanager_storage_set_password_salt(ProfileManager* _aManager, guchar* _salt)
{
	_aManager->GetStorageAgent()->setSalt(_salt);
}

void profilemanager_storage_set_password_hash(ProfileManager* _aManager, guchar* _hash)
{
	_aManager->GetStorageAgent()->setPasswordHash(_hash);
}

void profilemanager_storage_set_encrypted_profile(ProfileManager* _aManager, guchar* _profile_hash, guchar* _encrypted_profile, guint _e_profile_size)
{
	_aManager->GetStorageAgent()->setEncryptedProfile(_profile_hash, _encrypted_profile, _e_profile_size);
}

