/** file_helper.c is part of JamMo.
License: GPLv2, read more from COPYING

This file contain helper functions, which are meant to be used via chum.
*/

#include "chum.h"

#include <glib-object.h>
#include <string.h>

#include "../../cem/cem.h"

/**
compares 2 filenames.
This is function is needed because we want return directory listing sorted.
If another (or both) is NULL, will return 0.
**/
static int string_compare( gconstpointer a, gconstpointer b ){
   if (a==NULL)
      return 0;
   if (b==NULL)
      return 0;

   guint length = -1;
   gint rv = -1;

   length =  strlen( (gchar *) a );
   rv = memcmp( a, b, length);

   return rv;
}


static GList *scan_and_sort_directory( gchar *directory_name ) {
   GError *error = NULL;
   guint flags = 0;

   GDir *directory = g_dir_open( directory_name, flags, &error );

   if( directory == NULL ) 
      {
      gchar* message = g_strdup_printf("Couldn't not open the directory %s", directory_name);
      cem_add_to_log(message,J_LOG_ERROR);
      g_free(message);
      cem_add_to_log(error->message,J_LOG_ERROR);
      g_error_free( error );
      return NULL;
      }


   GList *list = NULL;
   while (TRUE) 
      {
      const gchar *temp = NULL;
      /* the return value of g_dir_read_name is own by glib
         Do not modify or free */
      temp = g_dir_read_name( directory );
      if( temp == NULL ) 
        {
        break;
        }
      gchar *copy = g_strndup( temp, strlen(temp) );
      list = g_list_append( list, (gchar *) copy );
      }

   list = g_list_sort( list, string_compare );
   g_dir_close( directory );
   return list;
}


/*
One song is in one folder, which contain songfile and picture (maybe lyrics and so on).
'name of song' is name of this folder.

This function returns list of all those folders (they all starts with SONGS_FOLDER macro)
*/
GList *file_helper_get_all_songs() {
  GList *song_list = NULL;
  gint index = 0;
  gchar *song_name = NULL;

  GList *song_folders = scan_and_sort_directory( SONGS_DIR );
  if( song_folders == NULL )
      {
      cem_add_to_log("List of songs is NULL",J_LOG_WARN);
      return NULL;
      }

  for( index=0 ; index < g_list_length(song_folders) ; index++ ) 
      {
      gchar *element = (gchar *) g_list_nth_data( song_folders, index );
      song_name = g_strdup_printf("%s/%s", SONGS_DIR,element);
      song_list = g_list_append( song_list,song_name);
      //printf("added to list: '%s'\n",song_name);
      }
  return song_list;
}

GList *file_helper_get_all_files(char* directory) {
  GList *song_list = NULL;
  gint index = 0;
  gchar *song_name = NULL;

  GList *song_folders = scan_and_sort_directory( directory );
  if( song_folders == NULL )
      {
      gchar* message = g_strdup_printf("Directory '%s' is empty, will return NULL",directory);
      cem_add_to_log(message,J_LOG_WARN);
      g_free(message);
      return NULL;
      }

  for( index=0 ; index < g_list_length(song_folders) ; index++ ) 
      {
      gchar *element = (gchar *) g_list_nth_data( song_folders, index );
      song_name = g_strdup_printf("%s/%s", directory,element);
      song_list = g_list_append( song_list,song_name);
      //printf("added to list: '%s'\n",song_name);
      }
  return song_list;
}

GList *file_helper_get_all_files_without_path(char* directory) {
  GList *song_list = NULL;
  gint index = 0;
  gchar *song_name = NULL;

  GList *song_folders = scan_and_sort_directory( directory );
  if( song_folders == NULL )
      {
      gchar* message = g_strdup_printf("Directory '%s' is empty, will return NULL",directory);
      cem_add_to_log(message,J_LOG_WARN);
      g_free(message);
      return NULL;
      }

  for( index=0 ; index < g_list_length(song_folders) ; index++ ) 
      {
      gchar *element = (gchar *) g_list_nth_data( song_folders, index );
      song_name = g_strdup_printf("%s",element);
      song_list = g_list_append( song_list,song_name);
      //printf("added to list: '%s'\n",song_name);
      }
  return song_list;
}
