/**sequencer.c is part of JamMo.
License: GPLv2, read more from COPYING

This file is center of 7-12 years full sequencer view.
View contains three part:
 *wheel-game (loop-view)
 *track-view
 *general-view (settings)

*/

#include <glib-object.h>
//#include <clutter/clutter.h>
#include <math.h>
#include <string.h>

#include <tangle.h>

#include "../../meam/jammo-meam.h"
#include "../../meam/jammo-slider-event.h"
#include "../../meam/jammo-slider-track.h"
#include "../../meam/jammo-backing-track.h"
#include "../../meam/jammo-metronome-track.h"

#include "../jammo.h"
#include "../jammo-mentor.h"
#include "../jammo-editing-track-view.h"
#include "../jammo-miditrack-view.h"
#include "../../cem/cem.h"

#include "sequencer.h"
#include "sequencer_loop.h"
#include "sequencer_track_view.h"
#include "startmenu.h"

#include "../jammo-game-task.h"
#include "../jammo-game-level.h"
#include "../jammo-game-level-view.h"

#include "../../configure.h"
/*
Sequencer and sequencer-track-view-area are defined in json.

When sequencer starts: it calls load_state_from_file(load_only_backingtrack=TRUE)
 It starts loading backing-track.
 When backing_track is loaded (on_duration_notify)
  Tell sequencer_track_view_area that now duration is known (used e.g. with cursor)
  Then load_state_from_file(load_only_backingtrack=FALSE) loads another tracks.
*/

//This is The sequencer of the game. All 'play'-buttons should play this.
//All new tracks are added to this
//If metronome is enabled it is added to this
JammoSequencer* static_sequencer;
JammoMetronomeTrack* static_metronome_track;




gboolean sequencer_change_to_loop_view(TangleActor *actor, gpointer data) {
	cem_add_to_log("changing view to loop",J_LOG_INFO);
	ClutterActor* scrolling_view = jammo_get_actor_by_id("fullsequencer-view-container");
	if (scrolling_view==NULL){
		cem_add_to_log("no actor 'fullsequencer-view-container'",J_LOG_ERROR);
		return TRUE;
	}
	ClutterAction* action = tangle_actor_get_action_by_type(scrolling_view,TANGLE_TYPE_SCROLL_ACTION);
	tangle_object_animate(G_OBJECT(action), CLUTTER_EASE_IN_CIRC, 400, "offset-y", 0.0, NULL);
	return TRUE;
}

gboolean sequencer_change_to_sequencer_view(TangleActor *actor, gpointer data) {
	cem_add_to_log("changing view to sequencer",J_LOG_INFO);
	ClutterActor* scrolling_view = jammo_get_actor_by_id("fullsequencer-view-container");
	if (scrolling_view==NULL){
		cem_add_to_log("no actor 'fullsequencer-view-container'",J_LOG_ERROR);
		return TRUE;
	}
	ClutterAction* action = tangle_actor_get_action_by_type(scrolling_view,TANGLE_TYPE_SCROLL_ACTION);
	tangle_object_animate(G_OBJECT(action), CLUTTER_EASE_IN_CIRC, 400, "offset-y", 480.0, NULL);
	return TRUE;
}

gboolean sequencer_change_to_bottom_view (TangleActor *actor, gpointer data) {
	cem_add_to_log("changing view to general menu",J_LOG_INFO);
	ClutterActor* scrolling_view = jammo_get_actor_by_id("fullsequencer-view-container");
	if (scrolling_view==NULL){
		cem_add_to_log("no actor 'fullsequencer-view-container'",J_LOG_ERROR);
		return TRUE;
	}
	ClutterAction* action = tangle_actor_get_action_by_type(scrolling_view,TANGLE_TYPE_SCROLL_ACTION);
	tangle_object_animate(G_OBJECT(action), CLUTTER_EASE_IN_CIRC, 400, "offset-y", 960.0, NULL);
	return TRUE;
}

//show-completed callback (=animation is ready)
static void levelview_loaded(ClutterActor *levelview, gpointer none) {
	g_signal_handlers_disconnect_by_func(levelview, levelview_loaded, NULL);
	const gchar* levelname = clutter_actor_get_name(levelview);

	//Make sample-buttons on wheel-game
	//This can take time (thus we have hourglass on screen)
	if (strcmp(levelname,"level1")==0)  //optimization: Level1 doesn't need them all.
		sequencer_loop_tune_wheels(1);    //load only sample-buttons for level1
	else
		sequencer_loop_tune_wheels(0);    //load all sample-buttons


	//Hide hourglass
	ClutterActor* wait = jammo_get_actor_by_id("wait");
	if (wait)
		clutter_actor_hide(wait);

	//Start task1 on this level
	JammoGameLevel* game_level = JAMMO_GAME_LEVEL(jammo_get_object_by_id("game-level"));
	if (game_level) {
		gchar* task_name = g_strdup_printf("%s-task1.json",levelname);
		jammo_game_level_start_task(game_level,task_name);
		g_free(task_name);
	}

}



static ClutterActor* current_levelview =NULL;
void start_fullsequencer_gui(ClutterActor* levelview)
{
	tangle_actor_hide_animated(TANGLE_ACTOR(jammo_mentor_get_default()));

	current_levelview=levelview;
	cem_add_to_log("Sequencer GUI starts",J_LOG_INFO);

	g_signal_connect(current_levelview, "show-completed", G_CALLBACK(levelview_loaded), NULL); //Continue after animated show
	clutter_actor_show(current_levelview);

	ClutterAction* action = tangle_actor_get_action_by_type(jammo_get_actor_by_id("fullsequencer-view-container"), TANGLE_TYPE_SCROLL_ACTION);
	g_signal_connect_swapped(action, "clamp-offset-y", G_CALLBACK(tangle_widget_clamp_child_boundaries), jammo_get_actor_by_id("fullsequencer-view-container"));

	static_sequencer = JAMMO_SEQUENCER(jammo_get_object_by_id("fullsequencer-the-sequencer"));
	static_metronome_track = NULL;

	sequencer_track_view_tune_width();

	sequencer_change_to_sequencer_view(NULL,NULL); //Start to middle view. //TODO:do inside JSON

	ClutterActor* wait = jammo_get_actor_by_id("wait");
	if (wait)
		clutter_actor_show(wait);

}

void return_to_fullsequencer_gui(){
	clutter_actor_show(CLUTTER_ACTOR(jammo_mentor_get_default()));
	clutter_actor_show(current_levelview);
}

/*
 * CallBack for JSON
 */

void fullsequencer_general_tempo_clicked(TangleButton *tanglebutton, gpointer user_data) {
	const gchar* name=clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton));
	if (name==NULL)
		return;

	int new_tempo = 110;
	if (strncmp(name,"fast",4)==0)
			new_tempo=130;
	else if (strncmp(name,"normal",6)==0)
			new_tempo=110;
	else if (strncmp(name,"slow",4)==0)
			new_tempo=90;

	gchar* message = g_strdup_printf("tempo '%s'='%d' clicked",name,new_tempo);
	cem_add_to_log(message,J_LOG_USER_ACTION);
	g_free(message);

	jammo_sequencer_set_tempo(static_sequencer, new_tempo);
}


void fullsequencer_general_pitch_clicked(TangleButton *tanglebutton, gpointer user_data) {
	const gchar* name=clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton));
	if (name==NULL)
		return;

	gchar* new_pitch = "G";
	if (strncmp(name,"high",4)==0)
			new_pitch = "G";
	else if (strncmp(name,"normal",6)==0)
			new_pitch = "D";
	else if (strncmp(name,"low",3)==0)
			new_pitch = "A";

	gchar* message = g_strdup_printf("pitch '%s'='%s' clicked",name,new_pitch);
	cem_add_to_log(message,J_LOG_USER_ACTION);
	g_free(message);

	jammo_sequencer_set_pitch(static_sequencer, new_pitch);
}


static void on_rendering_stopped(JammoSequencer* sequencer, gpointer none) {
	cem_add_to_log("Rendering ready",J_LOG_INFO);
	ClutterActor* wait = jammo_get_actor_by_id("wait");
	if (wait)
		clutter_actor_hide(wait);

	g_signal_handlers_disconnect_by_func(sequencer, on_rendering_stopped, NULL);
	if (!(jammo_sequencer_set_output_filename(sequencer, NULL))) {
		cem_add_to_log("Problem going back to playback-mode after render",J_LOG_ERROR);
	}

}

void sequencer_render_button_pressed (TangleButton* tanglebutton, gpointer none){
	cem_add_to_log("Render-button pressed",J_LOG_USER_ACTION);

	char timestamp [80];
	cem_get_time(timestamp);

	gchar* jammo_directory=configure_get_jammo_directory();
	gchar* outputFilename = g_strdup_printf("%s/%s.ogg",jammo_directory,timestamp);
	g_free(jammo_directory);

	gchar* message = g_strdup_printf("Rendering file named '%s'",outputFilename);
	cem_add_to_log(message,J_LOG_INFO);
	g_free(message);

	if (jammo_sequencer_set_output_filename(static_sequencer, outputFilename)) {
		ClutterActor* wait = jammo_get_actor_by_id("wait");
		if (wait)
			clutter_actor_show(wait);

		g_signal_connect(static_sequencer, "stopped", G_CALLBACK(on_rendering_stopped), NULL);
		jammo_sequencer_play(static_sequencer);
	}
	else
		cem_add_to_log("Can't render audiofile",J_LOG_ERROR);

	g_free(outputFilename);
}

void fullsequencer_metronome_clicked (TangleButton* tanglebutton, gpointer none){
	cem_add_to_log("Metronome pressed",J_LOG_USER_ACTION);

	if (static_metronome_track==NULL) {
		static_metronome_track = jammo_metronome_track_new();
		jammo_sequencer_add_track(static_sequencer, JAMMO_TRACK(static_metronome_track));
		jammo_metronome_track_set_time_signature_beats(static_metronome_track,4);
		jammo_metronome_track_set_time_signature_note_value(static_metronome_track,4);
		jammo_metronome_track_set_accent(static_metronome_track,FALSE);
		jammo_playing_track_set_muted(JAMMO_PLAYING_TRACK(static_metronome_track), FALSE);
	}
	else
		jammo_playing_track_set_muted(JAMMO_PLAYING_TRACK(static_metronome_track), !tangle_button_get_selected(tanglebutton));
}
