/*
 * radioelector_screen.c
 *
 *  Created on: 2009-12-11
 *      Author: marcin
 */

#include <hildon/hildon.h>
#include "radioselector_screen.h"
#include "data_structs.h"
#include "jamendo_api.h"
#include "http_utils.h"
#include "resource_utils.h"
#include "playlist_screen.h"

/****************************************************************************/

/**
 * list view column indexes
 */
enum {
	COLUMN_PIXBUF, COLUMN_RADIO, COLUMN_RADIO_PTR, NUM_COLUMNS
};

/****************************************************************************
 *
 */

static void show_radio_screen(GtkTreeView* treeview, GtkTreePath* path, gpointer data) {
	GtkWidget *win;
	GtkTreeModel *model;
	GtkTreeIter iter;
	Radio *radio = NULL;

	model = gtk_tree_view_get_model(treeview);
	if (gtk_tree_model_get_iter(model, &iter, path)) {
		gtk_tree_model_get(model,&iter,COLUMN_RADIO_PTR,&radio,-1);
	}

	if (radio) {
		GList* track_list = jamendo_get_radio_playlist(radio);
		win = playlist_screen_new(track_list,radio->name);
		gtk_widget_show_all(win);
	}
}


/****************************************************************************
 * radio listview functions
 *
 * creates tree view with radio graphics and name
 ****************************************************************************/

typedef struct {
	GtkTreeModel* model;
	GtkTreeIter iter;
	gboolean stop;
	GThread* thread;
} LoadImagesData;

static gpointer load_images(LoadImagesData *data) {
	Radio *radio = NULL;
	GdkPixbuf* radio_image = NULL;
	gboolean next = FALSE;

	g_object_ref(data->model);
	gdk_threads_enter();
	gtk_tree_model_get_iter_first(data->model, &data->iter);
	gdk_threads_leave();

	do {
		gdk_threads_enter();
		gtk_tree_model_get(data->model, &data->iter, COLUMN_RADIO_PTR, &radio, -1);
		gdk_threads_leave();

		radio_image = http_get_image(radio->image, -1, HILDON_ICON_PIXEL_SIZE_THUMB);

		gdk_threads_enter();
		gtk_list_store_set(GTK_LIST_STORE(data->model), &data->iter, COLUMN_PIXBUF, radio_image, -1);
		next = gtk_tree_model_iter_next(data->model, &data->iter);
		gdk_threads_leave();

		g_object_unref(radio_image);
		g_boxed_free(G_TYPE_RADIO, radio);
	} while (next && !data->stop);

	g_object_unref(data->model);

	return NULL;
}

void radio_listview_destroyed(GtkWidget* listview, LoadImagesData* data) {
	g_debug("radio_listview_destroyed join %p data->stop %d",data->thread,data->stop);
	data->stop = TRUE;
	gdk_threads_leave();
	g_thread_join(data->thread);
	gdk_threads_enter();
	g_debug("radio_listview_destroyed free");
	g_free(data);
}

static GdkPixbuf* no_cd_image = 0;

void radio_listview_update_model(GtkWidget* listview, GList* radio_list) {
	GtkListStore *store;
	GtkTreeIter iter;
	GList* radio_iter;

	store = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(listview)));

	gtk_list_store_clear(store);

	if (!radio_list) {
		return;
	}

	if (!no_cd_image) {
		no_cd_image = resource_get_pixbuf_at_scale("no_cd.png",-1,HILDON_ICON_PIXEL_SIZE_THUMB);
	}

	for (radio_iter = g_list_first(radio_list); radio_iter; radio_iter = radio_iter->next) {
		Radio* radio = (Radio*) radio_iter->data;
		gchar* radio_title = g_markup_printf_escaped("<span size='small'>%s</span>", radio->name);
		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter,
				COLUMN_PIXBUF, no_cd_image,
				COLUMN_RADIO, radio_title,
				COLUMN_RADIO_PTR, radio,
				-1);
		g_free(radio_title);
	}

	/** images downloaded in background */
	LoadImagesData* data = g_new0(LoadImagesData,1);
	data->model = GTK_TREE_MODEL(store);
	data->thread = g_thread_create((GThreadFunc)load_images, data, TRUE, NULL);

	/* when listview is destroyed, destroy also radio_list */
	g_signal_connect(G_OBJECT(listview), "destroy", G_CALLBACK(radio_listview_destroyed), data);
}

static void radio_listview_add_columns(GtkTreeView *treeview) {
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	renderer = gtk_cell_renderer_pixbuf_new();
	g_object_set(G_OBJECT (renderer), "follow-state", FALSE, NULL);

	column = gtk_tree_view_column_new_with_attributes("Image", renderer, "pixbuf", COLUMN_PIXBUF, NULL);
	gtk_tree_view_append_column(treeview, column);

	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new_with_attributes("Radio", renderer, "markup", COLUMN_RADIO, NULL);
	gtk_tree_view_append_column(treeview, column);

}

GtkWidget* radio_listview_create(GList* radio_list) {
	GtkWidget *listview;
	GtkTreeModel *model;

	model = GTK_TREE_MODEL( gtk_list_store_new(NUM_COLUMNS, G_TYPE_OBJECT, G_TYPE_STRING, G_TYPE_RADIO) );
	listview = gtk_tree_view_new_with_model(model);
	radio_listview_update_model(listview,radio_list);
	g_object_unref(model);

	gtk_tree_selection_set_mode(gtk_tree_view_get_selection(GTK_TREE_VIEW(listview)), GTK_SELECTION_NONE);
	radio_listview_add_columns(GTK_TREE_VIEW(listview));

	g_signal_connect(listview, "hildon-row-tapped", G_CALLBACK(show_radio_screen),NULL);

	return listview;
}

typedef struct {
	GtkWidget *win;
	GtkWidget *listview;
} PopulateRadioListParam;

static guint populate_radio_list(PopulateRadioListParam *param) {
	GList* radio_list;

	radio_list = jamendo_get_radios();
	radio_listview_update_model(param->listview, radio_list);
	radio_list_free(radio_list);

	hildon_gtk_window_set_progress_indicator(GTK_WINDOW(param->win),0);
	gtk_widget_show_all(param->win);

	g_free(param);
	return FALSE;
}

/****************************************************************************/

GtkWidget* radioselector_screen_new() {
	GtkWidget* win;
	GtkWidget *treeview;
	GtkWidget* radio_box;

	/* Create the main window */
	win = hildon_stackable_window_new();
	gtk_window_set_title(GTK_WINDOW(win), "Radios");

	//player_set_radio_list(radio_list);
	treeview = radio_listview_create(NULL);

	radio_box = hildon_pannable_area_new();
	hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(radio_box), treeview);

	GtkWidget* main_box = gtk_hbox_new(TRUE,0);
	gtk_box_pack_start(GTK_BOX(main_box), radio_box, TRUE,TRUE,0);
	gtk_container_add(GTK_CONTAINER(win), main_box);

	hildon_gtk_window_set_progress_indicator(GTK_WINDOW(win),1);
	hildon_banner_show_information(win,NULL,"Downloading");

	PopulateRadioListParam* param = g_new0(PopulateRadioListParam,1);
	param->win = win;
	param->listview = treeview;
	gdk_threads_add_idle((GSourceFunc)populate_radio_list, param);

	//hildon_program_add_window(hildon_program_get_instance(),HILDON_WINDOW(win));
	//gtk_widget_hide(player_toolbar_create());

	return win;
}
