#include "remotedbmgr.h"

#include <QFile> 
#include <QDomElement> 
#include <QDomNodeList> 
#include <QSettings> 
#include <QNetworkReply>
#include <QUrl>
#include <QNetworkRequest>
#include <QBuffer>
#include <QByteArray>

RemoteDBMgr::RemoteDBMgr()
    : processing(false)
{
    connect(&netAM, SIGNAL(finished(QNetworkReply *)), 
            this, SLOT(dbDownloadFinished(QNetworkReply *)));
}

RemoteDBMgr::~RemoteDBMgr()
{
    foreach(MfgMap mfgMap, db.values()) {
        foreach(RemoteList list, mfgMap.values()) {
            qDeleteAll(list);
        }
    }
}

void RemoteDBMgr::getDBAsync()
{
    if (db.isEmpty() && !processing) {
        QSettings settings(this);
        netAM.get(QNetworkRequest(QUrl(settings.value("baseUrl",
            "http://mercury.wipsl.com/irwi/").toString() + "db/db.xml")));
    } else {
        emit dbReady(&db);
    }

}

void RemoteDBMgr::dbDownloadFinished(QNetworkReply *reply)
{
    if (processing = (reply->error() == QNetworkReply::NoError)) {
        QByteArray ba = reply->readAll();
        QBuffer buf(&ba);
        buf.open(QIODevice::ReadOnly);
        buf.seek(0);
        QDomDocument doc;
        doc.setContent(&buf);
        parse(doc);

        if (!db.isEmpty()) {
            emit dbReady(&db);
        }
    } else {
        emit downloadFailed(reply->error());
    }

    processing = false;
}

void RemoteDBMgr::parse(QDomDocument &doc)
{
    QDomNodeList chars = doc.elementsByTagName("char");

    for (int i = 0; i < chars.size(); ++i) {
        QDomElement charEl = chars.at(i).toElement();
        if (!charEl.isNull()) {
            if (charEl.hasAttribute("id"))
                db.insert(charEl.attribute("id"), parseMfgMap(charEl));
        }
    }
}

MfgMap RemoteDBMgr::parseMfgMap(QDomElement &charEl)
{
    MfgMap mfgMap;
    QDomNodeList mfgs = charEl.elementsByTagName("mfg");

    for (int i = 0; i < mfgs.size(); ++i) {
        QDomElement mfgEl = mfgs.at(i).toElement();
        if (!mfgEl.isNull()) {
            if (mfgEl.hasAttribute("id")) {
                mfgMap.insert(mfgEl.attribute("id"), parseRemotes(mfgEl));
            }
        }
    }
    return mfgMap;
}

RemoteList RemoteDBMgr::parseRemotes(QDomElement &mfgEl)
{
    RemoteList remotes;
    QDomNodeList models = mfgEl.elementsByTagName("model");

    for (int i = 0; i < models.size(); ++i) {
        QDomElement modelEl = models.at(i).toElement();
        if (!modelEl.isNull()) {
            if (modelEl.hasAttribute("name")
                && modelEl.hasAttribute("rating")
                && modelEl.hasAttribute("vote_count")) {
                remotes << new Remote(modelEl.attribute("name"),
                                      mfgEl.attribute("id"),
                                      modelEl.attribute("rating").toInt(),
                                      modelEl.attribute("vote_count").toInt());
            }
        }
    }
    return remotes;
}


