/**
 * intervalometer
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to tomi.suviola@gmail.com so I can send you a copy immediately.
 *
 * @copyright  Copyright (c) 2010 Tomi Suviola
 * @license    http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

#include <QDebug>
#include <QSettings>
#include <QTimer>
#include <QDir>
#include <QApplication>
#include <QPluginLoader>
#include <QtGlobal>
#include "interval.h"

Interval::Interval() :
        m_running(0),
        m_cancel(false)
{
    QSettings settings;

    m_mode = settings.value("advanced", false).toBool();
//set ExamplePlugin as default
    QString pluginName = settings.value("plugin", QString("N900CamPlugin")).toString();

    if (!pluginName.isEmpty()) {
        //first see internal ones
        foreach (QObject *plugin, QPluginLoader::staticInstances()) {
            if (plugin && plugin->metaObject()->className() == pluginName) {
                intervalInterface = qobject_cast<CameraInterface *>(plugin);
                qDebug() << plugin->metaObject()->className() << " loaded";
            }
        }
        //then external ones
        if (!intervalInterface && !loadPlugin(pluginName)) {
        QString message = QString("Could not load the plugin: ") + pluginName;
//        QMessageBox::information(this, "Error", message);
        qDebug() << message;
    }
}
//    connect(this,SIGNAL(start()), this, SLOT(on_start()));
//    connect(this,SIGNAL(done()), this, SLOT(on_done()));
    resetSettings();
}

Interval::~Interval()
{
    for (int i = 0; i < m_timers.size(); ++i) {
        delete m_timers.at(i);

    }
}

bool Interval::isAdvancedMode()
{
    return m_mode;
}

void Interval::setAdvancedMode(bool advanced)
{
    m_mode = advanced;
    QSettings settings;
    settings.setValue("advanced", advanced);
}

bool Interval::isRunning()
{
    return m_running;
}

void Interval::resetSettings()
{
    m_SelfS = 0;
    m_IntS = 0;
    m_IntModS = 0;
    m_LongModS = 0;
    m_LongS = 0;
    m_Frames = 0;
    m_PreS = 0;
    m_MluS = 0;
}

void Interval::checkSettings()
{
    m_LongS = qMax((long)1,m_LongS);
    m_PreS = qMin(m_SelfS,m_PreS);
    m_MluS = qMin(m_SelfS,m_MluS);
    if(m_Frames > 1) {
        m_PreS = qMin(m_PreS,m_IntS);
        m_MluS = qMin(m_MluS,m_IntS);
    }
}

void Interval::setSelf(long val)
{
    qDebug() << "self: " << val;
    m_SelfS = val;
}

void Interval::setLong(long val)
{
    qDebug() << "long: " << val;
    m_LongS = val;
}

void Interval::setInt(long val)
{
    qDebug() << "int: " << val;
    m_IntS = val;
}

void Interval::setFrames(long val)
{
    qDebug() << "frames: " << val;
    m_Frames = val;
}

void Interval::addTimer(int msec, QObject *receiver, const char *member)
{

    QTimer::singleShot(1000*msec, receiver, member);
    emit nextEvent(msec);
/*
    QTimer *newTimer = new QTimer(this);
    newTimer->setSingleShot(true);
    newTimer->setInterval(msec);
    connect(newTimer, SIGNAL(timeout()), receiver, member);
    newTimer->start();
    m_timers.append(newTimer);
    */
}

void Interval::start()
{
    qDebug() << "start";

    checkSettings();
    m_FrameCounter = 0;
    m_BracketCounter = 0;
    m_running = true;
    m_cancel = false;
    intervalInterface->open();

    if (m_MluS) {
        addTimer(m_SelfS-m_MluS, this, SLOT(mlu()));
    } else if (m_PreS) {
        addTimer(m_SelfS-m_PreS, this, SLOT(focus()));
    } else {
        addTimer(m_SelfS, this, SLOT(exposure()));
    }
    emit started();
}

void Interval::stop()
{
    qDebug() << "stop";
    //kill all timers
    m_cancel = true;
    this->done();
}

void Interval::trigger()
{
    qDebug() << "trigger";
}

void Interval::exposure()
{
    if (m_cancel) { this->done(); return; }

    intervalInterface->focus(true);
    intervalInterface->shutter(true);
    long expTime = qMax((long)1, m_LongS+(m_FrameCounter*m_LongModS));
    addTimer(expTime, this, SLOT(exposure_off()));
}

void Interval::exposure_off()
{
    if (m_cancel) { this->done(); return; }

    intervalInterface->shutter(false);
    intervalInterface->focus(false);

    m_BracketCounter++;

    if (m_Bracket && m_BracketCounter < m_Brackets) {
        if (m_MluS) {
              QTimer::singleShot(m_MluS-1, this, SLOT(mlu()));
        } else {
              QTimer::singleShot(1, this, SLOT(exposure()));
        }
    return;
    }
    m_BracketCounter = 0;
    m_FrameCounter++;
    if (m_FrameCounter < m_Frames) {
        long timeout = m_IntS+(m_FrameCounter-1)*m_IntModS;
        if (m_MluS) {
              addTimer(timeout-m_MluS, this, SLOT(mlu()));
        } else  if (m_PreS) {
              addTimer(timeout-m_PreS, this, SLOT(focus()));
        } else {
              addTimer(timeout, this, SLOT(exposure()));
        }
    } else {
        addTimer(0, this, SLOT(done()));
    }
}

void Interval::mlu()
{
    if (m_cancel) { this->done(); return; }

    intervalInterface->focus(true);
    intervalInterface->shutter(true);
    addTimer(1, this, SLOT(mlu_off()));
}

void Interval::mlu_off()
{
    if (m_cancel) { this->done(); return; }

    intervalInterface->shutter(false);
    intervalInterface->focus(false);
    addTimer(m_MluS-1, this, SLOT(exposure()));
}

void Interval::focus()
{
    if (m_cancel) { this->done(); return; }

    /*focus sound*/
    intervalInterface->focus(true);
    addTimer(m_PreS, this, SLOT(exposure()));
}

void Interval::done()
{
    qDebug() << "done";
    //cancel all events
//      ClearTimers();
    //return to idle state
    intervalInterface->shutter(false);
    intervalInterface->focus(false);
    m_TimerStart = 0;
    m_running = false;
    intervalInterface->close();

    emit stopped();
}

bool Interval::loadPlugin(const QString &pluginName)
{
    QDir pluginsDir(qApp->applicationDirPath());
    pluginsDir.cd("plugins");
    foreach (QString fileName, pluginsDir.entryList(QDir::Files)) {
        qDebug("file: %s",fileName.toLocal8Bit().constData());
        if (fileName == pluginName) {
        QPluginLoader pluginLoader(pluginsDir.absoluteFilePath(fileName));
        QObject *plugin = pluginLoader.instance();
        if (plugin) {
            intervalInterface = qobject_cast<CameraInterface *>(plugin);
            if (intervalInterface)
                return true;
        }
        }
    }

    return false;
}
