#ifndef LUM_MODEL_NUMBER_H
#define LUM_MODEL_NUMBER_H

/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <assert.h>

#include <limits>

#include <Lum/Base/String.h>

#include <Lum/Model/Simple.h>

namespace Lum {
  namespace Model {

    template<class N>
    class LUMAPI Number : public Simple<N>
    {
    public:
      Number()
      : Simple<N>(),
        minimum(std::numeric_limits<N>::min()),
        maximum(std::numeric_limits<N>::max())
      {
        // no code
      }

      Number(const N& value)
      : Simple<N>(value),
        minimum(std::numeric_limits<N>::min()),
        maximum(std::numeric_limits<N>::max())
      {
        // no code
      }

      Number(const N& value, const N& min, const N& max)
      : Simple<N>(value),
        minimum(min),
        maximum(max)
      {
        // no code
      }

      void Inc()
      {
        assert(!this->IsNull());

        if (this->maximum-1>=this->value.value) {
          this->value.value++;

          this->Notify();
        }
      }

      void Dec()
      {
        assert(!this->IsNull());

        if (this->minimum+1<=this->value.value) {
          this->value.value--;

          this->Notify();
        }
      }

      void Add(N x)
      {
        assert(!this->IsNull());

        this->value.value+=x;

        this->Notify();
      }

      void Sub(N x)
      {
        assert(!this->IsNull());

        this->value.value-=x;

        this->Notify();
      }

      N GetMin() const
      {
        return minimum;
      }

      N GetMax() const
      {
        return maximum;
      }

      void Set(const N& value)
      {
        if (value>=this->minimum && value<=this->maximum) {
          Simple<N>::Set(value);
        }
        else {
          this->SetNull();
        }
      }

      void SetRange(N minimum, N maximum)
      {
        assert(maximum>=minimum);

        if (this->minimum!=minimum ||
            this->maximum!=maximum) {
          this->minimum=minimum;
          this->maximum=maximum;

          if (!this->IsNull() &&
              (this->value.value<this->minimum ||
               this->value.value>this->maximum)) {
            this->SetNull();
          }

          this->Notify();
        }
      }

      void SetToMin()
      {
        Set(minimum);
      }

      void SetToMax()
      {
        Set(maximum);
      }

      void SetString(const std::wstring& string)
      {
        N v;

        if (Base::WStringToNumber(string,v)) {
          this->Set(v);
        }
        else {
          this->SetNull();
        }
      }

    private:
      N minimum;
      N maximum;
    };

    /**
      This is a (temporary) pre gcc 4.0 hack, since attribute visibility does
      not work in this case together with typedefs.
     */

#define Number_INSTANCIATION(name, type)\
    class LUMAPI name : public Number<type>\
    {\
    public:\
      name() : Number<type>() {};\
      name(const type&value) : Number<type>(value) {};\
      name(const type&value, const type&min, const type&max) : Number<type>(value,min,max) {};\
    }

    //
    // int
    //

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Number<int>;
#endif

    Number_INSTANCIATION(Int,int);

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Base::Reference<Int>;
#endif

    typedef Base::Reference<Int> IntRef;

    //
    // unsigned int
    //

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Number<unsigned int>;
#endif

    Number_INSTANCIATION(UInt,unsigned int);

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Base::Reference<UInt>;
#endif

    typedef Base::Reference<UInt> UIntRef;

    //
    // long
    //

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Number<long>;
#endif

    Number_INSTANCIATION(Long,long);

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Base::Reference<Long>;
#endif

    typedef Base::Reference<Long> LongRef;

    //
    // unsigned long
    //

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Number<unsigned long>;
#endif

    Number_INSTANCIATION(ULong,unsigned long);

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Base::Reference<ULong>;
#endif

    typedef Base::Reference<ULong> ULongRef;

    //
    // size_t
    //

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Number<size_t>;
#endif

    Number_INSTANCIATION(SizeT,size_t);

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Base::Reference<SizeT>;
#endif

    typedef Base::Reference<SizeT> SizeTRef;

    //
    // float
    //

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Number<float>;
#endif

    Number_INSTANCIATION(Float,float);

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Base::Reference<Float>;
#endif

    typedef Base::Reference<Float> FloatRef;

    //
    // double
    //

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Number<double>;
#endif

    Number_INSTANCIATION(Double,double);

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Base::Reference<Double>;
#endif

    typedef Base::Reference<Double> DoubleRef;

#undef Number_INSTANCIATION
  }
}

#endif
