#ifndef LUM_MODEL_CONVERTER_H
#define LUM_MODEL_CONVERTER_H

/*
  This source is part of the Illumination library
  Copyright (C) 2010  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <string>

#include <Lum/Base/Model.h>
#include <Lum/Base/Reference.h>
#include <Lum/Base/String.h>

#include <Lum/Model/Number.h>

namespace Lum {
  namespace Model {

    /**
     * Abstract base class for converting model values from and to string values.
     *
     * This is useful for using Lum::String* and similar input controls together
     * with non-text models like ones derived from Lum::Model::Number.
     *
     */
    class LUMAPI StringConverter : public Lum::Base::Referencable
    {
    public:
      virtual ~StringConverter();

      virtual bool ConvertModelToString(Lum::Base::Model* model, std::wstring& value) = 0;
      virtual bool ConvertStringToModel(const std::wstring& value, Lum::Base::Model* model) = 0;
    };

    typedef Base::Reference<StringConverter> StringConverterRef;

    class LUMAPI StringStringConverter : public StringConverter
    {
    public:
      bool ConvertModelToString(Lum::Base::Model* model, std::wstring& value);
      bool ConvertStringToModel(const std::wstring& value, Lum::Base::Model* model);
    };

    typedef Base::Reference<StringStringConverter> StringStringConverterRef;

    template <class N>
    class LUMAPI NumberStringConverter : public StringConverter
    {
      bool ConvertModelToString(Lum::Base::Model* model, std::wstring& value)
      {
        Lum::Model::Number<N>* sourceModel=dynamic_cast<Lum::Model::Number<N>*>(model);

        if (sourceModel==NULL) {
          return false;
        }

        if (sourceModel->IsNull()) {
          value=L"";
          return true;
        }

        value=Base::NumberToWString(sourceModel->Get());

        return true;
      }

      bool ConvertStringToModel(const std::wstring& value, Lum::Base::Model* model)
      {
        Lum::Model::Number<N>* targetModel=dynamic_cast<Lum::Model::Number<N>*>(model);

        if (targetModel==NULL) {
          return false;
        }

        N v;

        if (Base::WStringToNumber(value,v)) {
          targetModel->Set(v);
        }
        else {
          targetModel->SetNull();
        }

        return true;
      }
    };

    /**
      This is a (temporary) pre gcc 4.0 hack, since attribute visibility does
      not work in this case together with typedefs.
     */

#define Number_INSTANCIATION(name, type)\
    class LUMAPI name : public NumberStringConverter<type>\
    {\
    public:\
      name() : NumberStringConverter<type>() {};\
    }

    Number_INSTANCIATION(IntStringConverter,int);
    typedef Base::Reference<IntStringConverter> IntStringConverterRef;

    Number_INSTANCIATION(UIntStringConverter,unsigned int);
    typedef Base::Reference<UIntStringConverter> UIntStringConverterRef;

    Number_INSTANCIATION(LongStringConverter,long);
    typedef Base::Reference<LongStringConverter> LongStringConverterRef;

    Number_INSTANCIATION(ULongStringConverter,unsigned long);
    typedef Base::Reference<ULongStringConverter> ULongStringConverterRef;

    Number_INSTANCIATION(SizeTStringConverter,size_t);
    typedef Base::Reference<SizeTStringConverter> SizeTStringConverterRef;

#undef Number_INSTANCIATION
  }
}

#endif
