/*
  This source is part of the Illumination library
  Copyright (C) 2009  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/OS/Hildon/Behaviour.h>

#include <hildon/hildon-version.h>

#include <Lum/Array.h>
#include <Lum/BooleanButton.h>
#include <Lum/Button.h>
#include <Lum/Dialog.h>
#include <Lum/Menu.h>
#include <Lum/Popup.h>
#include <Lum/Tab.h>
#include <Lum/ValueButton.h>

#include <Lum/OS/X11/Display.h>
#include <Lum/OS/X11/Window.h>
#include <Lum/OS/X11/GtkTheme.h>

namespace Lum {
  namespace OS {
    namespace Hildon {

#if HILDON_CHECK_VERSION(2,1,0)
      class CustomMenuHandlerPopup : public OS::Behaviour::CustomMenuHandler
      {
      private:
        static const size_t columns = 2;
        static const size_t rows    = 5;

        class MenuPopup : public Popup
        {
        private:
          std::set<Model::Action*> actions;

        private:
          void AddToPopupArray(Array* array,
                               Def::Menu* def,
                               const std::set<Lum::Base::Model*>& excludes)
          {
            for (std::list<Def::MenuItem*>::const_iterator iter=def->GetItems().begin();
                 iter!=def->GetItems().end();
                 ++iter) {
              assert((*iter)!=NULL);

              if (dynamic_cast<Def::Menu*>(*iter)!=NULL) {
                Def::Menu *sub=dynamic_cast<Def::Menu*>(*iter);

                AddToPopupArray(array,sub,excludes);
              }
              else if (dynamic_cast<Def::MenuAction*>(*iter)!=NULL) {
                Def::MenuAction *action=dynamic_cast<Def::MenuAction*>(*iter);

                if (excludes.find(action->GetAction().GetAction())==excludes.end()) {
                  actions.insert(action->GetAction().GetAction());
                  ObservePrior(action->GetAction().GetAction());

                  Button *button=Button::Create(action->GetAction().GetDesc().GetLabel(),
                                                action->GetAction().GetAction(),
                                                true,false);

                  array->Add(button);
                }
              }
              else if (dynamic_cast<Def::MenuBoolean*>(*iter)!=NULL) {
                Def::MenuBoolean *boolean=dynamic_cast<Def::MenuBoolean*>(*iter);

                if (excludes.find(boolean->GetBoolean().GetBoolean())==excludes.end()) {
                  // TODO
                }
              }
              else if (dynamic_cast<Def::MenuOneOfN*>(*iter)!=NULL) {
                Def::MenuOneOfN *oneOfN=dynamic_cast<Def::MenuOneOfN*>(*iter);

                if (excludes.find(oneOfN->GetOneOfN().GetValue())==excludes.end()) {
                  // TODO
                }
              }
            }
          }

        public:
          MenuPopup(Def::Menu* def,
                    const std::set<Lum::Base::Model*>& excludes)
          {
            Array        *array = new Array();
            X11::Display *display=dynamic_cast<X11::Display*>(OS::display);

            array->SetFlex(true,true);
            array->SetSpace(true,true);
            array->SetHorizontalCount(2);

            AddToPopupArray(array,def,excludes);

            SetMain(array,false,true);
            SetBackground(dynamic_cast<X11::GtkTheme*>(OS::display->GetTheme())->appMenuBackground);
            SetType(OS::Window::typeCustom);
            dynamic_cast<X11::Window*>(GetWindow())->SetCustomWindowType(XInternAtom(display->display,
                                                                         "_HILDON_WM_WINDOW_TYPE_APP_MENU",
                                                                         false));

            Observe(GetClosedAction());
          }

          void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
          {
            if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
              Hide();
              return;
            }
            else if (dynamic_cast<Model::Action*>(model)!=NULL) {
              Model::ActionRef action=dynamic_cast<Lum::Model::Action*>(model);

              if (actions.find(action)!=actions.end()) {
                if (action->IsEnabled() && action->IsFinished() && IsOpen()) {
                  Hide();
                  return;
                }
              }
            }

            Popup::Resync(model,msg);
          }
        };

      private:
        Menu      *menu;
        MenuPopup *popup;

      private:
        void CountModels(Def::Menu* def,
                         size_t& actions,
                         size_t& bools,
                         size_t& oneOfNs,
                         const std::set<Lum::Base::Model*>& excludes)
        {
          for (std::list<Def::MenuItem*>::const_iterator iter=def->GetItems().begin();
               iter!=def->GetItems().end();
               ++iter) {
            assert((*iter)!=NULL);

            if (dynamic_cast<Def::Menu*>(*iter)!=NULL) {
              Def::Menu *sub=dynamic_cast<Def::Menu*>(*iter);

              CountModels(sub,actions,bools,oneOfNs,excludes);
            }
            else if (dynamic_cast<Def::MenuAction*>(*iter)!=NULL) {
              Def::MenuAction *action=dynamic_cast<Def::MenuAction*>(*iter);

              if (excludes.find(action->GetAction().GetAction())==excludes.end()) {
                actions++;
              }
            }
            else if (dynamic_cast<Def::MenuBoolean*>(*iter)!=NULL) {
              Def::MenuBoolean *boolean=dynamic_cast<Def::MenuBoolean*>(*iter);

              if (excludes.find(boolean->GetBoolean().GetBoolean())==excludes.end()) {
                bools++;
              }
            }
            else if (dynamic_cast<Def::MenuOneOfN*>(*iter)!=NULL) {
              Def::MenuOneOfN *oneOfN=dynamic_cast<Def::MenuOneOfN*>(*iter);

              if (excludes.find(oneOfN->GetOneOfN().GetValue())==excludes.end()) {
                oneOfNs++;
              }
            }
          }
        }

        bool CanBeDisplayedAsButtonMenuPopup(Def::Menu* def,
                                             const std::set<Lum::Base::Model*>& excludes)
        {
          size_t actions=0;
          size_t bools=0;
          size_t oneOfNs=0;

          size_t buttonrows=0;

          CountModels(def,actions,bools,oneOfNs,excludes);

          if ((actions+bools) % columns!=0) {
            buttonrows=(actions+bools) / rows + 1;
          }
          else {
            buttonrows=(actions+bools) / columns;
          }

          buttonrows+=oneOfNs;

          return buttonrows<rows;
        }

      public:
        CustomMenuHandlerPopup(OS::Window* window)
        : OS::Behaviour::CustomMenuHandler(window),
          menu(NULL),
          popup(NULL)
        {
        }

        ~CustomMenuHandlerPopup()
        {
          delete menu;
          delete popup;
        }

        bool SetMenu(Def::Menu* def,
                const std::set<Lum::Base::Model*>& excludes)
        {
          delete menu;
          delete popup;

          menu=NULL;
          popup=NULL;

          if (CanBeDisplayedAsButtonMenuPopup(def,excludes)) {
            popup=new MenuPopup(def,excludes);
            popup->SetParent(window);

          }
          else {
            menu=ConvertToMenu(def,excludes);
            menu->SetParent(window);
          }

          return true;
        }

        bool Open()
        {
          if (menu!=NULL) {
            menu->SetPos(window->GetX()+50,window->GetY()+50);
            return menu->OpenInStickyMode();
          }
          else if (popup!=NULL) {
            if (popup->IsOpen()) {
              return popup->Show();
            }
            else if (popup->Open()) {
              return true;
            }
            else {
              return false;
            }
          }
          else {
            return false;
          }
        }
      };
  #else
      class CustomMenuHandlerPopup : public OS::Behaviour::CustomMenuHandler
      {
      private:
        Menu *menu;

      public:
        CustomMenuHandlerPopup(OS::Window* window)
        : OS::Behaviour::CustomMenuHandler(window),
          menu(NULL)
        {
        }

        ~CustomMenuHandlerPopup()
        {
          delete menu;
        }

        bool SetMenu(Def::Menu* def,
                const std::set<Lum::Base::Model*>& excludes)
        {
          delete menu;

          menu=ConvertToMenu(def,excludes);
          menu->SetParent(window);

          return true;
        }

        bool Open()
        {
          if (menu!=NULL && !menu->GetWindow()->IsShown()) {
            menu->SetPos(window->GetX()+12,window->GetY());
            return menu->OpenInStickyMode();
          }
          else {
            return false;
          }
        }
      };

#endif

      OS::Behaviour::CustomMenuHandler* Behaviour::GetCustomMenuHandler(OS::Window* window) const
      {
        return new CustomMenuHandlerPopup(window);
      }

#if HILDON_CHECK_VERSION(2,1,0)
      class ViewDialog : public Dialog
      {
      private:
        Lum::Object *view;

      public:
        ViewDialog(Lum::Object* view)
         :view(view)
        {
          SetType(OS::Window::typeMain);
          SetExitAction(GetClosedAction());
        }

        void PreInit()
        {
          SetMain(view);

          Dialog::PreInit();
        }
      };

      class MultiViewArray : public Array
      {
      private:
        Def::MultiView                    multiView;
        Model::SizeTRef                   value;
        std::vector<Model::ActionRef>     actions;
        std::vector<Def::MultiView::View> views;
        std::vector<Dialog*>              dialogs;

      public:
        MultiViewArray(const Def::MultiView& multiView)
         : multiView(multiView),
           value(multiView.GetValue())
        {
          if (!value.Valid()) {
            value=new Model::SizeT();
          }

          Observe(value);

          for (std::list<Def::MultiView::View>::const_iterator view=multiView.GetViews().begin();
               view!=multiView.GetViews().end();
               ++view) {
            Button           *button;
            Model::ActionRef action=new Model::Action();

            actions.push_back(action);
            dialogs.push_back(NULL);
            views.push_back(*view);
            Observe(action);

            button=Button::Create(view->GetDesc().GetLabel(),action,true,true);
            button->SetMaxHeight(Lum::Base::Size::pixel,105); //Shouldn't be hardcoded

            Add(button);
          }
        }

        ~MultiViewArray()
        {
          for (std::vector<Dialog*>::iterator dialog=dialogs.begin();
               dialog!=dialogs.end();
               ++dialog) {
            delete *dialog;
          }
        }

        void CalcSize()
        {
          assert(GetWindow()!=NULL);
          Observe(GetWindow()->GetOpenedAction());

          Array::CalcSize();
        }

        void ShowDialog()
        {
          if (dialogs[value->Get()]==NULL) {
            Def::Menu *menu=NULL;

            if (multiView.GetPreMenu()!=NULL) {
              menu=new Def::Menu(*multiView.GetPreMenu());

              if (views[value->Get()].GetMenu()!=NULL) {
                menu->Group(views[value->Get()].GetDesc())
                ->Append(views[value->Get()].GetMenu())
                ->End();
              }

              if (multiView.GetPostMenu()!=NULL) {
                menu->Append(multiView.GetPostMenu());
              }
            }

            dialogs[value->Get()]=new ViewDialog(views[value->Get()].GetObject());
            dialogs[value->Get()]->SetParent(GetWindow());
            dialogs[value->Get()]->SetMenu(menu);
          }

          Dialog* dialog=dialogs[value->Get()];

          if (dialog->Open()) {
            dialog->EventLoop();
            dialog->Close();

            value->SetNull();
          }
        }

        void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
        {
          if (GetWindow()!=NULL &&
              model==GetWindow()->GetOpenedAction() &&
              GetWindow()->GetOpenedAction()->IsFinished() && !value->IsNull()) {

            Array::Resync(model,msg);

            ShowDialog();

            return;
          }

          for (size_t i=0; i<actions.size(); i++) {
            if (model==actions[i]) {
              if (actions[i]->IsEnabled() && actions[i]->IsFinished()) {
                value->Set(i);
              }
            }
          }

          if (model==value && !value->IsNull() && IsVisible()) {
            ShowDialog();
          }

          Array::Resync(model,msg);
        }
      };
#endif

      bool Behaviour::FocusObjectOnWindowFocusIn() const
      {
#if HILDON_CHECK_VERSION(2,1,0)
        return true;
#else
        return false;
#endif
      }

      bool Behaviour::ScrollKnobActive() const
      {
#if HILDON_CHECK_VERSION(2,1,0)
        return false;
#else
        return true;
#endif
      }

      void Behaviour::ApplyMultiViewDlg(Dialog* dialog,
                                        const Def::MultiView& multiView) const
      {
#if HILDON_CHECK_VERSION(2,1,0)
        Array     *array=new MultiViewArray(multiView);
        Def::Menu *menu=NULL;

        if (multiView.GetPreMenu()!=NULL) {
          menu=new Def::Menu(*multiView.GetPreMenu());

          // TODO: Build View menu

          if (multiView.GetPostMenu()!=NULL) {
            menu->Append(multiView.GetPostMenu());
          }
        }

        array->SetFlex(true,true);
        array->SetSpace(true,true);
        array->SetHorizontalCount(std::min(multiView.GetViews().size(),(size_t)3));

        dialog->SetMain(array);
        dialog->SetMenu(menu);
#else
        Tab                  *tab=Tab::Create(true,true);
        Lum::Model::SizeTRef value=multiView.GetValue();
        Def::Menu            *menu=NULL;

        if (multiView.GetPreMenu()!=NULL) {
          menu=new Def::Menu(*multiView.GetPreMenu());

          if (multiView.GetPostMenu()!=NULL) {
            menu->Append(multiView.GetPostMenu());
          }
        }

        if (value.Valid()) {
          tab->SetModel(value);
        }

        for (std::list<Def::MultiView::View>::const_iterator view=multiView.GetViews().begin();
             view!=multiView.GetViews().end();
             ++view) {
          tab->Add(view->GetDesc().GetLabel(),view->GetObject());
        }

        dialog->SetMain(tab);
        dialog->SetMenu(menu);
#endif
      }

      Lum::Object* Behaviour::GetOneOfManyControl(Def::OneOfMany& def) const
      {
#if HILDON_CHECK_VERSION(2,1,0)
        TableIndexValueButton *control;

        control=new TableIndexValueButton(def.GetChoices());
        control->SetModel(def.GetValue());
        control->SetLabel(def.GetDesc().GetLabel());

        return control;
#else
        return DefaultBehaviour::GetOneOfManyControl(def);
#endif
      }

      Lum::Object* Behaviour::GetBooleanControl(Def::Boolean& def) const
      {
#if HILDON_CHECK_VERSION(2,1,0)
        BooleanButton *control;

        control=new BooleanButton();
        control->SetModel(def.GetBoolean());
        control->SetLabel(def.GetDesc().GetLabel());

        return control;
#else
        return DefaultBehaviour::GetBooleanControl(def);
#endif
      }
    }
  }
}
